package org.sqlite.jdbc;

import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Savepoint;
import org.sqlite.Database;
import java.util.Map;
import org.sqlite.auth.Authorizer;
import org.sqlite.event.BusyHandler;
import org.sqlite.event.CommitHook;
import org.sqlite.event.ProgressHandler;
import org.sqlite.event.RollbackHook;
import org.sqlite.event.UpdateHook;
import org.sqlite.text.Collator;
import org.sqlite.udf.Function;

/**
 *
 * @author calico
 */
public class JdbcConnection implements Connection {

    private Database db;
    private TransactionType type;
    private final String url;
    
    public JdbcConnection(Database db, String url) throws SQLException {
        this.db = db;
        db.pragma(
                new String[] {
                    "encoding = \"UTF-8\"",
                    "count_changes = off",
                    "case_sensitive_like = on",
                    "short_column_names = on",
                    "empty_result_callbacks = off",
//                    "short_column_names = off",
//                    "full_column_names = on",
//                    "show_datatypes = on", for SQLite ver 2.x
                }
            );
        this.url = url;
    }
    
    // START implements
    public JdbcStatement createStatement() throws SQLException {
        validateConnectionOpen();
        
        return new JdbcStatement(db, this);
    }

    public JdbcPreparedStatement prepareStatement(String sql) throws SQLException {
        validateConnectionOpen();
        
        return new JdbcPreparedStatement(db, this, sql);
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public CallableStatement prepareCall(String sql) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public String nativeSQL(String sql) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void setAutoCommit(boolean autoCommit) throws SQLException {
        validateConnectionOpen();
        
        if (!db.getAutoCommit()) {
            rollback();
        }
        if (!autoCommit && db.getAutoCommit()) {
            db.beginTransaction(type);
        }
    }

    public boolean getAutoCommit() throws SQLException {
        validateConnectionOpen();
        
        return db.getAutoCommit();
    }

    public void commit() throws SQLException {
        validateConnectionOpen();
        
        if (db.getAutoCommit()) {
            throw new SQLException("Connection is auto commit mode.", "90J10");
        }
        db.commitTransaction();
    }

    public void rollback() throws SQLException {
        validateConnectionOpen();
        
        if (db.getAutoCommit()) {
            throw new SQLException("Connection is auto commit mode.", "90J10");
        }
        db.rollbackTransaction();
    }

    public void close() throws SQLException {
        if (!isClosed()) {
            if (!db.getAutoCommit()) {
                rollback();
            }
            db.close();
            db = null;
        }
    }

    public boolean isClosed() throws SQLException {
        return (db == null || db.isClosed());
    }

    public JdbcDatabaseMetaData getMetaData() throws SQLException {
        return new JdbcDatabaseMetaData(db, this, url);
    }

    public void setReadOnly(boolean readOnly) throws SQLException {
        // TODO バージョン 3.5.0 以降は sqlite3_open_v2() 関数で読取専用でオープンすることができる
        validateConnectionOpen();
        
        if (!db.getAutoCommit()) {
            throw new SQLException("Connection is transaction mode.", "90J11");
        }
        if (readOnly) {
            throw new UnsupportedOperationException("Not supported yet.");
        }
    }

    public boolean isReadOnly() throws SQLException {
        validateConnectionOpen();
        
        return db.isReadOnly();
    }

    /**
     * Catalog is not supported yet.
     * @param catalog ignored
     * @throws java.sql.SQLException
     */
    public void setCatalog(String catalog) throws SQLException {
        validateConnectionOpen();
        
        // nothing
    }

    /**
     * Catalog is not supported yet.
     * It always returns null.
     * @return null
     * @throws java.sql.SQLException
     */
    public String getCatalog() throws SQLException {
        validateConnectionOpen();
        
        return null;
    }

    public void setTransactionIsolation(int level) throws SQLException {
        validateConnectionOpen();
        
        if (level != TRANSACTION_SERIALIZABLE) {
            throw new SQLException("Not supported isolation level.", "90J20");
        }
        // nothing
    }

    /**
     * It always returns TRANSACTION_SERIALIZABLE.
     * @return java.sql.Connection.TRANSACTION_SERIALIZABLE
     * @throws java.sql.SQLException
     */
    public int getTransactionIsolation() throws SQLException {
        validateConnectionOpen();
        
        return TRANSACTION_SERIALIZABLE;
    }

    /**
     * SQL Warning is not supported yet.
     * It always returns null.
     * @return null
     * @throws java.sql.SQLException
     */
    public SQLWarning getWarnings() throws SQLException {
        validateConnectionOpen();
        
        return null;
    }

    /**
     * SQL Warning is not supported yet.
     * @throws java.sql.SQLException
     */
    public void clearWarnings() throws SQLException {
        validateConnectionOpen();
        
        // nothing
    }

    public JdbcStatement createStatement(int resultSetType, int resultSetConcurrency) throws SQLException {
        JdbcResultSet.validateResultSetType(resultSetType);
        JdbcResultSet.validateResultSetConcurrency(resultSetConcurrency);
        validateConnectionOpen();
        
        if (resultSetType != ResultSet.TYPE_FORWARD_ONLY) {
            throw new SQLException("Not supported result set type.", "90J24");
        }
        if (resultSetConcurrency != ResultSet.CONCUR_READ_ONLY) {
            throw new SQLException("Not supported result set concurrency.", "90J25");
        }
        
        return new JdbcStatement(db, this);
    }

    public JdbcPreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency) throws SQLException {
        JdbcResultSet.validateResultSetType(resultSetType);
        JdbcResultSet.validateResultSetConcurrency(resultSetConcurrency);
        validateConnectionOpen();
        
        if (resultSetType != ResultSet.TYPE_FORWARD_ONLY) {
            throw new SQLException("Not supported result set type.", "90J24");
        }
        if (resultSetConcurrency != ResultSet.CONCUR_READ_ONLY) {
            throw new SQLException("Not supported result set concurrency.", "90J25");
        }
        
        return new JdbcPreparedStatement(db, this, sql);        
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public Map<String, Class<?>> getTypeMap() throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public void setTypeMap(Map<String, Class<?>> map) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Supported result set holdability is ResultSet.CLOSE_CURSORS_AT_COMMIT only.
     * @param holdability
     * @throws java.sql.SQLException
     */
    public void setHoldability(int holdability) throws SQLException {
        validateConnectionOpen();

        if (holdability != ResultSet.CLOSE_CURSORS_AT_COMMIT) {
            throw new SQLException("Not supported result set holdability.", "90J21");
        }
        // nothing
    }

    /**
     * It always returns CLOSE_CURSORS_AT_COMMIT.
     * @return java.sql.ResultSet.CLOSE_CURSORS_AT_COMMIT
     * @throws java.sql.SQLException
     */
    public int getHoldability() throws SQLException {
        validateConnectionOpen();

        return ResultSet.CLOSE_CURSORS_AT_COMMIT;
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public Savepoint setSavepoint() throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public Savepoint setSavepoint(String name) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public void rollback(Savepoint savepoint) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public void releaseSavepoint(Savepoint savepoint) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public JdbcStatement createStatement(int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
        JdbcResultSet.validateResultSetType(resultSetType);
        JdbcResultSet.validateResultSetConcurrency(resultSetConcurrency);
        JdbcResultSet.validateResultSetHoldability(resultSetHoldability);
        validateConnectionOpen();

        if (resultSetType != ResultSet.TYPE_FORWARD_ONLY) {
            throw new SQLException("Not supported result set type.", "90J24");
        }
        if (resultSetConcurrency != ResultSet.CONCUR_READ_ONLY) {
            throw new SQLException("Not supported result set concurrency.", "90J25");
        }
        if (resultSetHoldability != ResultSet.CLOSE_CURSORS_AT_COMMIT) {
            throw new SQLException("Not supported result set holdability.", "90J21");
        }
        
        return new JdbcStatement(db, this);
    }

    public JdbcPreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
        JdbcResultSet.validateResultSetType(resultSetType);
        JdbcResultSet.validateResultSetConcurrency(resultSetConcurrency);
        JdbcResultSet.validateResultSetHoldability(resultSetHoldability);
        validateConnectionOpen();
        
        if (resultSetType != ResultSet.TYPE_FORWARD_ONLY) {
            throw new SQLException("Not supported result set type.", "90J24");
        }
        if (resultSetConcurrency != ResultSet.CONCUR_READ_ONLY) {
            throw new SQLException("Not supported result set concurrency.", "90J25");
        }
        if (resultSetHoldability != ResultSet.CLOSE_CURSORS_AT_COMMIT) {
            throw new SQLException("Not supported result set holdability.", "90J21");
        }
        
        return new JdbcPreparedStatement(db, this, sql);        
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * 
     * @param sql
     * @param autoGeneratedKeys ignored
     * @return
     * @throws java.sql.SQLException
     */
    public JdbcPreparedStatement prepareStatement(String sql, int autoGeneratedKeys) throws SQLException {
        JdbcStatement.validateAutoGeneretedKeys(autoGeneratedKeys);
        validateConnectionOpen();

        return new JdbcPreparedStatement(db, this, sql);
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public JdbcPreparedStatement prepareStatement(String sql, int[] columnIndexes) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public JdbcPreparedStatement prepareStatement(String sql, String[] columnNames) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }
    // END implements
    
    protected void validateConnectionOpen() throws SQLException {
        if (isClosed()) {
            throw new SQLException("Connection is already closed.", "90007");
        }    
    }

    /**
     * 
     * @param type  'DEFERRED', 'IMMEDIATE', 'EXCLUSIVE'
     * @see <a href="http://www.sqlite.org/lang_transaction.html">BEGIN TRANSACTION</a>
     */
    public void setTransactionType(TransactionType type) throws SQLException {
        validateConnectionOpen();

        this.type = type;
    }

    /**
     * 
     * @return transaction type
     * @see <a href="http://www.sqlite.org/lang_transaction.html">BEGIN TRANSACTION</a>
     */
    public TransactionType getTransactionType() throws SQLException {
        validateConnectionOpen();

        return type;
    }

    /**
     * invoke sqlite3_create_function() function.
     * @param func User-Defined function
     * @throws java.sql.SQLException
     * @see org.sqlite.Database#createFunction(Function)
     */
    public void createFunction(Function func) throws SQLException {
        validateConnectionOpen();

        db.createFunction(func);
    }
    
    /**
     * invoke sqlite3_create_function() function.
     * @param func User-Defined function
     * @throws java.sql.SQLException
     * @see org.sqlite.Database#dropFunction(Function)
     */
    public void dropFunction(Function func) throws SQLException {
        validateConnectionOpen();

        db.dropFunction(func);
    }

    /**
     * invoke sqlite3_create_collation() function.
     * @param col User-Defined Collating Sequences
     * @throws java.sql.SQLException
     * @see org.sqlite.Database#createCollation(Collator)
     */
    public void createCollation(Collator col) throws SQLException {
        validateConnectionOpen();

        db.createCollation(col);
    }
    
    /**
     * invoke sqlite3_create_function() function.
     * @param col User-Defined Collating Sequences
     * @throws java.sql.SQLException
     * @see org.sqlite.Database#dropCollation(Collator)
     */
    public void dropCollation(Collator col) throws SQLException {
        validateConnectionOpen();

        db.dropCollation(col);
    }

    /**
     * invoke sqlite3_set_authorizer() function.
     * @param auth
     * @throws java.sql.SQLException
     * @see org.sqlite.Database#setAuthorizer(Authorizer)
     */
    public void setAuthorizer(Authorizer auth) throws SQLException {
        validateConnectionOpen();

        db.setAuthorizer(auth);
    }
    
    /**
     * invoke sqlite3_set_authorizer() function.
     * @throws java.sql.SQLException
     * @see org.sqlite.Database#clearAuthorizer()
     */
    public void clearAuthorizer() throws SQLException {
        validateConnectionOpen();

        db.clearAuthorizer();
    }
    
    /**
     * invoke sqlite3_busy_timeout() function.
     * @param ms    milliseconds
     * @throws java.sql.SQLException
     * @see org.sqlite.Database#setBusyTimeout(int)
     */
    public void setBusyTimeout(int ms) throws SQLException {
        validateConnectionOpen();

        db.setBusyTimeout(ms);
    }

    /**
     * Retrieves the timeout(ms) value.
     * @return  timeout(ms) value.
     * @throws java.sql.SQLException
     * @see org.sqlite.Database#getBusyTimeout()
     */
    public int getBusyTimeout() throws SQLException {
        validateConnectionOpen();

        return db.getBusyTimeout();
    }

    /**
     * invoke sqlite3_busy_handler() function.
     * @param busy busy handler
     * @throws java.sql.SQLException
     * @see org.sqlite.Database#setBusyHandler(BusyHandler)
     */
    public void setBusyHandler(BusyHandler busy) throws SQLException {
        validateConnectionOpen();

        db.setBusyHandler(busy);
    }
    
    /**
     * invoke sqlite3_busy_handler() function.
     * @throws java.sql.SQLException
     * @see org.sqlite.Database#clearBusyHandler()
     */
    public void clearBusyHandler() throws SQLException {
        validateConnectionOpen();

        db.clearBusyHandler();
    }

    /**
     * invoke sqlite3_progress_handler() function.
     * @param prog progress handler
     * @throws java.sql.SQLException
     * @see org.sqlite.Database#setProgressHandler(ProgressHandler)
     */
    public void setProgressHandler(ProgressHandler prog) throws SQLException {
        validateConnectionOpen();

        db.setProgressHandler(prog);
    }
    
    /**
     * invoke sqlite3_progress_handler() function.
     * @throws java.sql.SQLException
     * @see org.sqlite.Database#clearProgressHandler()
     */
    public void clearProgressHandler() throws SQLException {
        validateConnectionOpen();

        db.clearProgressHandler();
    }
    
    /**
     * invoke sqlite3_commit_hook() function.
     * @param hook commit hoot
     * @throws java.sql.SQLException
     * @see org.sqlite.Database#setCommitHook(CommitHook)
     */
    public void setCommitHook(CommitHook hook) throws SQLException {
        validateConnectionOpen();

        db.setCommitHook(hook);
    }
    
    /**
     * invoke sqlite3_commit_hook() function.
     * @throws java.sql.SQLException
     * @see org.sqlite.Database#clearCommitHook()
     */
    public void clearCommitHook() throws SQLException {
        validateConnectionOpen();

        db.clearCommitHook();
    }
    
    /**
     * invoke sqlite3_rollback_hook() function.
     * @param hook rollback hoot
     * @throws java.sql.SQLException
     * @see org.sqlite.Database#setRollbackHook(RollbackHook)
     */
    public void setRollbackHook(RollbackHook hook) throws SQLException {
        validateConnectionOpen();

        db.setRollbackHook(hook);
    }
    
    /**
     * invoke sqlite3_rollback_hook() function.
     * @throws java.sql.SQLException
     * @see org.sqlite.Database#clearRollbackHook()
     */
    public void clearRollbackHook() throws SQLException {
        validateConnectionOpen();

        db.clearRollbackHook();
    }
    
    /**
     * invoke sqlite3_update_hook() function.
     * @param hook update hoot
     * @throws java.sql.SQLException
     * @see org.sqlite.Database#setUpdateHook(UpdateHook)
     */
    public void setUpdateHook(UpdateHook hook) throws SQLException {
        validateConnectionOpen();

        db.setUpdateHook(hook);
    }
    
    /**
     * invoke sqlite3_update_hook() function.
     * @throws java.sql.SQLException
     * @see org.sqlite.Database#clearUpdateHook()
     */
    public void clearUpdateHook() throws SQLException {
        validateConnectionOpen();

        db.clearUpdateHook();
    }
}
