/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.sqlite.jdbc;

import java.sql.BatchUpdateException;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;
import org.sqlite.Database;

/**
 *
 * @author calico
 */
public class JdbcStatement implements Statement {

    protected Database db;
    private final Connection owner;
    protected ResultSet rs;
    protected int cntUpdate = -1;
    protected List<String> batch;
    
    public JdbcStatement(Database db, Connection owner) {
        this.db = db;
        this.owner = owner;
    }
    
    // START implements
    public ResultSet executeQuery(String sql) throws SQLException {
        validateStatementOpen();

        final org.sqlite.Statement stmt = db.prepare(sql);
        validateStaticSQL(stmt);
        if (!stmt.producedResultSet()) {
            stmt.close();
            throw new SQLException("No ResultSet was produced.");
        }
        rs = new JdbcResultSet(this, stmt);
        cntUpdate = -1;
        return rs;
    }

    public int executeUpdate(String sql) throws SQLException {
        validateStatementOpen();

        final org.sqlite.Statement stmt = db.prepare(sql);
        validateStaticSQL(stmt);
        try {
            if (stmt.producedResultSet()) {
                throw new SQLException("ResultSet was produced.");
            }
            stmt.execute();
            cntUpdate = db.changes();
//            cntUpdate = db.totalChanges();
            
        } finally {
            stmt.close();
        }
        rs = null;
        return cntUpdate;
    }

    public void close() throws SQLException {
        if (db != null) {
            db = null;
            if (rs != null) {
                rs.close();
                rs = null;
            }
            cntUpdate = -1;
            batch = null;
        }
    }

    public int getMaxFieldSize() throws SQLException {
        validateStatementOpen();

        return 0;
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public void setMaxFieldSize(int max) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * It always returns 0.
     * @return 0
     * @throws java.sql.SQLException
     */
    public int getMaxRows() throws SQLException {
        validateStatementOpen();

        return 0;
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public void setMaxRows(int max) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public void setEscapeProcessing(boolean enable) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public int getQueryTimeout() throws SQLException {
        validateStatementOpen();
        
        return (db.getBusyTimeout() / 1000);
    }

    public void setQueryTimeout(int seconds) throws SQLException {
        validateStatementOpen();
        
        db.setBusyTimeout(seconds * 1000);
    }

    public void cancel() throws SQLException {
        validateStatementOpen();

        db.interrupt();
    }

    /**
     * It always returns null.
     * @return null
     * @throws java.sql.SQLException
     */
    public SQLWarning getWarnings() throws SQLException {
        validateStatementOpen();

        return null;
    }

    public void clearWarnings() throws SQLException {
        validateStatementOpen();

        // nothing
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public void setCursorName(String name) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public boolean execute(String sql) throws SQLException {
        validateStatementOpen();

        return (executeQuery(sql) != null);
    }

    public ResultSet getResultSet() throws SQLException {
        validateStatementOpen();

        return rs;
    }

    public int getUpdateCount() throws SQLException {
        validateStatementOpen();

        return cntUpdate;
    }

    /**
     * CallableStatement is not supported yet.
     * It always returns false.
     * @return false
     * @throws java.sql.SQLException
     */
    public boolean getMoreResults() throws SQLException {
        return getMoreResults(CLOSE_CURRENT_RESULT);
    }

    /**
     * Supported fetch direction is FETCH_FORWARD only.
     * @param direction
     * @throws java.sql.SQLException
     */
    public void setFetchDirection(int direction) throws SQLException {
        JdbcResultSet.validateResultSetFetchDirection(direction);
        validateStatementOpen();
        
        if (direction != ResultSet.FETCH_FORWARD) {
            throw new SQLException("Not supported fetch direction.");
        }
    }

    /**
     * It always returns FETCH_FORWARD.
     * @return java.sql.ResultSet.FETCH_FORWARD
     * @throws java.sql.SQLException
     */
    public int getFetchDirection() throws SQLException {
        validateStatementOpen();

        return ResultSet.FETCH_FORWARD;
    }

    /**
     * Not supporetd yet.
     * @param rows
     * @throws java.sql.SQLException
     */
    public void setFetchSize(int rows) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * It always returns 0.
     * @return 0
     * @throws java.sql.SQLException
     */
    public int getFetchSize() throws SQLException {
        validateStatementOpen();

        return 0;
    }

    /**
     * It always returns CONCUR_READ_ONLY.
     * @return java.sql.ResultSet.CONCUR_READ_ONLY
     * @throws java.sql.SQLException
     */
    public int getResultSetConcurrency() throws SQLException {
        validateStatementOpen();

        return ResultSet.CONCUR_READ_ONLY;
    }

    /**
     * It always returns TYPE_FORWARD_ONLY.
     * @return java.sql.ResultSet.TYPE_FORWARD_ONLY
     * @throws java.sql.SQLException
     */
    public int getResultSetType() throws SQLException {
        validateStatementOpen();

        return ResultSet.TYPE_SCROLL_INSENSITIVE;
    }

    public void addBatch(String sql) throws SQLException {
        validateStatementOpen();

        if (batch == null) {
            batch = new ArrayList<String>();
        }
        batch.add(sql);
    }

    public void clearBatch() throws SQLException {
        validateStatementOpen();

        batch = null;
    }

    public int[] executeBatch() throws SQLException {
        validateStatementOpen();

        if (batch == null) {
            return new int[0];
        }
        
        int[] ret = new int[batch.size()];
        BatchUpdateException ex = null;
        for (int i = 0; i < ret.length; ++i) {
            try {
                ret[i] = executeUpdate(batch.get(i));
            } catch (SQLException ex2) {
                if (ex == null) {
                    ex = new BatchUpdateException(ex2.getMessage(), ex2.getSQLState(), ex2.getErrorCode(), ret);
                }
                ret[i] = EXECUTE_FAILED;
            }
        }
        if (ex != null) {
            throw ex;
        }
        return ret;
    }

    public Connection getConnection() throws SQLException {
        validateStatementOpen();

        return owner;
    }

    /**
     * CallableStatement is not supported yet.
     * It always returns false.
     * @param current
     * @return false
     * @throws java.sql.SQLException
     */
    public boolean getMoreResults(int current) throws SQLException {
        validateCurrentResult(current);
        validateStatementOpen();

        if (current != KEEP_CURRENT_RESULT && rs != null) {
            rs.close();
            rs = null;
        }
        return false;
    }

    /**
     * invoke executeQuery("SELECT last_insert_rowid()") method.
     * @return
     * @throws java.sql.SQLException
     */
    public ResultSet getGeneratedKeys() throws SQLException {
        return executeQuery("SELECT last_insert_rowid()");
    }

    public int executeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
        validateAutoGeneretedKeys(autoGeneratedKeys);
        validateStatementOpen();

        return executeUpdate(sql);
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public int executeUpdate(String sql, int[] columnIndexes) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public int executeUpdate(String sql, String[] columnNames) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public boolean execute(String sql, int autoGeneratedKeys) throws SQLException {
        validateAutoGeneretedKeys(autoGeneratedKeys);
        validateStatementOpen();

        return execute(sql);
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public boolean execute(String sql, int[] columnIndexes) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Not supporetd yet.
     * @throws java.sql.SQLException
     */
    public boolean execute(String sql, String[] columnNames) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * It always returns CLOSE_CURSORS_AT_COMMIT.
     * @return java.sql.ResultSet.CLOSE_CURSORS_AT_COMMIT
     * @throws java.sql.SQLException
     */
    public int getResultSetHoldability() throws SQLException {
        validateStatementOpen();

        return ResultSet.CLOSE_CURSORS_AT_COMMIT;
    }

    public boolean isClosed() throws SQLException {
        return (db == null || db.isClosed());
    }
    // END implements

    @Override
    protected void finalize() throws Throwable {
        if (db != null) {
            Logger.getLogger(JdbcStatement.class.getName()).warning("Statement is not closed.");
        }
        close();
        super.finalize();
    }
    
    protected void validateStatementOpen() throws SQLException {
        if (isClosed()) {
            throw new SQLException("Statement is already closed.");
        }
    }
    
    protected void validateStaticSQL(org.sqlite.Statement stmt) throws SQLException {
        if (stmt.getParameterCount() != 0) {
            stmt.close();
            throw new SQLException("Not static SQL.");            
        }
    }
    
    public static void validateAutoGeneretedKeys(int autoGeneratedKeys) throws SQLException {
        if (autoGeneratedKeys != NO_GENERATED_KEYS
                && autoGeneratedKeys != RETURN_GENERATED_KEYS) {
            throw new SQLException("Not supported auto generated keys.");
        }
    }
    
    public static void validateCurrentResult(int current) throws SQLException {
        if (current != CLOSE_CURRENT_RESULT
                && current != KEEP_CURRENT_RESULT
                && current != CLOSE_ALL_RESULTS) {
            throw new SQLException("Not supported current result.");
        }
    }
    
    /**
     * Statement#close()時にResultSetも一緒にcloseされないようにResultSetを切り離す。
     * @param drs
     */
    public void detach(ResultSet drs) throws SQLException {
        validateStatementOpen();

        if (rs != null && rs == drs) {
            rs = null;
        }
    }
    
    public long getLastInsertRowId() throws SQLException {
        validateStatementOpen();
        
        return db.lastInsertRowId();
    }
}
