package org.sqlite.udf;

import java.sql.SQLException;
import java.util.logging.Logger;

/**
 * User-Defined aggregate function class.
 * @author calico
 * @see <a href="http://sqlite.org/c3ref/create_function.html">Create Or Redefine SQL Functions</a>
 * @see org.sqlite.jdbc.JdbcConnection#createFunction(Function)
 * @see org.sqlite.jdbc.JdbcConnection#dropFunction(Function)
 * @see org.sqlite.jdbc.JdbcConnection#dropFunction(java.lang.String)
 */
public abstract class AggregateFunction<T> extends Function {
    public AggregateFunction(String name, int argc) {
        super(name, argc);
    }

    public AggregateFunction(String name) {
        super(name);
    }
    
    @Override
    protected final void xFunc(Context ctx) throws SQLException {
        xStep(ctx);
    }
    
    /**
     * Called from the sqlite3_step() function.
     * @param context
     */
    protected final void xFinal(long context) {
        // TODO Mysaifu JVMのBug#11980が解決したらアクセス修飾子をprivateに戻すこと！
        // @see http://sourceforge.jp/tracker/index.php?func=detail&aid=11980&group_id=1890&atid=7027
        Context ctx = null;
        try {
            ctx = new Context(context);
            xFinal(ctx);
            
        } catch (Throwable th) {
            if (ctx != null) {
                final String msg = th.toString();
                ctx.resultError((msg != null ? msg : "Unknown error."));
                
            } else {
                Logger.getLogger(AggregateFunction.class.getName()).fine("Exception occurred: " + th.toString());
            }
        }
    }

    /** thraed local storage */
    protected final ThreadLocal<T> tls
            = new ThreadLocal<T>() {
                    @Override
                    protected T initialValue() {
                        return internalInitialValue();
                    }
                };
    
    private T internalInitialValue() {
        return initialValue();
    }
    
    /**
     * Returns the current thread's "initial value" for this thread-local variable. 
     * @return the initial value for this thread-local
     * @see java.lang.ThreadLocal#initialValue()
     */
    protected abstract T initialValue();
    
    /**
     * Returns the value in the current thread's copy of this thread-local variable.
     * @return the current thread's value of this thread-local
     * @see java.lang.ThreadLocal#get()
     */
    protected T get() {
        return tls.get();
    }
    
    /**
     * Removes the current thread's value for this thread-local variable.
     * @see java.lang.ThreadLocal#remove()
     */
    protected void remove() {
        tls.remove();
    }
    
    /**
     * Sets the current thread's copy of this thread-local variable to the specified value.
     * @param value the value to be stored in the current thread's copy of this thread-local.
     * @see java.lang.ThreadLocal#set(Object)
     */
    protected void set(T value) {
        tls.set(value);
    }
    
    protected abstract void xStep(Context ctx) throws SQLException;

    protected abstract void xFinal(Context ctx) throws SQLException;
}
