import java.io.*;
import java.sql.*;
import org.sqlite.Context;
import org.sqlite.jdbc.JdbcConnection;
import org.sqlite.jdbc.JdbcPreparedStatement;
import org.sqlite.udf.AggregateFunction;
import org.sqlite.udf.Function;
import org.sqlite.udf.ScalarFunction;

public class HelloSQLite {
    public static void main(String[] args) throws Exception {
        // register 'org.sqlite.Driver' class
        Class.forName("org.sqlite.Driver");

        // open connection
        String url = null;
        if (args.length == 0) {
            System.out.println("open database in-memoery mode.");
            url = "jdbc:sqlite:mem:";
        } else {
            System.out.println("open database from '" + args[0] + "'.");
            url = "jdbc:sqlite:file:" + args[0];
        }
        final Connection conn = DriverManager.getConnection(url);

        try {
            // create table if not exists
            Statement stmt = conn.createStatement();
            stmt.executeUpdate(
                    "CREATE TABLE IF NOT EXISTS sample("
                        + "  id INTEGER PRIMARY KEY"
                        + ", tVal TEXT"
                        + ", bVal BLOB"
                        + ", rVal REAL"
                    + ")"
                );
            stmt.close();

            // insert new record by using a named parameter
            conn.setAutoCommit(false);
            JdbcPreparedStatement pstmt
                = (JdbcPreparedStatement) conn.prepareStatement(
                        "INSERT INTO sample(tVal, bVal, rVal) "
                        + "VALUES(@text, @blob, @real)"
                    );
            for (int i = 0; i < 5; ++i) {
                String text = Character.toString((char) ('1' + i));
                byte[] blob = ("Hello, SQLite3!" + text).getBytes();
                pstmt.setString("@text", text);
                pstmt.setBytes("@blob", blob);
                pstmt.setDouble("@real", Math.sqrt(i));
                pstmt.executeUpdate();
            }
            pstmt.close();
            conn.commit();

            // select all
            System.out.println("** select all records **");
            stmt = conn.createStatement();
            ResultSet rs = stmt.executeQuery("SELECT * FROM sample");
            ResultSetMetaData rsMeta = rs.getMetaData();
            final int cntCol = rsMeta.getColumnCount();
            while (rs.next()) {
                for (int i = 1; i <= cntCol; ++i) {
                    String val = null;
                    if (rsMeta.getColumnType(i) != Types.BLOB) {
                        val = rs.getString(i);
                    } else {
                        // convert UNICODE to java.lang.String
                        val = new String(rs.getBytes(i));
                    }
                    System.out.print(val + "|");
                }
                System.out.println();
            }
            rs.close();

            // create scalar function
            Function myFunc
                    = new ScalarFunction("strcat") {
                        @Override
                        protected void xFunc(Context ctx) throws SQLException {
                            if (!ctx.hasValues()) {
                                throw new IllegalArgumentException("argument is 0");
                            }

                            final StringBuilder ret = new StringBuilder();
                            for (int i = 0; i < ctx.getArgumentCount(); ++i) {
                                final int idx = i + 1;
                                ret.append(ctx.getString(idx));
                            }
                            ctx.result(ret.toString());
                        }
                    };

            // register function
            ((JdbcConnection) conn).registerFunction(myFunc);
            
            // execute scalar function
            System.out.println("** execute strcat **");
            rs = stmt.executeQuery("SELECT strcat(id, bVal) FROM sample");
            while (rs.next()) {
                System.out.println(rs.getString(1));
            }
            rs.close();
            
            // unregister function
            ((JdbcConnection) conn).unregisterFunction(myFunc);
            
            // create aggregate function
            myFunc
                = new AggregateFunction("initialcat", 1) {
                    private final ThreadLocal<StringBuilder> tlBuff
                            = new  ThreadLocal<StringBuilder>() {
                                    @Override
                                    protected StringBuilder initialValue() {
                                        return new StringBuilder();
                                    }
                                };

                    @Override
                    public void xStep(Context ctx) throws SQLException {
                        if (!ctx.hasValues()) {
                            throw new IllegalArgumentException("argument is 0");
                        }

                        // get StringBuilder from ThreadLocal
                        final StringBuilder buff = tlBuff.get();
                        buff.append(ctx.getString(1).charAt(0));
                        tlBuff.set(buff);
                    }

                    @Override
                    public void xFinal(Context ctx) throws SQLException {
                        ctx.result(tlBuff.get().toString());
                    }
                };

            // register function
            ((JdbcConnection) conn).registerFunction(myFunc);

            // execute aggregate function
            System.out.println("** execute initialcat **");
            rs = stmt.executeQuery("SELECT initialcat(tVal) FROM sample");
            while (rs.next()) {
                System.out.println(rs.getString(1));
            }
            rs.close();

            // unregister function
            ((JdbcConnection) conn).unregisterFunction(myFunc);

            stmt.close();

        } finally {
            // close connection
            conn.close();
        }
    }
}
