package org.sqlite;

import java.sql.SQLException;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.logging.Logger;
import org.sqlite.jdbc.JdbcSQLException;
import org.sqlite.swig.SQLite3ColumnMetaData;
import org.sqlite.types.SQLite3Handle;
import org.sqlite.swig.SWIGTYPE_p_p_sqlite3;
import org.sqlite.swig.SWIGTYPE_p_sqlite3;
import org.sqlite.types.SQLite3StmtPtrPtr;
import static org.sqlite.swig.SQLite3.*;

/**
 * sqlite3 wrapper class.
 * NOTE: SQLite 3.3.5 based.
 * @author calico
 */
public class Database {
    
    private final SQLite3Handle handle = new SQLite3Handle();
    private Map<Long, Statement> statements;
    protected final Properties info;
    private final boolean isInMemory;
    
    /** timeout(ms) : sqlite3_busy_timeout */
    private int timeout;
    
    public Database(String filename, Properties info) throws SQLException {
        this.info = info;
        this.isInMemory
                = (filename == null || getInMemoryFileName().equals(filename));
        open(filename);
    }
    
    public String getProductName() {
        return "SQLite";
    }
    
    public boolean isInMemoryMode() {
        return isInMemory;
    }
    
    /**
     * invoke sqlite3_open() function.
     * @param filename  filename
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/open.html">Opening A New Database Connection</a>
     */
    private void open(String filename) throws SQLException {
        final SWIGTYPE_p_p_sqlite3 ppDb = handle.getSQLite3PtrPtr();
        int ret = sqlite3_open(filename, ppDb);
        if (ret != SQLITE_OK) {
            SWIGTYPE_p_sqlite3 db = handle.getInstance();
            SQLException ex = new JdbcSQLException(db);
            handle.delete();
            throw ex;
        }
    }
    
    public boolean isReadOnly() {
        return false;
    }
    
    /**
     * Retrieves whether this Database object has been closed.
     * @return true if this Database object is closed. false if it is still open.
     */
    public boolean isClosed() {
        return (handle.isDeleted());
    }
    
    /**
     * invoke sqlite3_close() function.
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/close.html">Closing A Database Connection</a>
     */
    public void close() throws SQLException {
        if (!isClosed()) {
            // TODO もし Statement#closeForced() で例外が発生した場合、Database が閉じられない
            closeStatements();

            final SWIGTYPE_p_sqlite3 db = handle.getInstance();
            int ret = sqlite3_close(db);
            if (ret != SQLITE_OK) {
                throw new JdbcSQLException(db);
            }
            handle.delete();
        }
    }
    
    /**
     * invoke sqlite3_get_autocommit() function.
     * @return  true if auto commit mode.
     * @see <a href="http://sqlite.org/c3ref/get_autocommit.html">Test To See If The Database Is In Auto-Commit Mode</a>
     */
    public boolean getAutoCommit() {
        final SWIGTYPE_p_sqlite3 db = handle.getInstance();
        return (sqlite3_get_autocommit(db) != 0);
    }
    
    /**
     * invoke sqlite3_busy_timeout() function.
     * @param ms    milliseconds
     * @return
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/busy_timeout.html">Set A Busy Timeout</a>
     */
    public int setBusyTimeout(int ms) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = handle.getInstance();
        int ret = sqlite3_busy_timeout(db, ms);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(db);
        }
        timeout = (ms < 1 ? 0 : ms);
        return ret;
    }

    /**
     * Retrieves the timeout(ms) value.
     * @return  timeout(ms) value.
     */
    public int getBusyTimeout() {
        return timeout;
    }
    
    /**
     * invoke sqlite3_exec() function.
     * @param sql
     * @return
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/exec.html">One-Step Query Execution Interface</a>
     */
    public int execute(String sql) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = handle.getInstance();
        int ret = 0;
        if (timeout == 0) {
            // no limit
            while ((ret = sqlite3_exec(db, sql)) == SQLITE_BUSY) {
                // waiting...
            }
        } else {
            ret = sqlite3_exec(db, sql);
            if (ret == SQLITE_BUSY) {
                // timeout
                throw new SQLException("Timeout expired.");
            }
        }
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(db);
        }
        return ret;
    }

    /**
     * execute PRAGMA commands by sqlite3_exec() finction.
     * @param commands  command list
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/pragma.html">Pragma statements supported by SQLite</a>
     */
    public void pragma(String[] commands) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = handle.getInstance();
        for (final String cmd : commands) {
            int ret = sqlite3_exec(db, "PRAGMA " + cmd);
            if (ret != SQLITE_OK) {
                throw new JdbcSQLException(db);
            }
        }
    }
    
    /**
     * begin transaction.
     * @param type  transaction type.
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/lang_transaction.html">BEGIN TRANSACTION</a>
     */
    public void beginTransaction(TransactionType type) throws SQLException {
        closeStatements();
        if (type == null) {
            execute("BEGIN");
        } else {
            execute("BEGIN " + type);
        }
    }
    
    /**
     * commit toransaction.
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/lang_transaction.html">BEGIN TRANSACTION</a>
     */
    public void commitTransaction() throws SQLException {
        closeStatements();
        execute("COMMIT");
    }
    
    /**
     * rollback transaction.
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/lang_transaction.html">BEGIN TRANSACTION</a>
     */
    public void rollbackTransaction() throws SQLException {
        closeStatements();
        execute("ROLLBACK");
    }
    
    /**
     * create MANAGED Statement instance.
     * @param sql
     * @param ppStmt
     * @return
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/prepare.html">Compiling An SQL Statement</a>
     */
    public Statement prepare(String sql, SQLite3StmtPtrPtr ppStmt) throws SQLException {
        if (sql == null) {
            throw new NullPointerException("sql is null.");
        }
        if (ppStmt == null) {
            throw new NullPointerException("ppStmt is null.");
        }
        
        final SWIGTYPE_p_sqlite3 db = handle.getInstance();
        int ret = sqlite3_prepare(db, sql, -1, ppStmt, null);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(db);
        }
        return new Statement(this, ppStmt.getSQLite3StmtPtr());
    }
    
    /**
     * create UNMANAGED Statement instance.
     * @param sql
     * @return
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/prepare.html">Compiling An SQL Statement</a>
     */
    public Statement prepare(String sql) throws SQLException {
        if (sql == null) {
            throw new NullPointerException("sql is null.");
        }

        final SWIGTYPE_p_sqlite3 db = handle.getInstance();
        final SQLite3StmtPtrPtr ppStmt = new SQLite3StmtPtrPtr();
        int ret = sqlite3_prepare(db, sql, -1, ppStmt, null);
        if (ret != SQLITE_OK) {
            ppStmt.delete();
            throw new JdbcSQLException(db);
        }
        return new Statement(this, ppStmt);
    }
    
    /**
     * invoke sqlite3_interrupt() function.
     * @see <a href="http://sqlite.org/c3ref/interrupt.html">Interrupt A Long-Running Query</a>
     */
    public void interrupt() throws SQLException {
        final SWIGTYPE_p_sqlite3 db = handle.getInstance();
        sqlite3_interrupt(db);
        closeStatements();
    }
    
    /**
     * invoke sqlite3_changes() function.
     * @return
     * @see <a href="http://sqlite.org/c3ref/changes.html">Count The Number Of Rows Modified</a>
     */
    public int changes() {
        final SWIGTYPE_p_sqlite3 db = handle.getInstance();
        return sqlite3_changes(db);
    }
    
    /**
     * invoke sqlite3_total_changes() function.
     * @return
     * @see <a href="http://sqlite.org/c3ref/total_changes.html">Total Number Of Rows Modified</a>
     */
    public int totalChanges() {
        final SWIGTYPE_p_sqlite3 db = handle.getInstance();
        return sqlite3_total_changes(db);
    }
    
    /**
     * invoke sqlite3_last_insert_rowid() function.
     * @return
     * @see <a href="http://sqlite.org/c3ref/last_insert_rowid.html">Last Insert Rowid</a>
     */
    public long lastInsertRowId() {
        final SWIGTYPE_p_sqlite3 db = handle.getInstance();
        return sqlite3_last_insert_rowid(db);
    }
    
    public SQLite3ColumnMetaData getColumnMetaData(String dbName, String tableName, String columnName) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = handle.getInstance();
        final SQLite3ColumnMetaData meta = new SQLite3ColumnMetaData();
        int ret = sqlite3_table_column_metadata_by_struct(db, dbName, tableName, columnName, meta);
        if (ret != SQLITE_OK) {
            meta.delete();
            throw new JdbcSQLException(db);
        }
        return meta;
    }
    
    void addStatement(Statement stmt) throws SQLException {
        if (statements == null) {
            statements = new HashMap<Long, Statement>();
        }
        final long key = stmt.getHandle();
        if (statements.containsKey(key)) {
            throw new SQLException("Duplicate sqlite3_stmt handle error.");
        }
        statements.put(key, stmt);
    }
    
    void removeStatement(Statement stmt) throws SQLException {
        final long key = stmt.getHandle();
        if (statements == null || statements.remove(key) == null) {
            throw new SQLException("Unmanaged sqlite3_stmt handle error.");
        }
    }
    
    private void closeStatements() throws SQLException {
        if (statements != null) {
            for (final Statement stmt : statements.values()) {
                stmt.closeForced();
            }
            statements = null;
        }
    }
    
    // TODO sqlite3_create_function()に対応する！
    
    // TODO Statement毎にtimeoutが設定できるようにする！
    
    @Override
    protected void finalize() throws Throwable {
        if (!isClosed()) {
            // sqlite3_openを呼び出したスレッドと異なるスレッドから
            // sqlite3_closeを呼び出すことは出来ない（呼び出すとJVMが異常終了する）
            Logger.getLogger("global").severe("Database connection has leaked!");
        }
        super.finalize();
    }
    
}