﻿/*
 * マルチプラットフォーム描画エンジン「Sherry」
 * Copyright(C) 2010-2011 SherryProject. all rights reserved.
 *
 * The MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/*
 *	shMesh.c
 *		Copyright (C) 2010-2011 Cap5ule. all rights reserved.
 * 
 *	Date		|Version	|Author		|Summary
 *	2010/12/14	|v0.01		|Cap5ule	|初回版
 */


/*---- インクルードファイル ----*/

#include "sherry.h"			//!<共通ヘッダー
#include "shApplication.h"	//!<必要なSHヘッダー
#include "shMath.h"
#include "shMesh.h"			//!<ヘッダー

#include <xmlreader.h>		//!<LIBXML2のXMLリーダーヘッダー

/*---- マクロ定義 ----*/

#define MAX_SHNAME_DEVIDENUM 14		//!<要素名前の数
#define MAX_SHNAME_LENGTH 24		//!<解析用の名前の最大文字数(byte)
#define TRIANGLE_NUM 3				//!<三角形の頂点数

/*---- スタティックプロトタイプ宣言 ----*/
// 説明は実装部に記載

// COLLADAの要素名前から処理を分ける
static int COLLADA_NameDivide(const char* str);
// COLLADAの要素名前から入れ子関係を管理するフラグスイッチ
static void COLLADA_NameFlagSwitch(int procNo,
								   shColladaInfo* pCI);
// COLLADAの要素別処理関数群
static int COLLADA_Process_COLLADA(xmlTextReaderPtr pRd,
								   shColladaInfo* pCI);			//!<COLLADA
static int COLLADA_Process_asset(xmlTextReaderPtr pRd,
								   shColladaInfo* pCI);			//!<asset
static int COLLADA_Process_library_geometries(xmlTextReaderPtr pRd,
								   shColladaInfo* pCI);			//!<library_geometries
static int COLLADA_Process_geometry(xmlTextReaderPtr pRd,
								   shColladaInfo* pCI);			//!<geometry
static int COLLADA_Process_mesh(xmlTextReaderPtr pRd,
								   shColladaInfo* pCI);			//!<mesh
static int COLLADA_Process_source(xmlTextReaderPtr pRd,
								   shColladaInfo* pCI);			//!<source
static int COLLADA_Process_float_array(xmlTextReaderPtr pRd,
								   shColladaInfo* pCI);			//!<float_array
static int COLLADA_Process_technique_common(xmlTextReaderPtr pRd,
								   shColladaInfo* pCI);			//!<technique_common
static int COLLADA_Process_accessor(xmlTextReaderPtr pRd,
								   shColladaInfo* pCI);			//!<accessor
static int COLLADA_Process_param(xmlTextReaderPtr pRd,
								   shColladaInfo* pCI);			//!<param
static int COLLADA_Process_vertices(xmlTextReaderPtr pRd,
								   shColladaInfo* pCI);			//!<vertices
static int COLLADA_Process_triangles(xmlTextReaderPtr pRd,
								   shColladaInfo* pCI);			//!<triangles
static int COLLADA_Process_input(xmlTextReaderPtr pRd,
								   shColladaInfo* pCI);			//!<input
static int COLLADA_Process_p(xmlTextReaderPtr pRd,
								   shColladaInfo* pCI);			//!<p
// COLLADAの要素別テキスト処理関数群
static int COLLADA_ProcessText_float_array(xmlTextReaderPtr pRd,
										   shColladaInfo* pCI);	//!<float_array
static int COLLADA_ProcessText_p(xmlTextReaderPtr pRd,
								 shColladaInfo* pCI);			//!<p
// COLLADAデータフォーマットからメッシュデータへ変換
static int COLLADAToSHMeshData(shColladaInfo* pCI, shMesh* pMesh);
// COLLADAデータフォーマットのメモリを解放する
static int COLLADAFreeInfo(shColladaInfo* pCI);
// ＩＤ文字列を文字列としてみる
static const char* IDStringIsString(const char* srcIdStr);

/*---- グローバル変数 ----*/

/*
 *	static char* g_sh_NameDeivde
 *
 *	名前要素の割振り文字列比較用
 */
static char* g_sh_NameDeivde[MAX_SHNAME_LENGTH] = {
	"COLLADA",
	"asset",
	"library_geometries",
	"geometry",
	"mesh",
	"source",
	"float_array",
	"technique_common",
	"accessor",
	"param",
	"vertices",
	"triangles",
	"input",
	"p",
};

/*
 *	static int (*g_sh_COLLADAProcessFunc[])(xmlTextReaderPtr pRd,
 *											shColladaInfo* pCI)
 *
 *	名前要素の処理を関数別バインド用配列
 */
static int (*g_sh_COLLADAProcessFunc[])(xmlTextReaderPtr pRd,
											shColladaInfo* pCI) = {
	COLLADA_Process_COLLADA,
	COLLADA_Process_asset,
	COLLADA_Process_library_geometries,
	COLLADA_Process_geometry,
	COLLADA_Process_mesh,
	COLLADA_Process_source,
	COLLADA_Process_float_array,
	COLLADA_Process_technique_common,
	COLLADA_Process_accessor,
	COLLADA_Process_param,
	COLLADA_Process_vertices,
	COLLADA_Process_triangles,
	COLLADA_Process_input,
	COLLADA_Process_p,
};

/*
 *	static int (*g_sh_COLLADAProcessTextFunc[])(xmlTextReaderPtr pRd,
 *											shColladaInfo* pCI)
 *
 *	テキストの処理を関数別バインド用配列
 */
static int (*g_sh_COLLADAProcessTextFunc[])(xmlTextReaderPtr pRd,
											shColladaInfo* pCI) = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	COLLADA_ProcessText_float_array,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	COLLADA_ProcessText_p,
};

/*---- スタティック関数 ----*/

/*
 *	static int COLLADA_NameDivide(const char* token)
 *
 *	＜引数＞
 *　　char* token	: 要素の名前
 *	＜戻り値＞
 *　　int			: 処理配列番号
 *	＜説明＞
 *　　COLLADAの要素名前から処理番号を取得する
 *　　要素名前から処理を分けるために配列番号に変える
 */
static int COLLADA_NameDivide(const char* token)
{
	int i;	//カウンタ

	// 一致処理
	for(i = 0; i < MAX_SHNAME_DEVIDENUM; i++) {
		if(xmlStrcmp(token, g_sh_NameDeivde[i]) == 0)
			return i;	//一致したのでカウンタを返す
	}

	return (-1);	//見当たらない場合はエラー
}

/*
 *	static void COLLADA_NameFlagSwitch(int procNo, shColladaInfo* pCI)
 *
 *	＜引数＞
 *  shColladaInfo* pCI	: COLLADA解析用情報
 *  int procNo			: 処理配列番号
 *	＜戻り値＞
 *  なし
 *	＜説明＞
 *  要素名前から入れ子関係を管理するフラグをスイッチングする
 */
static void COLLADA_NameFlagSwitch(int procNo, shColladaInfo* pCI)
{
	pCI->flags ^= 0x00000001 << procNo;
}


// COLLADAの要素別処理関数群
// <COLLADA>タグ
static int COLLADA_Process_COLLADA(xmlTextReaderPtr pRd, shColladaInfo* pCI)
{
	int attrNum = xmlTextReaderAttributeCount(pRd);
	// version
	xmlChar* attrValue = xmlTextReaderGetAttribute(pRd,"version");
	strcpy(pCI->version, attrValue);

	return 0;
}

// <asset>タグ
static int COLLADA_Process_asset(xmlTextReaderPtr pRd, shColladaInfo* pCI)
{
	// ASSET開始フラグ
	/*なにもしない*/
	return 0;
}

// <geometries>タグ
static int COLLADA_Process_library_geometries(xmlTextReaderPtr pRd, shColladaInfo* pCI)
{
	// library_geometrie開始フラグ
	/*なにもしない*/
	return 0;
}

// <geometry>タグ
static int COLLADA_Process_geometry(xmlTextReaderPtr pRd, shColladaInfo* pCI)
{
	//library_geometrie入れ子確認
	if(pCI->flags & CISHFLAGS_LIBGEOMETRIES) {

		xmlChar* attrValue_id;
		xmlChar* attrValue_name;

		struct _shGeometriesNode *newNode, *curNode;
		// ジオメトリの末端ノードを検索
		curNode = pCI->shGeometriesNode_first;
		if(curNode != NULL) {
			while(curNode->next != NULL)
				curNode = curNode->next;
		}

		// ノードデータのメモリ確保
		newNode = (struct _shGeometriesNode*)malloc(sizeof(struct _shGeometriesNode));
		memset(newNode,0,sizeof(struct _shGeometriesNode));
		if(curNode == NULL)				// 前のノードがない場合
			pCI->shGeometriesNode_first = newNode;
		else
			curNode->next = newNode;	// 新しいノードを入れる

		// id, name
		attrValue_id = xmlTextReaderGetAttribute(pRd,"id");
		attrValue_name = xmlTextReaderGetAttribute(pRd,"name");

		strcpy(newNode->id, attrValue_id);
		strcpy(newNode->name, attrValue_name);

		// カレントノードにする
		pCI->shGeometriesNode_cur = newNode;

		//xmlFree( attrValue );

		return 0;
	}

	return 1;
}

// <mesh>タグ
static int COLLADA_Process_mesh(xmlTextReaderPtr pRd, shColladaInfo* pCI)
{
	/*なにもなし*/
	return 0;
}

// <source>タグ
static int COLLADA_Process_source(xmlTextReaderPtr pRd, shColladaInfo* pCI)
{
	xmlChar* attrValue_id;
	struct _shSourceNode *newNode, *curNode;

	if(!(pCI->flags & CISHFLAGS_GEOMETRY))
		return 0;
	if(!(pCI->flags & CISHFLAGS_MESH))
		return 0;

	// ジオメトリの末端ノードを検索
	curNode = pCI->shGeometriesNode_cur->shSourceNode_first;
	if(curNode != NULL) {
		while(curNode->next != NULL)
			curNode = curNode->next;
	}

	// ノードデータのメモリ確保
	newNode = (struct _shSourceNode*)malloc(sizeof(struct _shSourceNode));
	memset(newNode,0,sizeof(struct _shSourceNode));
	if(curNode == NULL)				// 前のノードがない場合
		pCI->shGeometriesNode_cur->shSourceNode_first = newNode;
	else
		curNode->next = newNode;	// 新しいノードを入れる

	// id
	attrValue_id = xmlTextReaderGetAttribute(pRd,"id");
	strcpy(newNode->id, attrValue_id);

	// カレントノードにする
	pCI->shSourceNode_cur = newNode;

	return 0;
}

// <float_array>タグ
static int COLLADA_Process_float_array(xmlTextReaderPtr pRd, shColladaInfo* pCI)
{
	// float型配列のデータを読み込む
	// 実際の読み込みはTextにて

	struct _shSourceNode* curNode = pCI->shSourceNode_cur;

	if(!(pCI->flags & CISHFLAGS_GEOMETRY))
		return 0;
	if(!(pCI->flags & CISHFLAGS_MESH))
		return 0;
	if(!(pCI->flags & CISHFLAGS_SOURCE))
		return 0;

	if(curNode != NULL) {
		// id, count
		xmlChar* attrValue_id = xmlTextReaderGetAttribute(pRd,"id");
		xmlChar* attrValue_count = xmlTextReaderGetAttribute(pRd,"count");
		strcpy(curNode->id_array, attrValue_id);
		curNode->count = (unsigned short)atoi(attrValue_count);
		curNode->arrayType = floatType;	// float型
		// メモリ確保
		curNode->float_array = (float*)malloc(sizeof(float) * curNode->count);
		memset(curNode->float_array,0,sizeof(float) * curNode->count);	// 初期化

		// 次回処理を許可：メモリを指定
		pCI->array_cur.float_array = curNode->float_array;

		return 0;
	}

	return 1;
}

// <common>タグ
static int COLLADA_Process_technique_common(xmlTextReaderPtr pRd, shColladaInfo* pCI)
{
	/*なにもなし*/
	return 0;
}

// <accessor>タグ
static int COLLADA_Process_accessor(xmlTextReaderPtr pRd, shColladaInfo* pCI)
{
	struct _shSourceNode* curNode = pCI->shSourceNode_cur;

	if(curNode != NULL) {
		// id, count, stride
		xmlChar* attrValue_id = xmlTextReaderGetAttribute(pRd,"source");
		xmlChar* attrValue_count = xmlTextReaderGetAttribute(pRd,"count");
		xmlChar* attrValue_stride = xmlTextReaderGetAttribute(pRd,"stride");

		strcpy(curNode->shAccessor.id_source, attrValue_id);
		curNode->shAccessor.count = (unsigned short)atoi(attrValue_count);
		curNode->shAccessor.stride = (unsigned short)atoi(attrValue_stride);

		// <param>のメモリを確保
		curNode->shAccessor.param_name = (char*)malloc(
			sizeof(char)*MAX_SHCHARLENGTH*curNode->shAccessor.count);
		curNode->shAccessor.param_type = (shDataType*)malloc(
			sizeof(shDataType)*curNode->shAccessor.count);

		pCI->count_cur = 0;

		return 0;
	}

	return 1;
}

// <param>タグ
static int COLLADA_Process_param(xmlTextReaderPtr pRd, shColladaInfo* pCI)
{
	struct _shSourceNode* curNode = pCI->shSourceNode_cur;

	if(!(pCI->flags & CISHFLAGS_GEOMETRY))
		return 0;
	if(!(pCI->flags & CISHFLAGS_MESH))
		return 0;
	if(!(pCI->flags & CISHFLAGS_SOURCE))
		return 0;
	if(!(pCI->flags & CISHFLAGS_ACCESSOR))
		return 0;

	if(curNode != NULL) {
		// name, type
		xmlChar* attrValue_name = xmlTextReaderGetAttribute(pRd,"name");
		xmlChar* attrValue_type = xmlTextReaderGetAttribute(pRd,"type");
		int param_name_ptr = sizeof(char) * MAX_SHCHARLENGTH * pCI->count_cur;

		strcpy(&curNode->shAccessor.param_name[param_name_ptr], attrValue_name);
		if(strcmp(attrValue_type,"float") == 0) {
			curNode->shAccessor.param_type[pCI->count_cur] = floatType;
		}

		// 次のメモリへ
		pCI->count_cur++;

		return 0;
	}

	return 0;
}

// <vertices>タグ
static int COLLADA_Process_vertices(xmlTextReaderPtr pRd, shColladaInfo* pCI)
{
	struct _shGeometriesNode* curNode = pCI->shGeometriesNode_cur;

	if(!(pCI->flags & CISHFLAGS_GEOMETRY))
		return 0;
	if(!(pCI->flags & CISHFLAGS_MESH))
		return 0;

	if(curNode != NULL) {
		// id
		xmlChar* attrValue_id = xmlTextReaderGetAttribute(pRd,"id");
		strcpy(curNode->shVertices.id, attrValue_id);

		pCI->proc_cur = 1;	// 処理：<vertices>
		pCI->count_cur = 0;
	}

	return 0;
}

// <triangles>タグ
static int COLLADA_Process_triangles(xmlTextReaderPtr pRd, shColladaInfo* pCI)
{
	struct _shGeometriesNode* curNode = pCI->shGeometriesNode_cur;

	if(!(pCI->flags & CISHFLAGS_GEOMETRY))
		return 0;
	if(!(pCI->flags & CISHFLAGS_MESH))
		return 0;

	if(curNode != NULL) {
		// count
		xmlChar* attrValue_count = xmlTextReaderGetAttribute(pRd,"count");
		curNode->shTriangles.count = (unsigned short)atoi(attrValue_count);

		pCI->proc_cur = 2;	// 処理：<triangles>
		pCI->count_cur = 0;
	}

	return 0;
}

// <input>タグ
static int COLLADA_Process_input(xmlTextReaderPtr pRd, shColladaInfo* pCI)
{
	struct _shGeometriesNode* curNode_gn = pCI->shGeometriesNode_cur;
	xmlChar *attrValue_semantic, *attrValue_id, *attrValue_offset;

	if(!(pCI->flags & CISHFLAGS_GEOMETRY))
		return 0;
	if(!(pCI->flags & CISHFLAGS_MESH))
		return 0;
	if(!(pCI->flags & CISHFLAGS_SOURCE))
		return 0;

	if(pCI->proc_cur == 1) {
		// 処理：<vertices>
		struct _shVInputNode *curNode, *newNode;

		// Inputの末端ノードを検索
		curNode = curNode_gn->shVertices.shVInputNode_first;
		if(curNode != NULL) {
			while(curNode->next != NULL)
				curNode = curNode->next;
		}

		// ノードデータのメモリ確保
		newNode = (struct _shVInputNode*)malloc(sizeof(struct _shVInputNode));
		memset(newNode,0,sizeof(struct _shVInputNode));
		if(curNode == NULL)				// 前のノードがない場合
			curNode_gn->shVertices.shVInputNode_first = newNode;
		else
			curNode->next = newNode;	// 新しいノードを入れる

		// semantic, id
		attrValue_id = xmlTextReaderGetAttribute(pRd,"source");
		attrValue_semantic = xmlTextReaderGetAttribute(pRd,"semantic");
		strcpy(newNode->id_source, attrValue_id);
		strcpy(newNode->semantic, attrValue_semantic);

		curNode_gn->shVertices.inputNum++;

	} else if(pCI->proc_cur == 2) {
		// 処理：<triangles>

		struct _shTInputNode *curNode, *newNode;

		// Inputの末端ノードを検索
		curNode = curNode_gn->shTriangles.shTInputNode_first;
		if(curNode != NULL) {
			while(curNode->next != NULL)
				curNode = curNode->next;
		}

		// ノードデータのメモリ確保
		newNode = (struct _shTInputNode*)malloc(sizeof(struct _shTInputNode));
		memset(newNode,0,sizeof(struct _shTInputNode));
		if(curNode == NULL)				// 前のノードがない場合
			curNode_gn->shTriangles.shTInputNode_first = newNode;
		else
			curNode->next = newNode;	// 新しいノードを入れる

		// semantic, id
		attrValue_id = xmlTextReaderGetAttribute(pRd,"source");
		attrValue_semantic = xmlTextReaderGetAttribute(pRd,"semantic");
		attrValue_offset = xmlTextReaderGetAttribute(pRd,"offset");
		strcpy(newNode->id_source, attrValue_id);
		strcpy(newNode->semantic, attrValue_semantic);
		newNode->offset = (unsigned short)atoi(attrValue_offset);

		curNode_gn->shTriangles.inputNum++;
	}

	return 0;
}

// <p>タグ
static int COLLADA_Process_p(xmlTextReaderPtr pRd, shColladaInfo* pCI)
{
	//<input>によってPの必要メモリが変動するので
	struct _shGeometriesNode* curNode = pCI->shGeometriesNode_cur;

	if(!(pCI->flags & CISHFLAGS_GEOMETRY))
		return 0;
	if(!(pCI->flags & CISHFLAGS_MESH))
		return 0;

	if(curNode != NULL) {
		//<p>用のメモリを確保
		curNode->shTriangles.p = (unsigned short*)malloc(sizeof(unsigned short) *
			curNode->shTriangles.inputNum * curNode->shTriangles.count * TRIANGLE_NUM);

		// 次回処理を許可：メモリを指定
		pCI->array_cur.p_array = curNode->shTriangles.p;
	}

	return 0;
}

// COLLADAの要素別テキスト処理関数群
// <float_array>テキスト</float_array>
static int COLLADA_ProcessText_float_array(xmlTextReaderPtr pRd,
										   shColladaInfo* pCI)
{
	// カウンタ
	int mem_i = 0;
	// 処理されるポインタ
	float* float_array = pCI->array_cur.float_array;
	// テキスト読み込み
	xmlChar* textValue = xmlTextReaderValue(pRd);
	const char* delim = " ";
	// 現在のテキストポインタ位置
	char *cp = strtok(textValue," ");

	// 区切り
	do {
		//抽出
		float_array[mem_i++] = (float)atof(cp);
		cp = strtok(NULL," ");
	} while(cp != NULL);

	pCI->array_cur.float_array = NULL;

	return 0;
}

// <p>テキスト</p>
static int COLLADA_ProcessText_p(xmlTextReaderPtr pRd,
								 shColladaInfo* pCI)
{
	// カウンタ
	int mem_i = 0;
	// 処理されるポインタ
	unsigned short* p_array = pCI->array_cur.p_array;
	// テキスト読み込み
	xmlChar* textValue = xmlTextReaderValue(pRd);
	const char* delim = " ";
	// 現在のテキストポインタ位置
	char *cp = strtok(textValue," ");

	// 区切り
	do {
		//抽出
		p_array[mem_i++] = (unsigned short)atoi(cp);
		cp = strtok(NULL," ");
	} while(cp != NULL);

	pCI->array_cur.p_array = NULL;

	return 0;
}

/*
 *	static int COLLADAToSHMeshData(shColladaInfo* pCI, shMesh* pMesh)
 *
 *	＜引数＞
 *　　shColladaInfo* pCI	: COLLADAフォーマット情報ポインタ
 *　　shMesh* pMesh		: メッシュデータポインタ
 *	＜戻り値＞
 *　　int					: エラーコードが返ります
 *	＜説明＞
 *　　COLLADAフォーマットからメッシュデータに変換します。
 *　　※形式によっては対応していない場合があります。
 */
static int COLLADAToSHMeshData(shColladaInfo* pCI, shMesh* pMesh)
{
	// カウンタ
	int i;
	// ジオメトリノード
	struct _shGeometriesNode* curNode = pCI->shGeometriesNode_first;

	// ジオメトリを調べる
	while(curNode != NULL)
	{
		// ルーパー
		int loop;
		// 必要変数
		struct _shTInputNode* curTINode;
		struct _shVInputNode* curVINode;
		struct _shSourceNode* curSrcNode;
		int inputNum;
		// 保存用
		struct _shSourceNode* position_src = NULL;
		struct _shSourceNode* texcoord_src = NULL;

		// 頂点数を入れる
		pMesh->vertex_size = (int)curNode->shTriangles.count * TRIANGLE_NUM;
		// ポリゴン数を入れる
		pMesh->face_size = (int)curNode->shTriangles.count;

		// メモリを確保
		pMesh->index = (unsigned short*)malloc(pMesh->vertex_size*sizeof(unsigned short));
		pMesh->vartex_p = (shVector3*)malloc(pMesh->vertex_size*sizeof(shVector3));
		pMesh->vartex_uv = (shVector2*)malloc(pMesh->vertex_size*sizeof(shVector2));

		// [VERTEX]のソースを取得する
		curTINode = curNode->shTriangles.shTInputNode_first;
		while(curTINode != NULL) {
			if(strcmp(curTINode->semantic, "VERTEX") == 0) {
				// <source>のidを調べる
				const char* srcId = IDStringIsString(curTINode->id_source);
				
				//ソースを検索
				loop = 1;
				while(loop) {
					curSrcNode = curNode->shSourceNode_first;
					while(curSrcNode != NULL) {
						if(strcmp(curSrcNode->id,srcId) == 0) {
							
							position_src = curSrcNode;
							loop = 0;
						}
						curSrcNode = curSrcNode->next;
					}
					//なければVertices内のInput
					curVINode = curNode->shVertices.shVInputNode_first;
					while(curVINode != NULL) {
						if(strcmp(curVINode->semantic, "POSITION") == 0) {
							srcId = IDStringIsString(curVINode->id_source);
						}
						curVINode = curVINode->next;
					}
				}
			}
			curTINode = curTINode->next;
		}

		// [TEXCOORD]のソースを取得する
		curTINode = curNode->shTriangles.shTInputNode_first;
		while(curTINode != NULL) {
			if(strcmp(curTINode->semantic, "TEXCOORD") == 0) {
				// <source>のidを調べる
				const char* srcId = IDStringIsString(curTINode->id_source);
				
				//ソースを検索
				loop = 1;
				while(loop) {
					curSrcNode = curNode->shSourceNode_first;
					while(curSrcNode != NULL) {
						if(strcmp(curSrcNode->id,srcId) == 0) {
							texcoord_src = curSrcNode;
							loop = 0;
						}
						curSrcNode = curSrcNode->next;
					}
				}
			}
			curTINode = curTINode->next;
		}

		// 頂点を入れる
		inputNum = curNode->shTriangles.inputNum;	//inputNum
		for(i = 0; i < pMesh->vertex_size; i++) {
			int offset_pos = curNode->shTriangles.p[i*inputNum + 0];
			int offset_tex = curNode->shTriangles.p[i*inputNum + 1];

			pMesh->vartex_p[i].x = position_src->float_array[offset_pos * 3 + 0];
			pMesh->vartex_p[i].y = position_src->float_array[offset_pos * 3 + 1];
			pMesh->vartex_p[i].z = position_src->float_array[offset_pos * 3 + 2];
			pMesh->vartex_uv[i].x = texcoord_src->float_array[offset_tex * 2 + 0];
			pMesh->vartex_uv[i].y = -texcoord_src->float_array[offset_tex * 2 + 1];
			pMesh->index[i] = i;
		}

		// 次のノード
		curNode = curNode->next;
	}
	
	return 0;
}

/*
 *	static int COLLADAFreeInfo(shColladaInfo* pCI)
 *
 *	＜引数＞
 *		shColladaInfo* pCI	: COLLADAフォーマット情報ポインタ
 *		shMesh* pMesh		: メッシュデータポインタ
 *	＜戻り値＞
 *		int					: エラーコードが返ります
 *	＜説明＞
 *		COLLADAデータフォーマットのメモリを解放する
 */
static int COLLADAFreeInfo(shColladaInfo* pCI)
{
	//全てのデータを解放し、０にする

	struct _shGeometriesNode* curNode_gn = pCI->shGeometriesNode_cur;
	// <Geometoryes>
	while(curNode_gn!=NULL) {
		struct _shGeometriesNode* curNode_gn_next = curNode_gn->next;						//!<Geometries_Next
		struct _shSourceNode* curNode_src = curNode_gn->shSourceNode_first;					//!<Source_First
		struct _shVInputNode* curNode_vIpt = curNode_gn->shVertices.shVInputNode_first;		//!<VInput_First
		struct _shTInputNode* curNode_tIpt = curNode_gn->shTriangles.shTInputNode_first;	//!<TInput_First

		// <mesh><source>
		while(curNode_src!=NULL) {
			struct _shSourceNode* curNode_src_next = curNode_src->next;	//!<Source_Next
			// 解放
			free(curNode_src->float_array);
			free(curNode_src->shAccessor.param_name);
			free(curNode_src->shAccessor.param_type);

			free(curNode_src);
			curNode_src = curNode_src_next;
		}

		// <vertices><input>
		while(curNode_vIpt!=NULL) {
			struct _shVInputNode* curNode_vIpt_next = curNode_vIpt->next;
			free(curNode_vIpt);
			curNode_vIpt = curNode_vIpt_next;
		}
		// <triangles><input>
		while(curNode_vIpt!=NULL) {
			struct _shTInputNode* curNode_tIpt_next = curNode_tIpt->next;
			free(curNode_tIpt);
			curNode_tIpt = curNode_tIpt_next;
		}

		free(curNode_gn);	//!<解放する
		curNode_gn = curNode_gn_next;
	}

	//メモリ内をクリア
	memset(pCI,0x00,sizeof(shColladaInfo));

	return 0;
}

/*
 *	static const char* IDStringIsString(const char* srcIdStr)
 *
 *	＜引数＞
 *　　char* srcStr		: ソース文字列
 *	＜戻り値＞
 *　　char*				: 通常の文字列を返す
 *	＜説明＞
 *　　「#」付きのID文字列を通常の文字列をして見る。比較に使える。
 */
static const char* IDStringIsString(const char* srcIdStr)
{
	// IDとしてみる「#」
	if(srcIdStr[0] == '#') {
		// 文字列として返す
		return &srcIdStr[1];
	}

	return &srcIdStr[0];
}

/*---- 関数 ----*/

/*
 *	int shCreateMeshArray(shMesh* pMesh, float* p, float* uv, unsigned short* idx,
 *					  int vsize, int idxsize);
 *	＜引数＞
 *　　shMesh* pMesh			: メッシュデータポインタ
 *　　float* p				: 頂点データ配列
 *　　float* uv				: ＵＶデータ配列
 *　　unsigned short* idx	: 結線配列
 *　　float* vsize			: 頂点数
 *　　float* idxsize		: ポリゴン数
 *	＜戻り値＞
 *　　int					: エラーコードが返ります
 *	＜説明＞
 *　　用意された配列からメッシュデータを作成します。
 */
int shCreateMeshArray(shMesh* pMesh, float* p, float* uv, unsigned short* idx,
					  int vsize, int idxsize)
{
	// サイズ
	pMesh->vertex_size = vsize;
	pMesh->face_size = idxsize;

	// メモリを確保
	pMesh->index = (unsigned short*)malloc(pMesh->vertex_size*sizeof(unsigned short));
	pMesh->vartex_p = (shVector3*)malloc(pMesh->vertex_size*sizeof(shVector3));
	pMesh->vartex_uv = (shVector2*)malloc(pMesh->vertex_size*sizeof(shVector2));

	//データを入れる
	memcpy(pMesh->vartex_p, p, sizeof(shVector3) * pMesh->vertex_size);
	memcpy(pMesh->vartex_uv, uv, sizeof(shVector2) * pMesh->vertex_size);
	memcpy(pMesh->index, idx, sizeof(unsigned short) * pMesh->face_size);

	return 0;
}

/*
 *	int shLoadMeshCOLLADA(shMesh* mesh, const char* daedoc)	
 *
 *	＜引数＞
 *　　shMesh* pMesh	: メッシュデータポインタ
 *　　char*			: daeドキュメントパス
 *	＜戻り値＞
 *　　int			: エラーコードが返ります
 *	＜説明＞
 *　　COLLADAフォーマットからデータを読み込みます。
 *　　※形式によっては対応していない場合があります。
 */
int shLoadMeshCOLLADA(shMesh* pMesh, const char* daedoc)
{
	// COLLADA解析用
	shColladaInfo colladaInfo;
	xmlTextReaderPtr reader;

	// 初期化
	memset(&colladaInfo,0,sizeof(shColladaInfo));
	// XMLファイルの読み込み
	reader = xmlNewTextReaderFilename(daedoc);
	if(reader == NULL)
		return 1;

	while (xmlTextReaderRead(reader) == 1) {
		int Type = xmlTextReaderNodeType(reader);
		if (Type == XML_READER_TYPE_ELEMENT) {
			// 要素の開始
			const xmlChar* name = xmlTextReaderConstName(reader);
			int procNo = COLLADA_NameDivide((const char*)name);
			if(procNo >= 0) {	// 検出されているかどうか
				// 関数を呼び出し処理をする
				if(g_sh_COLLADAProcessFunc[procNo](reader, &colladaInfo)) {
					/*エラー処理*/
				}
				// フラグスイッチング
				COLLADA_NameFlagSwitch(procNo,&colladaInfo);
			}
			if (xmlTextReaderIsEmptyElement(reader))		// 空の要素タグの場合
				COLLADA_NameFlagSwitch(procNo,&colladaInfo);	// フラグスイッチング

			//処理中のタグとして保存
			strcpy(colladaInfo.tagName,name);
		} else if (Type == XML_READER_TYPE_END_ELEMENT) {
			// 要素の終了
			const xmlChar* name = xmlTextReaderConstName(reader);
			int procNo = COLLADA_NameDivide((const char*)name);
			if(procNo >= 0) {	// 検出されているかどうか
				// フラグスイッチング
				COLLADA_NameFlagSwitch(procNo,&colladaInfo);
			}
			//処理中のタグとして保存
			strcpy(colladaInfo.tagName,name);
		} else if (Type == XML_READER_TYPE_TEXT) {
			// 文字列
			int procNo = COLLADA_NameDivide((const char*)colladaInfo.tagName);
			if(procNo >= 0) {	// 検出されているかどうか
				// 関数を呼び出し処理をする
				if(g_sh_COLLADAProcessTextFunc[procNo] != NULL) {
					if(g_sh_COLLADAProcessTextFunc[procNo](reader, &colladaInfo)) {
						/*エラー処理*/
					}
				}
			}
		}
	}
	//解放
	xmlFreeTextReader(reader);

	// COLLADAデータをもとにメッシュデータを生成
	if(COLLADAToSHMeshData(&colladaInfo, pMesh))
		return 1;

	// COLLADAデータのメモリを解放する
	COLLADAFreeInfo(&colladaInfo);

	return 0;
}

/*
 *	int shDrawMesh(shMesh* mesh)
 *
 *	＜引数＞
 *　　shMesh* pMesh	: メッシュデータポインタ
 *	＜戻り値＞
 *　　int				: エラーコードが返ります
 *	＜説明＞
 *　　メッシュデータを描画します。
 */
int shDrawMesh(shMesh* pMesh)
{
	if(g_sh_DrawMode == OpenGL) {
		// 頂点属性に情報を入れる
		glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, (float*)pMesh->vartex_p);
		glVertexAttribPointer(1, 2, GL_FLOAT, GL_FALSE, 0, (float*)pMesh->vartex_uv);
		glEnableVertexAttribArray(0);	// 頂点の性質を使用可能にする
		glEnableVertexAttribArray(1);

		// 描画コマンド発行
		glDrawElements(GL_TRIANGLES, pMesh->vertex_size, GL_UNSIGNED_SHORT, pMesh->index);

		glDisableVertexAttribArray(0);	// 頂点の性質を解除する
		glDisableVertexAttribArray(1);
	} else {
		/*使用しない*/
	}

	return 0;
}

/*
 *	int shCreateMesh(shMesh* pMesh, shVector3* vp, shVector2* vuv,
 *				 unsigned short* idx, int vsize, int fsize);
 *
 *	＜引数＞
 *　　shMesh* pMesh			: メッシュデータポインタ
 *　　shVector3* vp			: 頂点情報
 *　　shVector2* vuv		: ＵＶ座標
 *　　unsigned short* idx	: 結線
 *　　int vsize				: 頂点数
 *　　int fsize				: ポリゴン数
 *	＜戻り値＞
 *　　int					: エラーコードが返ります
 *	＜説明＞
 *　　メッシュデータを構成データから直接作る。
 *　　pMeshは
 */
int shCreateMesh(shMesh* pMesh, shVector3* vp, shVector2* vuv,
				 unsigned short* idx, int vsize, int fsize)
{
	return shCreateMeshArray(pMesh,&vp->v[0],&vuv->v[0],idx,vsize,fsize);
}


/*
 *	int shDeleteMesh(shMesh* mesh)
 *
 *	＜引数＞
 *　　shMesh* pMesh		: メッシュデータポインタ
 *	＜戻り値＞
 *　　int				: エラーコードが返ります
 *	＜説明＞
 *　　メッシュデータを解放し削除する。
 */
int shDeleteMesh(shMesh* pMesh)
{
	// メモリ解放
	free(pMesh->index);
	free(pMesh->vartex_p);
	free(pMesh->vartex_uv);
	// メモリ内をクリア
	memset(pMesh,0x00,sizeof(shMesh));

	return 0;
}
