﻿/*
 * マルチプラットフォーム描画エンジン「Sherry」
 * Copyright(C) 2010-2011 SherryProject. all rights reserved.
 *
 * The MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/*
 *	shMath.c
 *		Copyright (C) 2010-2011 Cap5ule. all rights reserved.
 *
 *	Date		|Version	|Author		|Summary
 *　2010/11/13	|v0.01		|onakah		|初回版
 *　2010/11/21	|v0.02		|onakah		|構造体定義の追加、関数群の宣言を追加
 *　2010/11/24	|v0.03		|onakah		|行列関連の関数を追加、定数マクロを追加
 *　			|			|			|2次元、4次元関連の関数追加
 *　2010/11/26	|v0.04		|onakah		|射影変換行列、ビュー行列作成関数を
 *				|			|			|左手右手系それぞれ追加。
 *　2010/11/26	|v0.05		|Cap5ule	|平行投影変換行列を追加。
 */

/*---- インクルードファイル ----*/

#include "sherry.h"		//!<共通ヘッダー
#include "shApplication.h"	//!<必要なSHヘッダー
#include "shMath.h"		//!<ヘッダー

/*---- マクロ定義 ----*/


/*---- スタティックプロトタイプ宣言 ----*/


/*---- グローバル変数 ----*/


/*---- スタティック関数 ----*/


/*---- 関数 ----*/


/*
 *	Vector2
 */

/*
 *	shVector2 shVec3Multiply(shVector2* pV1, shVector2* pV2)
 *
 *	＜引数＞
 *　　shVector2* pV1	: ベクトル
 *　　shVector2* pV2	: ベクトル
 *	＜戻り値＞
 *　　shVector2			: 計算後の結果
 *	＜説明＞
 *　　２次元ベクトルの積を求める
 *　　引数同士の掛け算を行い結果を返します。
 */
shVector2 shVec2Multiply(shVector2* pV1, shVector2* pV2)
{
	shVector2 vOut;

	vOut.x = pV1->x * pV2->x;
	vOut.y = pV1->y * pV2->y;

	return vOut;
}

/*
 *	shVector2 shVec2Division(shVector2* pV1, shVector2* pV2)
 *
 *	＜引数＞
 *　　shVector2* pV1	: ベクトル
 *　　shVector2* pV2	: ベクトル
 *	＜戻り値＞
 *　　shVector2			: 計算後の結果
 *	＜説明＞
 *　　２次元ベクトルの商を求める
 *　　pV1からpV2を割り、結果を返します。
 */
shVector2 shVec2Division(shVector2* pV1, shVector2* pV2)
{
	shVector2 vOut;

	vOut.x = pV1->x / pV2->x;
	vOut.y = pV1->y / pV2->y;

	return vOut;
}

/*
 *	shVector2 shVec2Add(shVector2* pV1, shVector2* pV2)
 *
 *	＜引数＞
 *　　shVector2* pV1	: ベクトル
 *　　shVector2* pV2	: ベクトル
 *	＜戻り値＞
 *　　shVector2		: 計算後の結果
 *	＜説明＞
 *　　２次元ベクトルの和を求める
 *　　引数同士の足し算を行い結果を返します。
 */
shVector2 shVec2Add(shVector2* pV1, shVector2* pV2)
{
	shVector2 vOut;

	vOut.x = pV1->x + pV2->x;
	vOut.y = pV1->y + pV2->y;

	return vOut;
}

/*
 *	shVector2 shVec2Sub(shVector2* pV1, shVector2* pV2)
 *
 *	＜引数＞
 *　　shVector2* pV1	: ベクトル
 *　　shVector2* pV2	: ベクトル
 *	＜戻り値＞
 *　　shVector2			: 計算後の結果
 *	＜説明＞
 *　　２次元ベクトルの差を求める
 *　　pV1からpV2を引いた結果を返します。
 */
shVector2 shVec2Sub(shVector2* pV1, shVector2* pV2)
{
	shVector2 vOut;

	vOut.x = pV1->x - pV2->x;
	vOut.y = pV1->y - pV2->y;

	return vOut;
}

/*
 *	float shVec2Dot(shVector2* pV1, shVector2* pV2)
 *
 *	＜引数＞
 *　　shVector2* pV1	: ベクトル
 *　　shVector2* pV2	: ベクトル
 *	＜戻り値＞
 *　　shVector2			: 計算結果
 *	＜説明＞
 *　　２次元ベクトルの内積の計算結果を返します。
 */
float shVec2Dot(shVector2* pV1, shVector2* pV2)
{
	return ( (pV1->x)*(pV2->x) + (pV1->y)*(pV2->y) );
}

/*
 *	float shVec3Cross(shVector2* pV1, shVector2* pV2)
 *
 *	＜引数＞
 *　　shVector2* pV1	: ベクトル
 *　　shVector2* pV2	: ベクトル
 *	＜戻り値＞
 *　　float				: 計算結果
 *	＜説明＞
 *　　２次元ベクトルの外積の計算結果を返します。
 */
float shVec2Cross(shVector2* pV1, shVector2* pV2)
{
	return pV1->x * pV2->y - pV1->y * pV2->x;
}

/*
 *	float shVec2Length(shVector2* pV)
 *
 *	＜引数＞
 *　　shVector2* pV	: ベクトル
 *	＜戻り値＞
 *　　float			: 長さ
 *	＜説明＞
 *　　ベクトルの長さを求めます。
 */
float shVec2Length(shVector2* pV)
{
	if(!pV) return 0.0f;

	return sqrtf((pV->x)*(pV->x) + (pV->y)*(pV->y));
}

/*
 *	void shVec2Normalize(shVector2* pV)
 *
 *	＜引数＞
 *　　shVector2* pV	: ベクトル
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　２次元ベクトルを正規化します。
 */
void shVec2dNormalize(shVector2* pV)
{
	shVector2 vOut;
	float len = sqrtf((pV->x*pV->x) + (pV->y*pV->y));

	if(len > 0){
		len = 1.0f / len;
		vOut.x = pV->x * len;
		vOut.y = pV->y * len;
	}else{
		vOut.x = 0.0f;
		vOut.y = 0.0f;
	}
	*pV = vOut;
}

/*
 *	shVector2 shVec2Linear(shVector2* pV1, shVector2* pV2, float t)
 *
 *	＜引数＞
 *　　shVector2* pV1	: ベクトル
 *　　shVector2* pV2	: ベクトル
 *　　float t			: 割合（0.0～1.0）
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　２次元ベクトルの線形補間を行う
 *　　pV1からpV2間を割合tで線形補間し、結果を返します。
 */
shVector2 shVec2Linear(shVector2* pV1, shVector2* pV2, float t)
{
	shVector2 vOut;

	vOut.x = (pV1->x - pV2->x) / t;
	vOut.y = (pV1->y - pV2->y) / t;

	return vOut;
}


/*
 *	Vector3
 */

/*
 *	shVector3 shVec3Multiply(shVector3* pV1, shVector3* pV2)
 *
 *	＜引数＞
 *		shVector3* pV1	: ベクトル
 *		shVector3* pV2	: ベクトル
 *	＜戻り値＞
 *		shVector3		: 計算後の結果
 *	＜説明＞
 *　　３次元ベクトルの積を求める
 *　　引数同士の掛け算を行い結果を返します。
 */
shVector3 shVec3Multiply(shVector3* pV1, shVector3* pV2)
{
	shVector3 vOut;

	vOut.x = pV1->x * pV2->x;
	vOut.y = pV1->y * pV2->y;
	vOut.z = pV1->z * pV2->z;

	return vOut;
}

/*
 *	shVector3 shVec3Division(shVector3* pV1, shVector3* pV2)
 *
 *	＜引数＞
 *　　shVector3* pV1	: ベクトル
 *　　shVector3* pV2	: ベクトル
 *	＜戻り値＞
 *　　shVector3		　　: 計算後の結果
 *	＜説明＞
 *　　３次元ベクトルの商を求める
 *　　pV1からpV2を割り、結果を返します。
 */
shVector3 shVec3Division(shVector3* pV1, shVector3* pV2)
{
	shVector3 vOut;

	vOut.x = pV1->x / pV2->x;
	vOut.y = pV1->y / pV2->y;
	vOut.z = pV1->z / pV2->z;

	return vOut;
}

/*
 *	shVector3 shVec3Add(shVector3* pV1, shVector3* pV2)
 *
 *	＜引数＞
 *　　shVector3* pV1	: ベクトル
 *　　shVector3* pV2	: ベクトル
 *	＜戻り値＞
 *　　shVector3			: 計算後の結果
 *	＜説明＞
 *　　３次元ベクトルの和を求める
 *　　引数同士の足し算を行い結果を返します。
 */
shVector3 shVec3Add(shVector3* pV1, shVector3* pV2)
{
	shVector3 vOut;

	vOut.x = pV1->x + pV2->x;
	vOut.y = pV1->y + pV2->y;
	vOut.z = pV1->z + pV2->z;

	return vOut;
}

/*
 *	shVector3 shVec3Sub(shVector3* pV1, shVector3* pV2)
 *
 *	＜引数＞
 *　　shVector3* pV1	: ベクトル
 *　　shVector3* pV2	: ベクトル
 *	＜戻り値＞
 *　　shVector3		: 計算後の結果
 *	＜説明＞
 *　　３次元ベクトルの差を求める
 *　　pV1からpV2を引いた結果を返します。
 */
shVector3 shVec3Sub(shVector3* pV1, shVector3* pV2)
{
	shVector3 vOut;

	vOut.x = pV1->x - pV2->x;
	vOut.y = pV1->y - pV2->y;
	vOut.z = pV1->z - pV2->z;

	return vOut;
}

/*
 *	float shVec3Dot(shVector3* pV1, shVector3* pV2)
 *
 *	＜引数＞
 *		shVector3* pV1	: ベクトル
 *		shVector3* pV2	: ベクトル
 *	＜戻り値＞
 *		shVector3		: 計算結果
 *	＜説明＞
 *		３次元ベクトル同士の内積を返します。
 */
float shVec3Dot(shVector3* pV1, shVector3* pV2)
{
	return ( (pV1->x)*(pV2->x) + (pV1->y)*(pV2->y) + (pV1->z)*(pV2->z) );
}

/*
 *	shVector3 shVec3Cross(shVector3* pV1, shVector3* pV2)
 *
 *	＜引数＞
 *　　shVector3* pV1	: ベクトル
 *　　shVector3* pV2	: ベクトル
 *	＜戻り値＞
 *　　shVector3		: 計算結果
 *	＜説明＞
 *　　３次元ベクトル同士の外積を返します。
 */
shVector3 shVec3Cross(shVector3* pV1, shVector3* pV2)
{
	shVector3 vOut;

	vOut.x = pV1->y * pV2->z - pV1->z * pV2->y;
	vOut.y = pV1->z * pV2->x - pV1->x * pV2->z;
	vOut.z = pV1->x * pV2->y - pV1->y * pV2->x;

	return vOut;
}

/*
 *	float shVec3Length(shVector3* pV)
 *
 *	＜引数＞
 *		shVector3* pV	: ベクトル
 *	＜戻り値＞
 *		float			: 長さ
 *	＜説明＞
 *		３次元ベクトルの長さを求めます。
 */
float shVec3Length(shVector3* pV)
{
	if(!pV) return 0.0f;

	return sqrtf((pV->x)*(pV->x) + (pV->y)*(pV->y) + (pV->z)*(pV->z));
}

/*
 *	void shVec3Normalize(shVector3* pV)	
 *
 *	＜引数＞
 *		shVector3* pV	: 行列
 *	＜戻り値＞
 *		なし
 *	＜説明＞
 *		３次元ベクトルのベクトルを正規化します。
 */
void shVec3Normalize(shVector3* pV)
{
	shVector3 vOut;
	float len = sqrtf((pV->x*pV->x) + (pV->y*pV->y) + (pV->z*pV->z));

	if(len > 0){
		len = 1.0f / len;
		vOut.x = pV->x * len;
		vOut.y = pV->y * len;
		vOut.z = pV->z * len;
	}else{
		vOut.x = 0.0f;
		vOut.y = 0.0f;
		vOut.z = 0.0f;
	}
	*pV = vOut;
}

/*
 *	shVector3 shVec3Linear(shVector3* pV1, shVector3* pV2, float t)
 *
 *	＜引数＞
 *　　shVector3* pV1	: 行列
 *　　shVector3* pV2	: 行列
 *　　float t			: 割合（0.0～1.0）
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　３次元ベクトルの線形補間を行う
 *　　pV1からpV2間を割合tで線形補間し、結果を返します。
 */
shVector3 shVec3Linear(shVector3* pV1, shVector3* pV2, float t)
{
	shVector3 vOut;

	vOut.x = (pV1->x - pV2->x) / t;
	vOut.y = (pV1->y - pV2->y) / t;
	vOut.z = (pV1->z - pV2->z) / t;

	return vOut;
}


/*
 *	Vector4
 */

/*
 *	shVector4 shVec4Add(shVector4* pV1, shVector4* pV2)
 *
 *	＜引数＞
 *　　shVector4* pV1	: ベクトル
 *　　shVector4* pV2	: ベクトル
 *	＜戻り値＞
 *　　shVector4			: 計算結果
 *	＜説明＞
 *　　ベクトルの和
 *　　引数同士の和を求め、結果を返します。
 */
shVector4 shVec4Add(shVector4* pV1, shVector4* pV2)
{
	shVector4 vOut;

	vOut.x = pV1->x + pV2->x;
	vOut.y = pV1->y + pV2->y;
	vOut.z = pV1->z + pV2->z;
	vOut.w = pV1->w + pV2->w;

	return vOut;
}

/*
 *	shVector4 shVec4Sub(shVector4* pV1, shVector4* pV2)
 *
 *	＜引数＞
 *　　shVector4* pV1	: ベクトル
 *　　shVector4* pV2	: ベクトル
 *	＜戻り値＞
 *　　shVector4			: 計算結果
 *	＜説明＞
 *　　ベクトルの差
 *　　pV1からpV2を引いた結果を返します。
 */
shVector4 shVec4Sub(shVector4* pV1, shVector4* pV2)
{
	shVector4 vOut;

	vOut.x = pV1->x - pV2->x;
	vOut.y = pV1->y - pV2->y;
	vOut.z = pV1->z - pV2->z;
	vOut.w = pV1->w - pV2->w;

	return vOut;
}

/*
 *	void shVec4Normalize(shVector4* pV)
 *
 *	＜引数＞
 *　　shVector4* pV		: ベクトル
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　ベクトルの正規化を行います。
 */
void shVec4Normalize(shVector4* pV)
{
	shVector4 vOut;
	float len = sqrtf((pV->x*pV->x) + (pV->y*pV->y) + (pV->z*pV->z) + (pV->w*pV->w));

	if(len > 0){
		len = 1.0f / len;
		vOut.x = pV->x * len;
		vOut.y = pV->y * len;
		vOut.z = pV->z * len;
		vOut.w = pV->w * len;
	}else{
		vOut.x = 0.0f;
		vOut.y = 0.0f;
		vOut.z = 0.0f;
		vOut.w = 0.0f;
	}
	*pV = vOut;
}

/*
 *	float shVec4Dot(shVector4* pV1, shVector4* pV2)
 * 
 *	＜引数＞
 *		shVector4* pV1	: ベクトル
 *		shVector4* pV2	: ベクトル
 *	＜戻り値＞
 *		float			: 計算結果
 *	＜説明＞
 *		４次元ベクトルの内積を求めます。
 */
float shVec4Dot(shVector4* pV1, shVector4* pV2)
{
	return ( (pV1->x)*(pV2->x) + (pV1->y)*(pV2->y) + (pV1->z)*(pV2->z) + (pV1->w)*(pV2->w) );
}




/*
 *	Matrix
 */

/*
 *	void shMatrixIdentity(shMatrix* pM)
 *
 *	＜引数＞
 *		shMatrix* pM	: 行列
 *	＜戻り値＞
 *		なし
 *	＜説明＞
 *		行列を単位行列化します。
 */
void shMatrixIdentity(shMatrix* pOut)
{
	pOut->m[0][1] = pOut->m[0][2] = pOut->m[0][3] =
	pOut->m[1][0] = pOut->m[1][2] = pOut->m[1][3] =
	pOut->m[2][0] = pOut->m[2][1] = pOut->m[2][3] =
	pOut->m[3][0] = pOut->m[3][1] = pOut->m[3][2] = 0.0f;

	pOut->m[0][0] = pOut->m[1][1] = pOut->m[2][2] = pOut->m[3][3] = 1.0f;
}

/*
 *	shMatrix* shMatrixMultiply(shMatrix* pOut, shMatrix* pM1, shMatrix* pM2)
 *
 *	＜引数＞
 *　　shMatrix* pOut	: 計算結果
 *　　shMatrix* pM1		: 行列
 *　　shMatrix* pM2		: 行列
 *	＜戻り値＞
 *　　shMatrix*			: 計算結果
 *	＜説明＞
 *　　行列同士の乗算	
 *　　引数同士の積を返します。
 */
shMatrix* shMatrixMultiply(shMatrix* pOut, shMatrix* pM1, shMatrix* pM2)
{
	int i, j, k;
	shMatrix mat;
	memset(&mat,0,sizeof(shMatrix));

	for(i = 0; i < 4; i ++){
		for(j = 0; j < 4; j ++){
			for(k = 0; k < 4; k ++){
				mat.m[i][j] += pM1->m[i][k] * pM2->m[k][j];
			}
		}
	}
	*pOut = mat;

	return pOut;
}

/*
 *	shVector4* shVec3Transform(shVector4* pOut, shVector3* pV, shMatrix* pM)
 *
 *	＜引数＞
 *		shVector4* pOut	: 計算結果
 *		shVector3* pV	: ベクトル
 *		shMatrix* pM	: 行列
 *	＜戻り値＞
 *		shVector4*		: 計算結果
 *	＜説明＞
 *		３次元ベクトルと行列の乗算を行い、４次元ベクトルを返します。
 */
shVector4* shVec3Transform(shVector4* pOut, shVector3* pV, shMatrix* pM)
{
	float vx, vy, vz;
	shVector4 vec;
	float *pF1, *pF2, *pF3, *pF4;

	vx = pV->x;
	vy = pV->y;
	vz = pV->z;

	pF1 = pM->m[0];
	pF2 = pM->m[1];
	pF3 = pM->m[2];
	pF4 = pM->m[3];

	vec.x = vx * (*pF1) + vy * (*pF2) + vz * (*pF3) + (*pF4);
	pF1++; pF2++; pF3++; pF4++;

	vec.y = vx * (*pF1) + vy * (*pF2) + vz * (*pF3) + (*pF4);
	pF1++; pF2++; pF3++; pF4++;

	vec.z = vx * (*pF1) + vy * (*pF2) + vz * (*pF3) + (*pF4);
	pF1++; pF2++; pF3++; pF4++;

	vec.w = vx * (*pF1) + vy * (*pF2) + vz * (*pF3) + (*pF4);

	*pOut = vec;

	return pOut;
}

/*
 *	shVector4* shVec4Transform(shVector4* pOut, shVector4* pV, shMatrix* pM)
 *
 *	＜引数＞
 *		shVector4* pOut	: 計算結果
 *		shVector4* pV	: ベクトル
 *		shMatrix* pM	: 行列
 *	＜戻り値＞
 *		shVector4*		: 計算結果
 *	＜説明＞
 *		４次元ベクトルと行列の乗算を行い、４次元ベクトルを返します。
 */
shVector4* shVec4Transform(shVector4* pOut, shVector4* pV, shMatrix* pM)
{
	float vx, vy, vz, vw;
	shVector4 vec;
	float *pF1, *pF2, *pF3, *pF4;

	vx = pV->x;
	vy = pV->y;
	vz = pV->z;
	vw = pV->w;

	pF1 = pM->m[0];
	pF2 = pM->m[1];
	pF3 = pM->m[2];
	pF4 = pM->m[3];
	vec.x = vx * (*pF1) + vy * (*pF2) + vz * (*pF3) + vw * (*pF4);
	pF1++; pF2++; pF3++; pF4++;

	vec.y = vx * (*pF1) + vy * (*pF2) + vz * (*pF3) + vw * (*pF4);
	pF1++; pF2++; pF3++; pF4++;

	vec.z = vx * (*pF1) + vy * (*pF2) + vz * (*pF3) + vw * (*pF4);
	pF1++; pF2++; pF3++; pF4++;

	vec.w = vx * (*pF1) + vy * (*pF2) + vz * (*pF3) + vw * (*pF4);

	*pOut = vec;

	return pOut;
}

/*
 *	shMatrix shMatrixScaling(shMatrix* pOut, float fScaleX,	
 *							 float fScaleY, float fScaleZ)
 *
 *	＜引数＞
 *		shMatrix* pOut	: 行列
 *		float fScaleX	: X軸方向拡大率
 *		float fScaleY	: Y軸方向拡大率
 *		float fScaleZ	: Z軸方向拡大率
 *	＜戻り値＞
 *		shMatrix*		: 結果行列
 *	＜説明＞
 *		拡大縮小を行う行列を作成します。
 */
shMatrix* shMatrixScaling(shMatrix* pOut, float fScaleX,
						  float fScaleY, float fScaleZ)
{
	pOut->m[0][0] = fScaleX;
	pOut->m[0][1] = 0.0f;
	pOut->m[0][2] = 0.0f;
	pOut->m[0][3] = 0.0f;
	
	pOut->m[1][0] = 0.0f;
	pOut->m[1][1] = fScaleY;
	pOut->m[1][2] = 0.0f;
	pOut->m[1][3] = 0.0f;
	
	pOut->m[2][0] = 0.0f;
	pOut->m[2][1] = 0.0f;
	pOut->m[2][2] = fScaleZ;
	pOut->m[2][3] = 0.0f;
	
	pOut->m[3][0] = 0.0f;
	pOut->m[3][1] = 0.0f;
	pOut->m[3][2] = 0.0f;
	pOut->m[3][3] = 1.0f;

	return pOut;
}

/*
 *	shMatrix* shMatrixRotationX(shMatrix* pOut, float Degree)
 *
 *	＜引数＞
 *		shMatrix* pOut	: 行列
 *		float Degree	: 回転角度
 *	＜戻り値＞
 *		shMatrix*		: 結果行列
 *	＜説明＞
 *		X軸を中心に左ねじ方向に、Degree度回転するための行列を作成します。
 */
shMatrix* shMatrixRotationX(shMatrix* pOut, float Degree)
{
	pOut->m[0][0] = 1.0f;
	pOut->m[0][1] = 0.0f;
	pOut->m[0][2] = 0.0f;
	pOut->m[0][3] = 0.0f;

	pOut->m[1][0] = 0.0f;
	pOut->m[1][1] = (float)cos(TO_RADIAN(Degree));
	pOut->m[1][2] = (float)sin(TO_RADIAN(Degree));
	pOut->m[1][3] = 0.0f;

	pOut->m[2][0] = 0.0f;
	pOut->m[2][1] = -(float)sin(TO_RADIAN(Degree));
	pOut->m[2][2] = (float)cos(TO_RADIAN(Degree));
	pOut->m[2][3] = 0.0f;
	
	pOut->m[3][0] = 0.0f;
	pOut->m[3][1] = 0.0f;
	pOut->m[3][2] = 0.0f;
	pOut->m[3][3] = 1.0f;

	return pOut;
}

/*
 *	shMatrix* shMatrixRotationY(shMatrix* pOut, float Degree)
 *
 *	＜引数＞
 *		shMatrix* pOut	: 行列
 *		float Degree	: 回転角度
 *	＜戻り値＞
 *		shMatrix*		: 結果行列
 *	＜説明＞
 *		Y軸を中心に左ねじ方向に、Degree度回転するための行列を作成します。
 */
shMatrix* shMatrixRotationY(shMatrix* pOut, float Degree)
{
	pOut->m[0][0] = (float)cos(TO_RADIAN(Degree));
	pOut->m[0][1] = 0.0f;
	pOut->m[0][2] = -(float)sin(TO_RADIAN(Degree));
	pOut->m[0][3] = 0.0f;

	pOut->m[1][0] = 0.0f;
	pOut->m[1][1] = 1.0f;
	pOut->m[1][2] = 0.0f;
	pOut->m[1][3] = 0.0f;

	pOut->m[2][0] = (float)sin(TO_RADIAN(Degree));
	pOut->m[2][1] = 0.0f;
	pOut->m[2][2] = (float)cos(TO_RADIAN(Degree));
	pOut->m[2][3] = 0.0f;
	
	pOut->m[3][0] = 0.0f;
	pOut->m[3][1] = 0.0f;
	pOut->m[3][2] = 0.0f;
	pOut->m[3][3] = 1.0f;

	return pOut;
}

/*
 *	shMatrix* shMatrixRotationZ(shMatrix* pOut, float Degree)
 *
 *	＜引数＞
 *　　shMatrix* pOut	: 行列
 *　　float Degree	: 回転角度
 *	＜戻り値＞
 *　　shMatrix*		: 結果行列
 *	＜説明＞
 *　　Z軸を中心に左ねじ方向に、Degree度回転するための行列を作成します。
 */
shMatrix* shMatrixRotationZ(shMatrix* pOut, float Degree)
{
	pOut->m[0][0] = (float)cos(TO_RADIAN(Degree));
	pOut->m[0][1] = (float)sin(TO_RADIAN(Degree));
	pOut->m[0][2] = 0.0f;
	pOut->m[0][3] = 0.0f;

	pOut->m[1][0] = -(float)sin(TO_RADIAN(Degree));
	pOut->m[1][1] = (float)cos(TO_RADIAN(Degree));
	pOut->m[1][2] = 0.0f;
	pOut->m[1][3] = 0.0f;

	pOut->m[2][0] = 0.0f;
	pOut->m[2][1] = 0.0f;
	pOut->m[2][2] = 1.0f;
	pOut->m[2][3] = 0.0f;
	
	pOut->m[3][0] = 0.0f;
	pOut->m[3][1] = 0.0f;
	pOut->m[3][2] = 0.0f;
	pOut->m[3][3] = 1.0f;

	return pOut;
}

/*
 *	shMatrix* shMatrixRotationYawPitchRoll(shMatrix* pOut, float Yaw,
 *										   float Pitch, float Roll)
 *
 *	＜引数＞
 *　　shMatrix* pOut	: 行列
 *　　float Yaw			: ヨー（Degree）
 *　　float Pitch		: ピッチ（Degree）
 *　　float Roll		: ロー（Degree）
 *	＜戻り値＞
 *　　shMatrix*			: 結果行列
 *	＜説明＞
 *　　Y軸、X軸、Z軸の順に左ねじ方向に、それぞれのヨーピッチロー角度（Degree）
 *　　回転するための行列を作成します。
 */
shMatrix* shMatrixRotationYawPitchRoll(shMatrix* pOut, float Yaw,
									   float Pitch, float Roll)
{
	const float sin_y = (float)sin(TO_RADIAN(Yaw));
	const float cos_y = (float)cos(TO_RADIAN(Yaw));
	const float sin_p = (float)sin(TO_RADIAN(Pitch));
	const float cos_p = (float)cos(TO_RADIAN(Pitch));
	const float sin_r = (float)sin(TO_RADIAN(Roll));
	const float cos_r = (float)cos(TO_RADIAN(Roll));

	pOut->m[0][0] = cos_y * cos_r + sin_y * sin_p * sin_r;
	pOut->m[0][1] = sin_r * cos_p;
	pOut->m[0][2] = cos_r * -sin_y + sin_r * sin_p * cos_y;
	pOut->m[0][3] = 0.0f;

	pOut->m[1][0] = -sin_r * cos_y + cos_r * sin_p * sin_y;
	pOut->m[1][1] = cos_r * cos_p;
	pOut->m[1][2] = sin_r * sin_y + cos_r * sin_p * cos_y;
	pOut->m[1][3] = 0.0f;

	pOut->m[2][0] = cos_p * sin_y;
	pOut->m[2][1] = -sin_p;
	pOut->m[2][2] = cos_p * cos_y;
	pOut->m[2][3] = 0.0f;

	pOut->m[3][0] = 0.0f;
	pOut->m[3][1] = 0.0f;
	pOut->m[3][2] = 0.0f;
	pOut->m[3][3] = 1.0f;

	return pOut;
}

/*
 *	shMatrix* shMatrixRotationEuler(shMatrix* pOut, float x, float y, float z)
 *
 *	＜引数＞
 *　　shMatrix* pOut	: 行列
 *　　float x			: X回転角度（Degree）
 *　　float y			: Y回転角度（Degree）
 *　　float z			: Z回転角度（Degree）
 *	＜戻り値＞
 *　　shMatrix*		: 結果
 *	＜説明＞
 *　　Y軸、X軸、Z軸の順に左ねじ方向に、それぞれの角度（Degree）
 *　　回転するための行列を作成します。
 */
shMatrix* shMatrixRotationEuler(shMatrix* pOut, float x, float y, float z)
{
	pOut->m[0][0] = (float)(cos(TO_RADIAN(x)) * cos(TO_RADIAN(y)) * cos(TO_RADIAN(z)) -
					sin(TO_RADIAN(x)) * sin(TO_RADIAN(z)));
	pOut->m[0][1] = (float)(sin(TO_RADIAN(x)) * cos(TO_RADIAN(y)) * cos(TO_RADIAN(z)) -
					cos(TO_RADIAN(x)) * sin(TO_RADIAN(z)));
	pOut->m[0][2] = (float)(-sin(TO_RADIAN(y)) * cos(TO_RADIAN(z)));
	pOut->m[0][3] = 0.0f;

	pOut->m[1][0] = (float)(-cos(TO_RADIAN(x)) * cos(TO_RADIAN(y)) * sin(TO_RADIAN(z)) -
					sin(TO_RADIAN(x)) * cos(TO_RADIAN(z)));
	pOut->m[1][1] = (float)(-sin(TO_RADIAN(x)) * cos(TO_RADIAN(y)) * sin(TO_RADIAN(z)) -
					cos(TO_RADIAN(x)) * cos(TO_RADIAN(z)));
	pOut->m[1][2] = (float)(sin(TO_RADIAN(y)) * sin(TO_RADIAN(z)));
	pOut->m[1][3] = 0.0f;

	pOut->m[2][0] = (float)(cos(TO_RADIAN(x)) * sin(TO_RADIAN(y)));
	pOut->m[2][1] = (float)(sin(TO_RADIAN(x)) * sin(TO_RADIAN(y)));
	pOut->m[2][2] = (float)cos(TO_RADIAN(y));
	pOut->m[2][3] = 0.0f;
	
	pOut->m[3][0] = 0.0f;
	pOut->m[3][1] = 0.0f;
	pOut->m[3][2] = 0.0f;
	pOut->m[3][3] = 1.0f;

	return pOut;
}

/*
 *	shMatrix* shMatrixTranslation(shMatrix* pOut, float fTransX,
 *								  float fTransY, float fTransZ)
 *
 *	＜引数＞
 *		shMatrix* pOut	: 行列
 *		float fTransX	: X移動量
 *		float fTransY	: Y移動量
 *		float fTransZ	: Z移動量
 *	＜戻り値＞
 *		shMatrix*		: 結果行列
 *	＜説明＞
 *		x, y, zだけ平行移動するための行列を作成します。
 */
shMatrix* shMatrixTranslation(shMatrix* pOut, float fTransX,
							  float fTransY, float fTransZ)
{
	pOut->m[0][0] = 1.0f;
	pOut->m[0][1] = 0.0f;
	pOut->m[0][2] = 0.0f;
	pOut->m[0][3] = 0.0f;

	pOut->m[1][0] = 0.0f;
	pOut->m[1][1] = 1.0f;
	pOut->m[1][2] = 0.0f;
	pOut->m[1][3] = 0.0f;

	pOut->m[2][0] = 0.0f;
	pOut->m[2][1] = 0.0f;
	pOut->m[2][2] = 1.0f;
	pOut->m[2][3] = 0.0f;
	
	pOut->m[3][0] = fTransX;
	pOut->m[3][1] = fTransY;
	pOut->m[3][2] = fTransZ;
	pOut->m[3][3] = 1.0f;

	return pOut;
}

/*
 *	shMatrix* shMatrixInverse(shMatrix* pOut, shMatrix* pIn)
 *
 *	＜引数＞
 *		shMatrix* pOut	: 行列
 *		shMatrix* pIn	: 行列
 *	＜戻り値＞
 *		shMatrix*		: 結果
 *	＜説明＞
 *		逆行列を求めます。
 */
shMatrix* shMatrixInverse(shMatrix* pOut, shMatrix* pIn)
{
	shMatrix mat;
	int		i, j, loop;
	double	fDat, fDat2;
	double	mat_8x4[4][8];
	int		flag;
	float	*pF;
	double	*pD;

	//8 x 4行列に値を入れる
	for(i = 0; i < 4; i++) {
		pF = pIn->m[i];
		for(j = 0; j < 4; j++, pF++) mat_8x4[i][j] = (double)(*pF);
		pD  = mat_8x4[i] + 4;
		for(j = 0; j < 4; j++){
			if(i == j)	*pD = 1.0;
			else		*pD = 0.0;
			pD++;
		}
	}

	flag = 1;
	for(loop = 0; loop < 4; loop++){
		fDat = mat_8x4[loop][loop];
		if(fDat != 1.0) {
			if(fDat == 0.0) {
				for(i = loop + 1; i < 4; i++){
					fDat = mat_8x4[i][loop];
					if(fDat != 0.0) break;
				}
				if(i >= 4){
					flag = 0;
					break;
				}
				//行を入れ替える
				for(j = 0; j < 8; j++){
					fDat = mat_8x4[i][j];
					mat_8x4[i][j] = mat_8x4[loop][j];
					mat_8x4[loop][j] = fDat;
				}
				fDat = mat_8x4[loop][loop];
			}

			for(i = 0; i < 8; i++) mat_8x4[loop][i] /= fDat;
		}
		for(i = 0; i < 4; i++){
			if(i != loop) {
				fDat = mat_8x4[i][loop];
				if(fDat != 0.0f) {
					//mat[i][loop]をmat[loop]の行にかけて
					//(mat[j] - mat[loop] * fDat)を計算
					for(j = 0; j < 8; j++){
						fDat2 = mat_8x4[loop][j] * fDat;
						mat_8x4[i][j] -= fDat2;
					}
				}
			}
		}
	}

	if(flag){
		for(i = 0; i < 4; i++){
			pF = mat.m[i];
			pD = mat_8x4[i] + 4;
			for(j = 0; j < 4; j++){
				*pF = (float)(*pD);
				pF++;
				pD++;
			}
		}
	}else{
		//単位行列を求める
		shMatrixIdentity(&mat);
	}

	if(!flag){
		return NULL;
	}

	*pOut = mat;
	return pOut;
}

/*
 *	shMatrix* shLookAtMatrixLH(shMatrix* pOut, shVector3* pEye,
 *						   shVector3* pAt, shVector3* pUp)
 *
 *	＜引数＞
 *		shMatrix* pOut	: 結果行列を表すポインタ
 *		shVector3* pEye	: 視点
 *		shVector3* pAt	: カメラの注視点
 *		shVector3* pUp	: 上方向のベクトル
 *	＜戻り値＞
 *		shMatrix*		: 結果行列を表すポインタ
 *	＜説明＞
 *		ビュー行列の作成
 *		左手座標系ビュー行列を作成します。
 */
shMatrix* shLookAtMatrixLH(shMatrix* pOut, shVector3* pEye,
						   shVector3* pAt, shVector3* pUp)
{
	/*
	 	xaxis.x           yaxis.x           zaxis.x				 0
	 	xaxis.y           yaxis.y           zaxis.y				 0
	 	xaxis.z           yaxis.z           zaxis.z				 0
	 	-dot(xaxis, eye)  -dot(yaxis, eye)  -dot(zaxis, eye)	 1
	 */
	shVector3 zAxis;
	shVector3 xAxis;
	shVector3 yAxis;

	zAxis = shVec3Sub(pAt, pEye);
	shVec3Normalize(&zAxis);

	xAxis = shVec3Cross(pUp, &zAxis);
	shVec3Normalize(&xAxis);

	yAxis = shVec3Cross(&zAxis, &xAxis);

	pOut->_11 = xAxis.x;
	pOut->_12 = yAxis.x;
	pOut->_13 = zAxis.x;
	pOut->_14 = 0.0f;

	pOut->_21 = xAxis.y;
	pOut->_22 = yAxis.y;
	pOut->_23 = zAxis.y;
	pOut->_24 = 0.0f;

	pOut->_31 = xAxis.z;
	pOut->_32 = yAxis.z;
	pOut->_33 = zAxis.z;
	pOut->_34 = 0.0f;

	pOut->_41 = -shVec3Dot(&xAxis, pEye);
	pOut->_42 = -shVec3Dot(&yAxis, pEye);
	pOut->_43 = -shVec3Dot(&zAxis, pEye);
	pOut->_44 = 1.0f;

	return pOut;
}

/*
 *	shMatrix* shLookAtMatrixRH(shMatrix* pOut, shVector3* pEye,
 *						   shVector3* pAt, shVector3* pUp)
 *
 *	＜引数＞
 *　　shMatrix* pOut	: 結果行列を表すポインタ
 *　　shVector3* pEye	: 視点
 *　　shVector3* pAt	: カメラの注視点
 *　　shVector3* pUp	: 上方向のベクトル
 *	＜戻り値＞
 *　　shMatrix*			: 結果行列を表すポインタ
 *	＜説明＞
 *　　ビュー行列の作成
 *　　右手座標系ビュー行列を作成します。
 */
shMatrix* shLookAtMatrixRH(shMatrix* pOut, shVector3* pEye,
						   shVector3* pAt, shVector3* pUp)
{
	shVector3 zAxis;
	shVector3 xAxis;
	shVector3 yAxis;

	zAxis = shVec3Sub(pEye, pAt);
	shVec3Normalize(&zAxis);

	xAxis = shVec3Cross(pUp, &zAxis);
	shVec3Normalize(&xAxis);

	yAxis = shVec3Cross(&zAxis, &xAxis);

	pOut->_11 = xAxis.x;
	pOut->_12 = yAxis.x;
	pOut->_13 = zAxis.x;
	pOut->_14 = 0.0f;

	pOut->_21 = xAxis.y;
	pOut->_22 = yAxis.y;
	pOut->_23 = zAxis.y;
	pOut->_24 = 0.0f;

	pOut->_31 = xAxis.z;
	pOut->_32 = yAxis.z;
	pOut->_33 = zAxis.z;
	pOut->_34 = 0.0f;

	pOut->_41 = -shVec3Dot(&xAxis, pEye);
	pOut->_42 = -shVec3Dot(&yAxis, pEye);
	pOut->_43 = -shVec3Dot(&zAxis, pEye);
	pOut->_44 = 1.0f;

	return pOut;
}

/*
 *	shMatrix* shMatrixPerspectiveFovLH(shMatrix* pOut, float fovY,
 *								   float Aspect, float zn, float zf)
 *
 *	＜引数＞
 *　　shMatrix* pOut	: 結果行列を表すポインタ
 *　　float fovY		: Y方向の視野角（ラジアン）
 *　　float Aspect		: アスペクト比
 *　　float zn			: 近くのビュープレーンのZ値	
 *　　float zf			: 遠くのビュープレーンのZ値
 *	＜戻り値＞
 *　　shMatrix*			: 結果行列を表すポインタ
 *	＜説明＞
 *　　パースペクティブ射影行列の作成
 *　　視野に基づいて、左手座標系のパースペクティブ射影行列を作成します。
 */
shMatrix* shMatrixPerspectiveFovLH(shMatrix* pOut, float fovY,
								   float Aspect, float zn, float zf)
{
	/*
		w		0		0				0
		0		h		0				0
		0		0		zf/(zf-zn)		1
		0		0		-zn*zf/(zf-zn)	0
		where:
		h is the view space height. It is calculated from
		h = cot(fovY/2);

		w is the view space width. It is calculated from
		w = h / Aspect.
	*/
	float f = (float)(1.0f / tan(fovY/2.0f));

	pOut->_11 = f / Aspect;
	pOut->_12 = 0.0f;
	pOut->_13 = 0.0f;
	pOut->_14 = 0.0f;

	pOut->_21 = 0.0f;
	pOut->_22 = f;
	pOut->_23 = 0.0f;
	pOut->_24 = 0.0f;

	pOut->_31 = 0.0f;
	pOut->_32 = 0.0f;
	pOut->_33 = zf / (zf - zn);
	pOut->_34 = 1.0f;

	pOut->_41 = 0.0f;
	pOut->_42 = 0.0f;
	pOut->_43 = -zn * zf / (zf - zn);
	pOut->_44 = 0.0f;

	return pOut;
}

/*
 *	shMatrix* shMatrixPerspectiveFovRH(shMatrix* pOut, float fovY,
 *									   float Aspect, float zn, float zf)
 *
 *	＜引数＞
 *　　shMatrix* pOut	: 結果行列を表すポインタ
 *　　float fovY		: Y方向の視野角（ラジアン）
 *　　float Aspect	: アスペクト比
 *　　float zn		: 近くのビュープレーンのZ値
 *　　float zf		: 遠くのビュープレーンのZ値
 *	＜戻り値＞
 *　　shMatrix*		: 結果行列を表すポインタ
 *	＜説明＞
 *　　パースペクティブ射影行列の作成
 *　　視野に基づいて、右手座標系のパースペクティブ射影行列を作成します。
 */
shMatrix* shMatrixPerspectiveFovRH(shMatrix* pOut, float fovY, float Aspect,
								   float zn, float zf)
{
	/*
		w		0		0				0
		0		h		0				0
		0		0		zf/(zn-zf)		-1
		0		0		zn*zf/(zn-zf)	0
		where:
		h is the view space height. It is calculated from
		h = cot(fovY/2);

		w is the view space width. It is calculated from
		w = h / Aspect.
	*/
	float f = (float)(1.0f / tan(fovY/2.0f));

	pOut->_11 = f / Aspect;
	pOut->_12 = 0.0f;
	pOut->_13 = 0.0f;
	pOut->_14 = 0.0f;

	pOut->_21 = 0.0f;
	pOut->_22 = f;
	pOut->_23 = 0.0f;
	pOut->_24 = 0.0f;

	pOut->_31 = 0.0f;
	pOut->_32 = 0.0f;
	pOut->_33 = zf / (zn - zf);
	pOut->_34 = -1.0f;

	pOut->_41 = 0.0f;
	pOut->_42 = 0.0f;
	pOut->_43 = zn * zf / (zn - zf);
	pOut->_44 = 0.0f;

	return pOut;
}

/*
 *	shMatrix* shMatrixOrthoLH(shMatrix* pOut,
 *						  float left, float right, float bottom, float top,	
 *						  float zn, float zf)
 *
 *	＜引数＞
 *　　shMatrix* pOut	: 結果行列を表すポインタ
 *　　float left		: 左端座標
 *　　float right		: 右端座標
 *　　float bottom	: 下側座標
 *　　float top		: 上側座標
 *　　float zn		: 近くのビュープレーンのZ値
 *　　float zf		: 遠くのビュープレーンのZ値
 *	＜戻り値＞
 *　　shMatrix*		: 結果行列を表すポインタ
 *	＜説明＞
 *　　オーソ射影行列の作成
 *　　視野に基づいて、左手座標系のオーソ射影行列を作成します。
 */
shMatrix* shMatrixOrthoLH(shMatrix* pOut,
						  float left, float right, float bottom, float top,
						  float zn, float zf)
{
	pOut->_11 = 2.0f / (left - right);
	pOut->_12 = 0.0f;
	pOut->_13 = 0.0f;
	pOut->_14 = 0.0f;

	pOut->_21 = 0.0f;
	pOut->_22 = 2.0f / (top - bottom);
	pOut->_23 = 0.0f;
	pOut->_24 = 0.0f;

	pOut->_31 = 0.0f;
	pOut->_32 = 0.0f;
	pOut->_33 = -2.0f / (zf - zn);
	pOut->_34 = 0.0f;

	pOut->_41 = (left + right) / (left - right);
	pOut->_42 = (top + bottom) / (top - bottom);
	pOut->_43 = (zf + zn) / (zf - zn);
	pOut->_44 = 1.0f;

	return pOut;
}

/*
 *	shMatrix* shMatrixOrthoRH(shMatrix* pOut,
 *						  float left, float right, float bottom, float top,
 *						  float zn, float zf)
 *
 *	＜引数＞
 *　　shMatrix* pOut	: 結果行列を表すポインタ
 *　　float left		: 左端座標
 *　　float right		: 右端座標
 *　　float bottom		: 下側座標
 *　　float top			: 上側座標
 *　　float zn			: 近くのビュープレーンのZ値
 *　　float zf			: 遠くのビュープレーンのZ値
 *	＜戻り値＞
 *　　shMatrix*			: 結果行列を表すポインタ
 *	＜説明＞
 *　　オーソ射影行列の作成
 *　　視野に基づいて、右手座標系のオーソ射影行列を作成します。
 */
shMatrix* shMatrixOrthoRH(shMatrix* pOut,
						  float left, float right, float bottom, float top,
						  float zn, float zf)
{
	pOut->_11 = 2.0f / (left - right);
	pOut->_12 = 0.0f;
	pOut->_13 = 0.0f;
	pOut->_14 = 0.0f;

	pOut->_21 = 0.0f;
	pOut->_22 = 2.0f / (top - bottom);
	pOut->_23 = 0.0f;
	pOut->_24 = 0.0f;

	pOut->_31 = 0.0f;
	pOut->_32 = 0.0f;
	pOut->_33 = 2.0f / (zf - zn);
	pOut->_34 = 0.0f;

	pOut->_41 = (left + right) / (left - right);
	pOut->_42 = (top + bottom) / (top - bottom);
	pOut->_43 = (zf + zn) / (zf - zn);
	pOut->_44 = 1.0f;

	return pOut;
}

/*
 *	void shQuaternionMul(shQuaternion* pOut, shQuaternion* pIn0, shQuaternion* pIn1)
 *
 *	＜引数＞
 *　　shQuaternion* pOut	: 結果四元数を表すポインタ
 *　　shQuaternion* pIn0	: 計算する四元数を表すポインタ
 *　　shQuaternion* pIn1	: 計算する四元数を表すポインタ
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　四元数の掛け算
 *　　pOut = pIn0 * pIn1
 */
void shQuaternionMul(shQuaternion* pOut, shQuaternion* pIn0, shQuaternion* pIn1)
{
	float pw, px, py, pz;
	float qw, qx, qy, qz;

	pw = pIn0->w; px = pIn0->x; py = pIn0->y; pz = pIn0->z;
	qw = pIn1->w; qx = pIn1->x; qy = pIn1->y; qz = pIn1->z;

	pOut->w = pw * qw - px * qx - py * qy - pz * qz;
	pOut->x = pw * qx + px * qw + py * qz - pz * qy;
	pOut->y = pw * qy - px * qz + py * qw + pz * qx;
	pOut->z = pw * qz + px * qy - py * qx + pz * qw;
}

/*
 *	void shQuaternionRotation(shQuaternion* pOut, float rad,
 *						  float ax, float ay, float az)
 *
 *	＜引数＞
 *　　shQuaternion* pOut	: 結果四元数を表すポインタ
 *　　float rad				: 回転角度（Radian）
 *　　float ax				: 回転軸X
 *　　float ay				: 回転軸Y
 *　　float az				: 回転軸Z
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　回転軸を中心に回転する回転四元数を作成します。
 */
void shQuaternionRotation(shQuaternion* pOut, float rad,
						  float ax, float ay, float az)
{
	float hrad;
	float s;

	hrad = 0.5f * rad;
	s = sinf(hrad);

	pOut->w = cosf(hrad);
	pOut->x = s * ax;
	pOut->y = s * ay;
	pOut->z = s * az;
}

/*
 *	void shQuaternionToMatrix(shMatrix *pOut, shQuaternion *pIn)
 *
 *	＜引数＞
 *　　shMatrix* pOut		: 結果行列を表すポインタ
 *　　shQuaternion* pIn		: 変換する四元数を表すポインタ
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　四元数を行列に変換します。
 */
void shQuaternionToMatrix(shMatrix *pOut, shQuaternion *pIn)
{
	float qw, qx, qy, qz;
	float x2, y2, z2;
	float xy, yz, zx;
	float wx, wy, wz;

	qw = pIn->w; qx = pIn->x; qy = pIn->y; qz = pIn->z;

	x2 = 2.0f * qx * qx;
	y2 = 2.0f * qy * qy;
	z2 = 2.0f * qz * qz;

	xy = 2.0f * qx * qy;
	yz = 2.0f * qy * qz;
	zx = 2.0f * qz * qx;

	wx = 2.0f * qw * qx;
	wy = 2.0f * qw * qy;
	wz = 2.0f * qw * qz;

	pOut->m[0][0] = 1.0f - y2 - z2;
	pOut->m[0][1] = xy - wz;
	pOut->m[0][2] = zx + wy;
	pOut->m[0][3] = 0.0f;

	pOut->m[1][0] = xy + wz;
	pOut->m[1][1] = 1.0f - z2 - x2;
	pOut->m[1][2] = yz - wx;
	pOut->m[1][3] = 0.0f;

	pOut->m[2][0] = zx - wy;
	pOut->m[2][1] = yz + wx;
	pOut->m[2][2] = 1.0f - x2 - y2;
	pOut->m[2][3] = 0.0f;

	pOut->m[3][0] = pOut->m[3][1] = pOut->m[3][2] = 0.0f;
	pOut->m[3][3] = 1.0f;
}

//EOF