﻿/*
 * マルチプラットフォーム描画エンジン「Sherry」
 * Copyright(C) 2010-2011 SherryProject. all rights reserved.
 *
 * The MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/*
 *	shApplication_ios.c
 *		Copyright (C) 2010-2011 Cap5ule. all rights reserved.
 * 
 *	Date		|Version	|Author		|Summary
 *	2011/12/26	|v0.01		|Cap5ule	|初回版
 */

// Mac OS X(Snow Leopard)のみ
#ifdef MACOSX

/*---- インクルードファイル ----*/

#import <Cocoa/Cocoa.h>
#import <OpenGL/OpenGL.h>
#import <OpenGL/gl.h>

#include "sherry.h"				//!<共通ヘッダー
#include "shApplication.h"		//!<ヘッダー
#include "shApplication_osx_sys.h"

/*---- マクロ定義 ----*/


/*---- スタティックプロトタイプ宣言 ----*/
// 説明は実装部に記載

// OpenGLの初期化を行う
static int shInitOpenGL(int width, int height);

/*---- グローバル変数 ----*/

// ウィンドウとコンテキスト
GLContextView*		g_sh_context;			//!<コンテキスト
shDrawMode			g_sh_DrawMode;			//!<描画モード
// 描画スレッド（メインスレッド）
void			(*g_sh_DrawFunc)();		//!<スレッド内で呼び出される関数のポインタ
// 更新スレッド
void			(*g_sh_UpdateFunc)();	//!<スレッド内で呼び出される関数のポインタ
int				g_sh_UdFrameRate;
// 描画設定
unsigned int	g_sh_clsFlag;			//!<クリアターゲットフラグ
// システム変数
NSAutoreleasePool* g_pool;				//!<オープリリースプール
unsigned long	g_sh_DrawPreTime;		//!<描画：前フレームの時間
float			g_sh_DrawFPS;			//!<描画：現在のFPS
unsigned long	g_sh_UpdatePreTime;		//!<更新：前フレームの時間
float			g_sh_UpdateFPS;			//!<更新：現在のFPS
// 入力系
int				g_sh_MButton;			//!<内部入力 押されているかどうか
float			g_sh_MCursor[2];		//!<内部入力 0:X,1:Y

/*---- スタティック関数 ----*/

/*
 *	static int shInitOpenGL(int width, int height)
 * 
 *	＜引数＞
 *	　int width		：OpenGLの描画横幅
 *	　int height	：OpenGLの描画縦幅
 *	＜戻り値＞
 *	　int			： 初期化に成功した場合は、0が返ります。
 *	＜説明＞
 *	　OpenGLとGLEWを初期化し、描画を可能にします。
 */
static int shInitOpenGL(int width, int height)
{
	NSRect frameRect = NSMakeRect(0, 0, width, height);
	//コンテキストビューを生成
	g_sh_context = [[GLContextView alloc] initWithFrame:frameRect];
	if(!g_sh_context)
		return 1;

	// フレームバッファとレンダーバッファを生成
	//		NSOpenGLContextに任せる
	
	// OpenGLデフォルト設定
	glClearColor(0, 0, 0, 1);	// デフォルト背景色
	g_sh_clsFlag = GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT;	// デフォルトクリアターゲットフラグ
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glEnable(GL_TEXTURE_2D);
	
	//glEnable(GL_CULL_FACE); 
	//glCullFace(GL_BACK);
	
	// ビューポートの設定
	glViewport(0, 0, width , height);
	glScissor(0, 0, width, height);

	return 0;
}

/*---- 関数 ----*/

/*
 *	int shInitCreateSherry(const char* title,
 *							 int width, int height, int cmd)
 *
 *	＜引数＞
 *	　HINSTANCE hInst	: モジュールのハンドル
 *	　char* title		: ウィンドウタイトル
 *　　int	width		: ウィンドウ幅サイズ
 *　　int height		: ウィンドウ縦サイズ
 *　　shDrawMode mode : 描画モード
 *	＜戻り値＞
 *　　int				: エラーコードが返ります
 *	＜説明＞
 *　　描画エンジン「Sherry」を使用するための初期化とウィンドウの作成を行い
 *　　ます。必ず必要となる。
 */
int shInitCreateSherry(const char* title,
						int width, int height, shDrawMode mode)
{
	//[AppDelegate class];
	
	//オートリリースプールを作成
	g_pool = [[NSAutoreleasePool alloc] init];
	
	if(shInitOpenGL(width,height))
		return 1;
	
	g_sh_DrawMode = mode;
	
	// 変数初期化
	g_sh_DrawFunc = NULL;
	g_sh_UpdateFunc = NULL;
	g_sh_UdFrameRate = 60;
	g_sh_DrawPreTime = 0;
	g_sh_DrawFPS = 0.0f;
	g_sh_UpdatePreTime = 0;
	g_sh_UpdateFPS = 0.0f;
	g_sh_MButton = 0;
	g_sh_MCursor[0] = g_sh_MCursor[1] = 0.0f;
	
    return 0;
}

/*
 *	int shExitSherry()
 *
 *	＜引数＞
 *　　なし
 *	＜戻り値＞
 *　　int				: エラーコードが返ります
 *	＜説明＞
 *　　描画エンジン「Sherry」を使用した際に使ったメモリを解放します。
 *　　アプリケーションの終了時に必ず呼んでください。
 */
int shExitSherry()
{
	// OpenGLContext関連をリリース
    [g_sh_context dealloc];
	g_sh_context = nil;
	
	//リリースプールをリリース
	[g_pool release];
	
	return 0;
}

/*
 *	void shMainSherry()
 *
 *	＜引数＞
 *　　なし
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　描画エンジン「Sherry」メインループ関数です。
 *　　この関数を呼び出すことで、プログラムはイベントの待ち受け状態になります。
 */
void shMainSherry()
{
	// アプリケーションを立ち上げる
	id app = [NSApplication sharedApplication];
	AppDelegate* delegate = [[AppDelegate alloc] init]; 
	[app setDelegate:delegate]; 
	[app run];
}

/*
 *	int shSetUpdateThread(int framerate, void(*thd)())
 *
 *	＜引数＞
 *　　int	framerate	: 更新間隔（フレームレート）
 *　　void(*thd)()	: 更新用関数のポインタ：使用しない場合は[NULL]
 *	＜戻り値＞
 *　　int				: エラーコードが返ります
 *	＜説明＞
 *　　更新用スレッドを使用します。更新関数を別スレッドとして動作し、
 *　　そうすることで一定のアニメーションを可能にします。
 */
int shSetUpdateThread(int framerate, void(*thd)())
{
	// 更新間隔
	g_sh_UdFrameRate = framerate;
	
	// 関数アドレスを入れる
	g_sh_UpdateFunc = thd;

	return 0;
}

/*
 *	int shSetDrawThread(void(*thd)())
 *
 *	＜引数＞
 *　　void(*thd)()	: 描画用関数のポインタ：使用しない場合は[NULL]
 *	＜戻り値＞
 *　　int				: エラーコードが返ります
 *	＜説明＞
 *　　描画関数を別スレッドとして動作します。
 *　　そうすることで最高のパフォーマンスの提供を可能にします。
 */
int shSetDrawThread(void(*thd)())
{
	// 関数アドレスを入れる
	g_sh_DrawFunc = thd;
	
	return 0;
}

/*
 *	void shDCP()
 *
 *	＜引数＞
 *		char* filePath	: 変換するファイルパス
 *	＜戻り値＞
 *		char*			: 変換されたコンテンツのパスを返す
 *	＜説明＞
 *　　DeviceContentsPathの省略形
 *　　デバイス等のハードウェア内部パスの違いをなくし、
 *　　コンテンツパスへとつなげる統一性を図る関数です。
 */
char* shDCP(const char* filePath)
{
	/* 変換内容
	 * ASCII⇒NSString⇒デバイスパスを追加し変換⇒NSString⇒ASCII
	 */
	static char str[1024];
	
	// 変換
	NSString* nsStr = [NSString stringWithCString: filePath encoding:NSASCIIStringEncoding];
	NSString* pathStr = [[[NSBundle mainBundle] resourcePath] stringByAppendingPathComponent:nsStr];
	strcpy(str, [pathStr cStringUsingEncoding:NSUTF8StringEncoding]);
	[nsStr release];
	[pathStr release];
	
	return (char*)str;
}

/*
 *	float shGetUpdateFPS()
 * 
 *	＜引数＞
 *　　なし
 *	＜戻り値＞
 *　　float			: 平均更新回数
 *	＜説明＞
 *　　１秒間あたりの更新回数を取得する。
 */
float shGetUpdateFPS()
{
	return g_sh_UpdateFPS;
}

/*
 *	float shGetDrawFPS()
 *
 *	＜引数＞
 *　　なし
 *	＜戻り値＞
 *　　float			: 平均描画回数
 *	＜説明＞
 *　　１秒間あたりの描画回数を取得する。
 */
float shGetDrawFPS()
{
	return g_sh_DrawFPS;
}

/*
 *	int shGetCursorPosX(float* x, float* y)
 *
 *	＜引数＞
 *　　float* x		: カーソルのＸ座標を入れる
 *　　float* y		: カーソルのＹ座標を入れる
 *	＜戻り値＞
 *　　int			: 押されている場合は１が返り、押されてない場合は０が返ります。
 *	＜説明＞
 *　　基本共通入力取得用関数。
 *　　ＰＣの場合は、カーソル位置を取得でき、
 *　　スマートフォンの場合は、押されている位置を取得できる。※実用に使うべきでない。
 */
int shGetCursorPos(float* x, float* y)
{
	//カーソル情報を代入
	(*x) = g_sh_MCursor[0];
	(*y) = g_sh_MCursor[1];
	//押下判定
	return g_sh_MButton;
}


#endif	/*TARGET_OS_MACOSX*/
