﻿/*
 * マルチプラットフォーム描画エンジン「Sherry」
 * Copyright(C) 2010-2011 SherryProject. all rights reserved.
 *
 * The MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/*
 *	shMath.h
 *		Copyright (C) 2010-2011 Cap5ule. all rights reserved.
 *
 *	Date		|Version	|Author		|Summary
 *　2010/11/13	|v0.01		|onakah		|初回版
 *　2010/11/21	|v0.02		|onakah		|構造体定義の追加、関数群の宣言を追加
 *　2010/11/24	|v0.03		|onakah		|行列関連の関数を追加、定数マクロを追加
 *　			|			|			|2次元、4次元関連の関数追加
 *　2010/11/26	|v0.04		|onakah		|射影変換行列、ビュー行列作成関数を
 *				|			|			|左手右手系それぞれ追加。
 *　2010/11/26	|v0.05		|Cap5ule	|平行投影変換行列を追加。
 *　2011/02/02	|v0.06		|onakah		|四元数関連関数群追加。
 */

#ifndef _SH_MATH_H_
#define _SH_MATH_H_

//todo 4次元。向き変換ベクトル。四元数。

/*---- 定数定義 ----*/

#define PI			(3.1415926535897932f)	//!<円周率
#define HALF_PI		PI/2					//!<半円周率

/*---- マクロ定義 ----*/

#define TO_DEGREE(Radian)	((float)Radian * (180.0f / PI))
#define TO_RADIAN(Degree)	((float)(Degree) * (PI/180.0f))


/*---- データ定義 ----*/
/*	必要なら
typedef   signed char		s8;
typedef unsigned char		u8;
typedef   signed short		s16;
typedef unsigned short		u16;
typedef   signed int		s32;
typedef unsigned int		u32;
typedef			 __int64	s64;
typedef unsigned __int64	u64;
typedef float				f32;
typedef long double			f64;
*/

/*---- 構造体定義 ----*/

/*
 *	Vector2　：　２次元ベクトル
 */
typedef union _shvector2
{
	union {
		float v[2];
		struct {
			float x;
			float y;
		};
	};
} shVector2;

/*
 *	Vector3　：　３次元ベクトル
 */
typedef struct _shvector3
{
	union {
		float v[3];
		struct {
			float x;
			float y;
			float z;
		};
	};
} shVector3;

/*
 *	Vector4　：　４次元ベクトル
 */
typedef struct _shvector4
{
	union {
		float v[4];
		struct {
			float x;
			float y;
			float z;
			float w;
		};
	};
} shVector4;

/*
 *	Matrix　：　４×４行列
 */
typedef struct _shmatrix
{
	union {
		float m[4][4];
		struct {
			float _11, _12, _13, _14;
			float _21, _22, _23, _24;
			float _31, _32, _33, _34;
			float _41, _42, _43, _44;
		};
	};
} shMatrix;

/*
 *	Quaternion　：　四元数
 */
typedef struct _shquaternion
{
	float x, y, z, w;
} shQuaternion;


/*---- 外部グローバル変数 ----*/


/*---- プロトタイプ宣言 ----*/

#ifdef __cplusplus
extern "C" {
#endif

/*
 *	Vector2
 */

/*
 *	shVector2 shVec3Multiply(shVector2* pV1, shVector2* pV2)
 *
 *	＜引数＞
 *　　shVector2* pV1	: ベクトル
 *　　shVector2* pV2	: ベクトル
 *	＜戻り値＞
 *　　shVector2			: 計算後の結果
 *	＜説明＞
 *　　２次元ベクトルの積を求める
 *　　引数同士の掛け算を行い結果を返します。
 */
shVector2 shVec2Multiply(shVector2* pV1, shVector2* pV2);

/*
 *	shVector2 shVec2Division(shVector2* pV1, shVector2* pV2)
 *
 *	＜引数＞
 *　　shVector2* pV1	: ベクトル
 *　　shVector2* pV2	: ベクトル
 *	＜戻り値＞
 *　　shVector2			: 計算後の結果
 *	＜説明＞
 *　　２次元ベクトルの商を求める
 *　　pV1からpV2を割り、結果を返します。
 */
shVector2 shVec2Division(shVector2* pV1, shVector2* pV2);

/*
 *	shVector2 shVec2Add(shVector2* pV1, shVector2* pV2)
 *
 *	＜引数＞
 *　　shVector2* pV1	: ベクトル
 *　　shVector2* pV2	: ベクトル
 *	＜戻り値＞
 *　　shVector2		: 計算後の結果
 *	＜説明＞
 *　　２次元ベクトルの和を求める
 *　　引数同士の足し算を行い結果を返します。
 */
shVector2 shVec2Add(shVector2* pV1, shVector2* pV2);

/*
 *	shVector2 shVec2Sub(shVector2* pV1, shVector2* pV2)
 *
 *	＜引数＞
 *　　shVector2* pV1	: ベクトル
 *　　shVector2* pV2	: ベクトル
 *	＜戻り値＞
 *　　shVector2			: 計算後の結果
 *	＜説明＞
 *　　２次元ベクトルの差を求める
 *　　pV1からpV2を引いた結果を返します。
 */
shVector2 shVec2Sub(shVector2* pV1, shVector2* pV2);

/*
 *	float shVec2Dot(shVector2* pV1, shVector2* pV2)
 *
 *	＜引数＞
 *　　shVector2* pV1	: ベクトル
 *　　shVector2* pV2	: ベクトル
 *	＜戻り値＞
 *　　shVector2			: 計算結果
 *	＜説明＞
 *　　２次元ベクトルの内積の計算結果を返します。
 */
float shVec2Dot(shVector2* pV1, shVector2* pV2);

/*
 *	float shVec2Cross(shVector2* pV1, shVector2* pV2)
 *
 *	＜引数＞
 *　　shVector2* pV1	: ベクトル
 *　　shVector2* pV2	: ベクトル
 *	＜戻り値＞
 *　　float				: 計算結果
 *	＜説明＞
 *　　２次元ベクトルの外積の計算結果を返します。
 */
float shVec2Cross(shVector2* pV1, shVector2* pV2);

/*
 *	float shVec2Length(shVector2* pV)
 *
 *	＜引数＞
 *　　shVector2* pV	: ベクトル
 *	＜戻り値＞
 *　　float			: 長さ
 *	＜説明＞
 *　　ベクトルの長さを求めます。
 */
float shVec2Length(shVector2* pV);

/*
 *	void shVec2Normalize(shVector2* pV)
 *
 *	＜引数＞
 *　　shVector2* pV	: ベクトル
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　２次元ベクトルを正規化します。
 */
void shVec2dNormalize(shVector2* pV);

/*
 *	shVector2 shVec2Linear(shVector2* pV1, shVector2* pV2, float t)
 *
 *	＜引数＞
 *　　shVector2* pV1	: ベクトル
 *　　shVector2* pV2	: ベクトル
 *　　float t			: 割合（0.0～1.0）
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　２次元ベクトルの線形補間を行う
 *　　pV1からpV2間を割合tで線形補間し、結果を返します。
 */
shVector2 shVec2Linear(shVector2* pV1, shVector2* pV2, float t);


/*
 *	Vector3
 */

/*
 *	shVector3 shVec3Multiply(shVector3* pV1, shVector3* pV2)
 *
 *	＜引数＞
 *		shVector3* pV1	: ベクトル
 *		shVector3* pV2	: ベクトル
 *	＜戻り値＞
 *		shVector3		: 計算後の結果
 *	＜説明＞
 *　　３次元ベクトルの積を求める
 *　　引数同士の掛け算を行い結果を返します。
 */
shVector3 shVec3Multiply(shVector3* pV1, shVector3* pV2);

/*
 *	shVector3 shVec3Division(shVector3* pV1, shVector3* pV2)
 *
 *	＜引数＞
 *　　shVector3* pV1	: ベクトル
 *　　shVector3* pV2	: ベクトル
 *	＜戻り値＞
 *　　shVector3		　　: 計算後の結果
 *	＜説明＞
 *　　３次元ベクトルの商を求める
 *　　pV1からpV2を割り、結果を返します。
 */
shVector3 shVec3Division(shVector3* pV1, shVector3* pV2);

/*
 *	shVector3 shVec3Add(shVector3* pV1, shVector3* pV2)
 *
 *	＜引数＞
 *　　shVector3* pV1	: ベクトル
 *　　shVector3* pV2	: ベクトル
 *	＜戻り値＞
 *　　shVector3			: 計算後の結果
 *	＜説明＞
 *　　３次元ベクトルの和を求める
 *　　引数同士の足し算を行い結果を返します。
 */
shVector3 shVec3Add(shVector3* pV1, shVector3* pV2);

/*
 *	shVector3 shVec3Sub(shVector3* pV1, shVector3* pV2)
 *
 *	＜引数＞
 *　　shVector3* pV1	: ベクトル
 *　　shVector3* pV2	: ベクトル
 *	＜戻り値＞
 *　　shVector3		: 計算後の結果
 *	＜説明＞
 *　　３次元ベクトルの差を求める
 *　　pV1からpV2を引いた結果を返します。
 */
shVector3 shVec3Sub(shVector3* pV1, shVector3* pV2);

/*
 *	float shVec3Dot(shVector3* pV1, shVector3* pV2)
 *
 *	＜引数＞
 *		shVector3* pV1	: ベクトル
 *		shVector3* pV2	: ベクトル
 *	＜戻り値＞
 *		shVector3		: 計算結果
 *	＜説明＞
 *		３次元ベクトル同士の内積を返します。
 */
float shVec3Dot(shVector3* pV1, shVector3* pV2);

/*
 *	shVector3 shVec3Cross(shVector3* pV1, shVector3* pV2)
 *
 *	＜引数＞
 *　　shVector3* pV1	: ベクトル
 *　　shVector3* pV2	: ベクトル
 *	＜戻り値＞
 *　　shVector3		: 計算結果
 *	＜説明＞
 *　　３次元ベクトル同士の外積を返します。
 */
shVector3 shVec3Cross(shVector3* pV1, shVector3* pV2);

/*
 *	float shVec3Length(shVector3* pV)
 *
 *	＜引数＞
 *		shVector3* pV	: ベクトル
 *	＜戻り値＞
 *		float			: 長さ
 *	＜説明＞
 *		３次元ベクトルの長さを求めます。
 */
float shVec3Length(shVector3* pV);

/*
 *	void shVec3Normalize(shVector3* pV)	
 *
 *	＜引数＞
 *		shVector3* pV	: 行列
 *	＜戻り値＞
 *		なし
 *	＜説明＞
 *		３次元ベクトルのベクトルを正規化します。
 */
void shVec3Normalize(shVector3* pV);

/*
 *	shVector3 shVec3Linear(shVector3* pV1, shVector3* pV2, float t)
 *
 *	＜引数＞
 *　　shVector3* pV1	: 行列
 *　　shVector3* pV2	: 行列
 *　　float t			: 割合（0.0～1.0）
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　３次元ベクトルの線形補間を行う
 *　　pV1からpV2間を割合tで線形補間し、結果を返します。
 */
shVector3 shVec3Linear(shVector3* pV1, shVector3* pV2, float t);

// Transform（回転移動行列で変換した座標を返す。たぶん。）
//shVector4 shTransform(shVector3 pV, shMatrix* pM);


/*
 *	Vector4
 */

/*
 *	shVector4 shVec4Add(shVector4* pV1, shVector4* pV2)
 *
 *	＜引数＞
 *　　shVector4* pV1	: ベクトル
 *　　shVector4* pV2	: ベクトル
 *	＜戻り値＞
 *　　shVector4			: 計算結果
 *	＜説明＞
 *　　ベクトルの和
 *　　引数同士の和を求め、結果を返します。
 */
shVector4 shVec4Add(shVector4* pV1, shVector4* pV2);

/*
 *	shVector4 shVec4Sub(shVector4* pV1, shVector4* pV2)
 *
 *	＜引数＞
 *　　shVector4* pV1	: ベクトル
 *　　shVector4* pV2	: ベクトル
 *	＜戻り値＞
 *　　shVector4			: 計算結果
 *	＜説明＞
 *　　ベクトルの差
 *　　pV1からpV2を引いた結果を返します。
 */
shVector4 shVec4Sub(shVector4* pV1, shVector4* pV2);

/*
 *	void shVec4Normalize(shVector4* pV)
 *
 *	＜引数＞
 *　　shVector4* pV		: ベクトル
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　ベクトルの正規化を行います。
 */
void shVec4Normalize(shVector4* pV);

/*
 *	float shVec4Dot(shVector4* pV1, shVector4* pV2)
 * 
 *	＜引数＞
 *		shVector4* pV1	: ベクトル
 *		shVector4* pV2	: ベクトル
 *	＜戻り値＞
 *		float			: 計算結果
 *	＜説明＞
 *		４次元ベクトルの内積を求めます。
 */
float shVec4Dot(shVector4* pV1, shVector4* pV2);

//float shVec4Length(shVector4* pV);


/********************************************************************************
 *	Matrix																		*
 ********************************************************************************/

/*
 *	void shMatrixIdentity(shMatrix* pM)
 *
 *	＜引数＞
 *		shMatrix* pM	: 行列
 *	＜戻り値＞
 *		なし
 *	＜説明＞
 *		行列を単位行列化します。
 */
void shMatrixIdentity(shMatrix* pOut);

/*
 *	shMatrix* shMatrixMultiply(shMatrix* pOut, shMatrix* pM1, shMatrix* pM2)
 *
 *	＜引数＞
 *　　shMatrix* pOut	: 計算結果
 *　　shMatrix* pM1		: 行列
 *　　shMatrix* pM2		: 行列
 *	＜戻り値＞
 *　　shMatrix*			: 計算結果
 *	＜説明＞
 *　　行列同士の乗算	
 *　　引数同士の積を返します。
 */
shMatrix* shMatrixMultiply(shMatrix* pOut, shMatrix* pM1, shMatrix* pM2);

/*
 *	shVector4* shVec3Transform(shVector4* pOut, shVector3* pV, shMatrix* pM)
 *
 *	＜引数＞
 *		shVector4* pOut	: 計算結果
 *		shVector3* pV	: ベクトル
 *		shMatrix* pM	: 行列
 *	＜戻り値＞
 *		shVector4*		: 計算結果
 *	＜説明＞
 *		３次元ベクトルと行列の乗算を行い、４次元ベクトルを返します。
 */
shVector4* shVec3Transform(shVector4* pOut, shVector3* pV, shMatrix* pM);

/*
 *	shVector4* shVec4Transform(shVector4* pOut, shVector4* pV, shMatrix* pM)
 *
 *	＜引数＞
 *		shVector4* pOut	: 計算結果
 *		shVector4* pV	: ベクトル
 *		shMatrix* pM	: 行列
 *	＜戻り値＞
 *		shVector4*		: 計算結果
 *	＜説明＞
 *		４次元ベクトルと行列の乗算を行い、４次元ベクトルを返します。
 */
shVector4* shVec4Transform(shVector4* pOut, shVector4* pV, shMatrix* pM);

/*
 *	shMatrix shMatrixScaling(shMatrix* pOut, float fScaleX,	
 *							 float fScaleY, float fScaleZ)
 *
 *	＜引数＞
 *		shMatrix* pOut	: 行列
 *		float fScaleX	: X軸方向拡大率
 *		float fScaleY	: Y軸方向拡大率
 *		float fScaleZ	: Z軸方向拡大率
 *	＜戻り値＞
 *		shMatrix*		: 結果行列
 *	＜説明＞
 *		拡大縮小を行う行列を作成します。
 */
shMatrix* shMatrixScaling(shMatrix* pOut, float fScaleX,
						  float fScaleY, float fScaleZ);

/*
 *	shMatrix* shMatrixRotationX(shMatrix* pOut, float Degree)
 *
 *	＜引数＞
 *		shMatrix* pOut	: 行列
 *		float Degree	: 回転角度
 *	＜戻り値＞
 *		shMatrix*		: 結果行列
 *	＜説明＞
 *		X軸を中心に左ねじ方向に、Degree度回転するための行列を作成します。
 */
shMatrix* shMatrixRotationX(shMatrix* pOut, float Degree);

/*
 *	shMatrix* shMatrixRotationY(shMatrix* pOut, float Degree)
 *
 *	＜引数＞
 *		shMatrix* pOut	: 行列
 *		float Degree	: 回転角度
 *	＜戻り値＞
 *		shMatrix*		: 結果行列
 *	＜説明＞
 *		Y軸を中心に左ねじ方向に、Degree度回転するための行列を作成します。
 */
shMatrix* shMatrixRotationY(shMatrix* pOut, float Degree);

/*
 *	shMatrix* shMatrixRotationZ(shMatrix* pOut, float Degree)
 *
 *	＜引数＞
 *　　shMatrix* pOut	: 行列
 *　　float Degree	: 回転角度
 *	＜戻り値＞
 *　　shMatrix*		: 結果行列
 *	＜説明＞
 *　　Z軸を中心に左ねじ方向に、Degree度回転するための行列を作成します。
 */
shMatrix* shMatrixRotationZ(shMatrix* pOut, float Degree);

/*
 *	shMatrix* shMatrixRotationYawPitchRoll(shMatrix* pOut, float Yaw,
 *										   float Pitch, float Roll)
 *
 *	＜引数＞
 *　　shMatrix* pOut	: 行列
 *　　float Yaw			: ヨー（Degree）
 *　　float Pitch		: ピッチ（Degree）
 *　　float Roll		: ロー（Degree）
 *	＜戻り値＞
 *　　shMatrix*			: 結果行列
 *	＜説明＞
 *　　Y軸、X軸、Z軸の順に左ねじ方向に、それぞれのヨーピッチロー角度（Degree）
 *　　回転するための行列を作成します。
 */
shMatrix* shMatrixRotationYawPitchRoll(shMatrix* pOut, float Yaw,
									   float Pitch, float Roll);

/*
 *	shMatrix* shMatrixRotationEuler(shMatrix* pOut, float x, float y, float z)
 *
 *	＜引数＞
 *　　shMatrix* pOut	: 行列
 *　　float x			: X回転角度（Degree）
 *　　float y			: Y回転角度（Degree）
 *　　float z			: Z回転角度（Degree）
 *	＜戻り値＞
 *　　shMatrix*		: 結果
 *	＜説明＞
 *　　Y軸、X軸、Z軸の順に左ねじ方向に、それぞれの角度（Degree）
 *　　回転するための行列を作成します。
 */
shMatrix* shMatrixRotationEuler(shMatrix* pOut, float x, float y, float z);

/*
 *	shMatrix* shMatrixTranslation(shMatrix* pOut, float fTransX,
 *								  float fTransY, float fTransZ)
 *
 *	＜引数＞
 *		shMatrix* pOut	: 行列
 *		float fTransX	: X移動量
 *		float fTransY	: Y移動量
 *		float fTransZ	: Z移動量
 *	＜戻り値＞
 *		shMatrix*		: 結果行列
 *	＜説明＞
 *		x, y, zだけ平行移動するための行列を作成します。
 */
shMatrix* shMatrixTranslation(shMatrix* pOut, float fTransX,
							  float fTransY, float fTransZ);

/*
 *	shMatrix* shMatrixInverse(shMatrix* pOut, shMatrix* pIn)
 *
 *	＜引数＞
 *		shMatrix* pOut	: 行列
 *		shMatrix* pIn	: 行列
 *	＜戻り値＞
 *		shMatrix*		: 結果
 *	＜説明＞
 *		逆行列を求めます。
 */
shMatrix* shMatrixInverse(shMatrix* pOut, shMatrix* pIn);

/*
 *	shMatrix* shLookAtMatrixLH(shMatrix* pOut, shVector3* pEye,
 *						   shVector3* pAt, shVector3* pUp)
 *
 *	＜引数＞
 *		shMatrix* pOut	: 結果行列を表すポインタ
 *		shVector3* pEye	: 視点
 *		shVector3* pAt	: カメラの注視点
 *		shVector3* pUp	: 上方向のベクトル
 *	＜戻り値＞
 *		shMatrix*		: 結果行列を表すポインタ
 *	＜説明＞
 *		ビュー行列の作成
 *		左手座標系ビュー行列を作成します。
 */
shMatrix* shLookAtMatrixLH(shMatrix* pOut, shVector3* pEye,
						   shVector3* pAt, shVector3* pUp);
/*
 *	shMatrix* shLookAtMatrixRH(shMatrix* pOut, shVector3* pEye,
 *						   shVector3* pAt, shVector3* pUp)
 *
 *	＜引数＞
 *　　shMatrix* pOut	: 結果行列を表すポインタ
 *　　shVector3* pEye	: 視点
 *　　shVector3* pAt	: カメラの注視点
 *　　shVector3* pUp	: 上方向のベクトル
 *	＜戻り値＞
 *　　shMatrix*			: 結果行列を表すポインタ
 *	＜説明＞
 *　　ビュー行列の作成
 *　　右手座標系ビュー行列を作成します。
 */
shMatrix* shLookAtMatrixRH(shMatrix* pOut, shVector3* pEye,
						   shVector3* pAt, shVector3* pUp);
/*
 *	shMatrix* shMatrixPerspectiveFovLH(shMatrix* pOut, float fovY,
 *								   float Aspect, float zn, float zf)
 *
 *	＜引数＞
 *　　shMatrix* pOut	: 結果行列を表すポインタ
 *　　float fovY		: Y方向の視野角（ラジアン）
 *　　float Aspect		: アスペクト比
 *　　float zn			: 近くのビュープレーンのZ値	
 *　　float zf			: 遠くのビュープレーンのZ値
 *	＜戻り値＞
 *　　shMatrix*			: 結果行列を表すポインタ
 *	＜説明＞
 *　　パースペクティブ射影行列の作成
 *　　視野に基づいて、左手座標系のパースペクティブ射影行列を作成します。
 */
shMatrix* shMatrixPerspectiveFovLH(shMatrix* pOut, float fovY,
								   float Aspect, float zn, float zf);

/*
 *	shMatrix* shMatrixPerspectiveFovRH(shMatrix* pOut, float fovY,
 *								   float Aspect, float zn, float zf)
 *
 *	＜引数＞
 *　　shMatrix* pOut	: 結果行列を表すポインタ
 *　　float fovY		: Y方向の視野角（ラジアン）
 *　　float Aspect	: アスペクト比
 *　　float zn		: 近くのビュープレーンのZ値
 *　　float zf		: 遠くのビュープレーンのZ値
 *	＜戻り値＞
 *　　shMatrix*		: 結果行列を表すポインタ
 *	＜説明＞
 *　　パースペクティブ射影行列の作成
 *　　視野に基づいて、右手座標系のパースペクティブ射影行列を作成します。
 */
shMatrix* shMatrixPerspectiveFovRH(shMatrix* pOut, float fovY, float Aspect,
								   float zn, float zf);

/*
 *	shMatrix* shMatrixOrthoLH(shMatrix* pOut,
 *						  float left, float right, float bottom, float top,	
 *						  float zn, float zf)
 *
 *	＜引数＞
 *　　shMatrix* pOut	: 結果行列を表すポインタ
 *　　float left		: 左端座標
 *　　float right		: 右端座標
 *　　float bottom	: 下側座標
 *　　float top		: 上側座標
 *　　float zn		: 近くのビュープレーンのZ値
 *　　float zf		: 遠くのビュープレーンのZ値
 *	＜戻り値＞
 *　　shMatrix*		: 結果行列を表すポインタ
 *	＜説明＞
 *　　オーソ射影行列の作成
 *　　視野に基づいて、左手座標系のオーソ射影行列を作成します。
 */
shMatrix* shMatrixOrthoLH(shMatrix* pOut,
						  float left, float right, float bottom, float top,
						  float zn, float zf);

/*
 *	shMatrix* shMatrixOrthoRH(shMatrix* pOut,
 *						  float left, float right, float bottom, float top,
 *						  float zn, float zf)
 *
 *	＜引数＞
 *　　shMatrix* pOut	: 結果行列を表すポインタ
 *　　float left		: 左端座標
 *　　float right		: 右端座標
 *　　float bottom		: 下側座標
 *　　float top			: 上側座標
 *　　float zn			: 近くのビュープレーンのZ値
 *　　float zf			: 遠くのビュープレーンのZ値
 *	＜戻り値＞
 *　　shMatrix*			: 結果行列を表すポインタ
 *	＜説明＞
 *　　オーソ射影行列の作成
 *　　視野に基づいて、右手座標系のオーソ射影行列を作成します。
 */
shMatrix* shMatrixOrthoRH(shMatrix* pOut,
						  float left, float right, float bottom, float top,
						  float zn, float zf);

/*
 *	Quaternion
 */

/*
 *	void shQuaternionMul(shQuaternion* pOut, shQuaternion* pIn0, shQuaternion* pIn1)
 *
 *	＜引数＞
 *　　shQuaternion* pOut	: 結果四元数を表すポインタ
 *　　shQuaternion* pIn0	: 計算する四元数を表すポインタ
 *　　shQuaternion* pIn1	: 計算する四元数を表すポインタ
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　四元数の掛け算
 *　　pOut = pIn0 * pIn1
 */
void shQuaternionMul(shQuaternion* pOut, shQuaternion* pIn0, shQuaternion* pIn1);

/*
 *	void shQuaternionRotation(shQuaternion* pOut, float rad,
 *						  float ax, float ay, float az)
 *
 *	＜引数＞
 *　　shQuaternion* pOut	: 結果四元数を表すポインタ
 *　　float rad				: 回転角度（Radian）
 *　　float ax				: 回転軸X
 *　　float ay				: 回転軸Y
 *　　float az				: 回転軸Z
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　回転軸を中心に回転する回転四元数を作成します。
 */
void shQuaternionRotation(shQuaternion* pOut, float rad,
						  float ax, float ay, float az);

/*
 *	void shQuaternionToMatrix(shMatrix *pOut, shQuaternion *pIn)
 *
 *	＜引数＞
 *　　shMatrix* pOut		: 結果行列を表すポインタ
 *　　shQuaternion* pIn		: 変換する四元数を表すポインタ
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　四元数を行列に変換します。
 */
void shQuaternionToMatrix(shMatrix *pOut, shQuaternion *pIn);



#ifdef __cplusplus
}
#endif

#endif /*_SH_MATH_H_*/
