#pragma once

#include <cassert>

#include "GlInclude.h"


namespace lib_gl
{


class GLBitmapRender
{
public:
	GLBitmapRender(void);
	virtual ~GLBitmapRender(void);

	bool InitializeRender(int w, int h, int num_bits);

	void FinalizeRender(void);

	void BeginRender(void);
	void EndRender(void);

	HDC     GetBmpDC        (void) const { return m_hdc;     }
	HGLRC   GetBmpRC        (void) const { return m_hglrc;   }
	HBITMAP GetBmpHandle    (void) const { return m_hbmp;    }
	HBITMAP GetBmpPreHandle (void) const { return m_hbmpOld; }

	int GetWidth(void) const { return m_Width; }
	int GetHeight(void) const { return m_Height; }
	int GetNumPixelBits(void) const { return m_NumPixelBits; }

protected:
	HBITMAP CreateBitmapRenderBuffer(int w, int h, int num_bits, const HDC& hdc);
	bool SetPfdForGDIBuffer(const HDC& hdc, int color_bits);


protected:
	HDC     m_hdc;
	HGLRC   m_hglrc;
	HBITMAP m_hbmp;
	HBITMAP m_hbmpOld;

	int m_Width;
	int m_Height;
	int m_NumPixelBits;

private:
	bool m_Initialized;
};


inline GLBitmapRender::GLBitmapRender(void) :
	m_Initialized(false),
	m_Width(0),
	m_Height(0),
	m_NumPixelBits(0)
{
}

inline GLBitmapRender::~GLBitmapRender(void)
{
	if( m_Initialized )
		FinalizeRender();
}

inline bool GLBitmapRender::InitializeRender(int w, int h, int num_bits)
{
	assert( !m_Initialized );
	if( m_Initialized )
		return false;

	// ݂̃fBXvCƓl̐ݒHDC𐶐
	m_hdc = CreateCompatibleDC( NULL );

	// OpenGL`̃rbg}bv𐶐
	m_hbmp = CreateBitmapRenderBuffer(w, h, num_bits, m_hdc);

	// HDC̕`obt@ݒ
	m_hbmpOld = (HBITMAP)::SelectObject(m_hdc, m_hbmp);

	// HDCGDĨrbg}bvΏۂƂPFDݒ
	if( !SetPfdForGDIBuffer(m_hdc, num_bits) )
		return false;

	m_hglrc = wglCreateContext( m_hdc );
	if( m_hglrc == NULL )
		return false;

	m_Width = w;
	m_Height = h;
	m_NumPixelBits = num_bits;

	m_Initialized = true;
	return true;
}

//! OpenGL`p̃rbg}bvobt@𐶐
inline HBITMAP GLBitmapRender::CreateBitmapRenderBuffer(int w, int h, int num_bits, const HDC& hdc)
{
	BITMAPINFOHEADER bih;
	memset(&bih, 0, sizeof(BITMAPINFOHEADER));
	bih.biSize = sizeof(BITMAPINFOHEADER);
	bih.biWidth  = w;
	bih.biHeight = h;
	bih.biPlanes = 1;
	bih.biBitCount = num_bits;
	bih.biCompression = BI_RGB;

	void* pbits;
	return CreateDIBSection( hdc, (BITMAPINFO*)&bih, DIB_PAL_COLORS, &pbits, NULL, 0);
}

//! GDĨrbg}bvobt@ɕ`悷邽߂PFDݒ肷
inline bool GLBitmapRender::SetPfdForGDIBuffer(const HDC& hdc, int color_bits)
{
	PIXELFORMATDESCRIPTOR pfd =
	{
		sizeof (PIXELFORMATDESCRIPTOR),  //\̂̃TCY
		1,                               //OpenGL o[W
		PFD_DRAW_TO_BITMAP |             //EBhEX^C
		PFD_SUPPORT_OPENGL |             //OpenGL g
		PFD_SUPPORT_GDI,                 //GDĨobt@g?
		PFD_TYPE_RGBA,                   //sNZ̃J[f[^
		color_bits,                      //F̃rbg
		0, 0,                            //RGBÃrbgƃVtgݒ
		0, 0,                            //G
		0, 0,                            //B
		0, 0,                            //A
		0,                               //AL[Vobt@
		0, 0, 0, 0,                      //RGBAAL[Vobt@
		32,                              //fvXobt@     ̃sNZ̃rbg
		8,                               //XeVobt@ ̃sNZ̃rbg
		0,                               //⏕obt@       ̃sNZ̃rbg
		PFD_MAIN_PLANE,                  //C[^Cv
		0,                               //\ 
		0, 0, 0                          //gp
	};

	int pixelformat = ChoosePixelFormat( hdc , &pfd );
	if( pixelformat == 0 )
		return false;

	if( SetPixelFormat( hdc , pixelformat , &pfd ) == FALSE )
		return false;

	return true;
}


inline void GLBitmapRender::FinalizeRender(void)
{
	if( !m_Initialized )
		return;

	BOOL hglrc_deleted = wglDeleteContext( m_hglrc );

	// HDC̃obt@ɖ߂
	SelectObject(m_hdc, m_hbmpOld);

	BOOL bmp_deleted = DeleteObject(m_hbmp);
	BOOL dc_deleted = DeleteDC(m_hdc);

	assert( hglrc_deleted );
	assert( bmp_deleted   );
	assert( dc_deleted    );

	m_Initialized = false;

	m_Width = 0;
	m_Height = 0;
	m_NumPixelBits = 0;
}


inline void GLBitmapRender::BeginRender(void)
{
	assert( m_Initialized );
	wglMakeCurrent( m_hdc , m_hglrc );
}

inline void GLBitmapRender::EndRender(void)
{
	assert( m_Initialized );
	wglMakeCurrent( NULL , NULL );
}


}
