﻿/**
 * $Revision$
 * $Date$
*/

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Diagnostics;
using System.Text;
using System.Windows.Forms;
using Lugens.Utils;
using System.Drawing.Imaging;
using System.Drawing.Drawing2D;
using System.Windows.Forms.VisualStyles;
using Lugens.Components.IME;

namespace Lugens.Components
{
    public enum ViewMode{ IconText, Text };

    public enum IconListBoxType
    {
        None,
        Alias,
        Command,
        Directory,
        File,
        History,
        Macro,
        Sentence,
        SysCommand,
        SearchEngine,
        SearchEngineText
    }


    public struct IconListBoxItem
    {
        public override string ToString()
        {
            return "";
        }

        private Image iconImage;
        public Image IconImage
        {
            get { return iconImage; }
            set { iconImage = value; }
        }

        private string name;
        public string Name
        {
            get { return name; }
            set { name = value; }
        }

        private string regex;
        public string Regex
        {
            get { return regex; }
            set { regex = value; }
        }

        private string text;
        public string Text
        {
            get { return text; }
            set { text = value; }
        }

        private IconListBoxType type;
        public IconListBoxType Type
        {
            get { return this.type; }
            set { this.type = value; }
        }

        private string toolTipText;
        public string ToolTipText
        {
            get { return toolTipText; }
            set { toolTipText = value; }
        }

        private object data;
        public object Data
        {
            get { return data; }
            set { data = value; }
        }

        public IconListBoxItem(Image iconImage, string text) : this(iconImage, text, null, null, IconListBoxType.None)
        {
        }

        public IconListBoxItem(Image iconImage, string text, object data) : this(iconImage, text, text, data, IconListBoxType.None)
        {
        }

        public IconListBoxItem(Image iconImage, string text, string toolTipText, object data) : this(iconImage, text, toolTipText, data, IconListBoxType.None)
        {
        }

        public IconListBoxItem(Image iconImage, string text, string toolTipText, object data, IconListBoxType type)
        {
            this.iconImage = iconImage;
            this.name = text;
            this.text = text;
            this.toolTipText = toolTipText;
            this.data = data;
            this.type = type;
            this.regex = null;
        }
    
    }

    public class IconListBox2 : ListBox
    {
        public IconListBox2() : base()
        {
            base.DrawMode = DrawMode.OwnerDrawFixed;
        }
    }

    public partial class IconListBox : ListBox
    {
        /// <summary>
        /// 表示ステータス
        /// </summary>
        private int status = 1;
        public int Status
        {
            get
            {
                return status;
            }
            set
            {
                status = value;
            }
        }

        /// <summary>
        /// 表示モード
        /// </summary>
        private ViewMode viewMode;
        public ViewMode ViewMode
        {
            get
            {
                return viewMode;
            }
            set
            {
                viewMode = value;
            }
        }

        private Bitmap backBuffer = null;

        /// <summary>
        /// DnDの区切り線位置
        /// </summary>
        private int dividePoint = 0;
        public int DividePoint
        {
            get { return this.dividePoint; }
        }

        private static Point mousePoint;
        public static Point MousePoint
        {
            get { return mousePoint; }
            set { mousePoint = value; }
        }

        private static ImageList imageList;

        private Bitmap selectedItemImage = null;
        public Bitmap SelectedItemImage
        {
            get { return selectedItemImage; }
        }

        private int preSelectedIndex = -1;
        private int preSelectedIndexBack = -1;

        private Image preSelectedBackgroundImage;
        public Image PreSelectedBackgroundImage
        {
            get { return preSelectedBackgroundImage; }
            set { this.preSelectedBackgroundImage = value; }
        }

        private Image selectedBackgroundImage;
        public Image SelectedBackgroundImage
        {
            get { return selectedBackgroundImage; }
            set { this.selectedBackgroundImage = value; }
        }

        public override string Text
        {
            get
            {
                try
                {
                    object o = this.Items[this.SelectedIndex];
                    if(o.GetType() == typeof(IconListBoxItem)){
                        return ((IconListBoxItem)o).Text;
                    }
                    return o.ToString();
                }
                catch
                {
                    return "";
                }
            }
        }

        public object Data
        {
            get
            {
                try
                {
                    object o = this.Items[this.SelectedIndex];
                    if (o.GetType() == typeof(IconListBoxItem))
                    {
                        return ((IconListBoxItem)o).Data;
                    }
                    return o.ToString();
                }
                catch
                {
                    return "";
                }
            }
        }

        /// <summary>
        /// 表示されるアイテム数
        /// </summary>
        private int showItemCount;
        public int ShowItemCount
        {
            get { return showItemCount; }
        }

        /// <summary>
        /// アイテム検索用
        /// </summary>
        private long lastSearchTime = DateTime.Now.Ticks;
        private bool searchTextSameChar = false;
        private StringBuilder searchText = new StringBuilder();

        /// <summary>
        /// DnD時の垂直スクロール用
        /// </summary>
        private long lastDnDTime = DateTime.Now.Ticks;


        private bool searchItemMode = false;
        public bool SearchItemMode
        {
            get { return searchItemMode; }
            set { searchItemMode = value; }
        }
        
        public static ImeLanguage Ime = new ImeLanguage();


        public IconListBox() : base()
        {
            InitializeComponent();
            base.DrawMode = DrawMode.Normal;
            this.DragOver += OnDragOver;
            this.MouseDown += OnMouseDown;
            this.MouseMove += OnMouseMove;
            this.MouseLeave += OnMouseLeave;
            this.MouseWheel += OnMouseWheel;
            this.MouseHover += OnMouseHover;
            this.SizeChanged += OnSizeChanged;
            this.SelectedIndexChanged += OnSelectedIndexChanged;
            this.showItemCount = this.ClientRectangle.Height / this.ItemHeight;
            this.SetStyle(ControlStyles.UserPaint, true);
            //this.SetStyle(ControlStyles.AllPaintingInWmPaint, true);
            this.DoubleBuffered = true;
        }

        private void OnDragOver(object sender, DragEventArgs e)
        {
            int curY = this.PointToClient(Cursor.Position).Y;
            Win32.ImageList_DragShowNolock(false);
            PrePaint(curY, true);
            Win32.ImageList_DragShowNolock(true);

            long now = DateTime.Now.Ticks;
            if (now - this.lastDnDTime > 500000)
            {
                this.lastDnDTime = now;
                if (curY <= 8 && this.TopIndex > 0)
                    this.TopIndex--;
                else if (this.ClientSize.Height - 8 <= curY && this.TopIndex + this.showItemCount < this.Items.Count)
                    this.TopIndex++;
            }

        }

        private void PrePaint(int y, bool repaintAll)
        {
            this.preSelectedIndex = this.TopIndex + (y / this.ItemHeight);
            int div;

            if (this.preSelectedIndex >= this.Items.Count)
                div = this.Items.Count;
            else
            {
                div = (y / this.ItemHeight);
                if (y % this.ItemHeight > ((this.ItemHeight >> 1)))
                    div++;
            }

            if (this.preSelectedIndex != this.preSelectedIndexBack)
            {
                if (repaintAll)
                {
                    this.Invalidate();
                    this.Update();
                }
                else
                {
                    Rectangle r = new Rectangle();
                    r.X = 0;
                    r.Y = (this.preSelectedIndexBack - this.TopIndex) * this.ItemHeight;
                    r.Width = this.Size.Width;
                    r.Height = this.ItemHeight;
                    this.Invalidate(r);

                    r.X = 0;
                    r.Y = (this.preSelectedIndex - this.TopIndex) * this.ItemHeight;
                    r.Width = this.Size.Width;
                    r.Height = this.ItemHeight;
                    this.Invalidate(r);
                }
                this.preSelectedIndexBack = this.preSelectedIndex;
            }
            else if (this.dividePoint != div)
            {
                this.dividePoint = div;
                this.Invalidate();
                this.Update();
            }
        }

        private void OnMouseHover(object sender, EventArgs e)
        {
            //Point p = this.Parent.PointToScreen(this.Location);
            //Point c = Cursor.Position;
            //if(p.X <= c.X && p.X + this.Size.Width >= c.X && p.Y <= c.Y  && p.Y + this.Size.Height >= c.Y)
            //{
            //    PrePaint(c.Y - p.Y, false);
            //}
        }

        private void OnMouseDown(object sender, MouseEventArgs e)
        {
            int index = this.IndexFromPoint(e.Location);
            if (index < 0)
            {
                this.SelectedIndex = -1;
                return;
            }
        }
        
        private void OnMouseMove(object sender, MouseEventArgs e)
        {
            PrePaint(e.Y, false);
        }

        private void OnMouseWheel(object sender, MouseEventArgs e)
        {
            PrePaint(e.Y, false);
        }

        private void OnMouseLeave(object sender, EventArgs e)
        {
            Rectangle r = new Rectangle();
            r.X = 0;
            r.Y = (this.preSelectedIndexBack - this.TopIndex) * this.ItemHeight;
            r.Width = this.Size.Width;
            r.Height = this.ItemHeight;
            this.Invalidate(r);
            this.preSelectedIndex = this.preSelectedIndexBack = -1;
        }

        private void DrawBackground(Graphics g, DrawItemEventArgs e, Image image)
        {
            if (image == null)
                return;
            Rectangle srcRect = new Rectangle();
            Rectangle destRect = new Rectangle();
            srcRect.X = 0;
            srcRect.Y = 0;
            srcRect.Width = 3;
            srcRect.Height = image.Height;
            destRect.X = 0;
            destRect.Y = 0;
            destRect.Width = 3;
            destRect.Height = e.Bounds.Height;
            g.DrawImage(image, destRect, srcRect, GraphicsUnit.Pixel);

            srcRect.X = 3;
            srcRect.Y = 0;
            srcRect.Width = 1;
            srcRect.Height = image.Height;
            destRect.X = 3;
            destRect.Y = 0;
            destRect.Width = e.Bounds.Width - 6;
            destRect.Height = e.Bounds.Height;
            g.DrawImage(image, destRect, srcRect, GraphicsUnit.Pixel);

            srcRect.X = 4;
            srcRect.Y = 0;
            srcRect.Width = 3;
            srcRect.Height = image.Height;
            destRect.X = e.Bounds.Width - 3;
            destRect.Y = 0;
            destRect.Width = 3;
            destRect.Height = e.Bounds.Height;
            g.DrawImage(image, destRect, srcRect, GraphicsUnit.Pixel);
        }

        private void OnSelectedIndexChanged(object sender, EventArgs e)
        {
            if (this.SelectedIndex == -1)
                this.preSelectedIndex = -1;
            //PrePaint(this.PointToClient(Cursor.Position).Y, false);
        }

        private void OnSizeChanged(object sender, EventArgs e)
        {
            this.showItemCount = this.ClientRectangle.Height / this.ItemHeight;
        }

        protected override void OnPaint(PaintEventArgs e)
        {
            DrawItemEventArgs ea;
            Rectangle r = new Rectangle();
            for (int i = this.TopIndex; (i < this.Items.Count) && (i < this.TopIndex + this.showItemCount); i++)
            {
                r.X = 0;
                r.Y = (i - this.TopIndex) * this.ItemHeight;
                r.Width = this.ClientSize.Width;
                r.Height = this.ItemHeight;

                ea = new DrawItemEventArgs(e.Graphics, this.Font, r, i, i == this.SelectedIndex ? DrawItemState.Selected : DrawItemState.None);
                OnDrawItem(ea);
            }

            //DnDの線
            if (this.status == 2 && this.dividePoint >= 0 && Control.MouseButtons == MouseButtons.Left)
            {
                int y = this.dividePoint * this.ItemHeight;
                int x = this.ClientSize.Width;
                int x2 = (x >> 1) + (x & 0x01);

                using (Pen pen = new Pen(new LinearGradientBrush(new Point(0, 0), new Point(x2, 0), Color.LightGray, Color.Black)))
                //using(Pen pen = new Pen(new SolidBrush(Color.Green)))
                {
                    e.Graphics.DrawLine(pen, 0, y - 1, x2, y - 1);
                    e.Graphics.DrawLine(pen, 0, y, x2, y);
                }
                using (Pen pen = new Pen(new LinearGradientBrush(new Point(0, 0), new Point(x - x2 - 1, 0), Color.Black, Color.LightGray)))
                //using (Pen pen = new Pen(new SolidBrush(Color.Gray)))
                {
                    e.Graphics.DrawLine(pen, x2, y - 1, x, y - 1);
                    e.Graphics.DrawLine(pen, x2, y, x, y);
                }
                using (Pen pen = new Pen(Color.FromArgb(211, 211, 211)))
                {
                    e.Graphics.DrawLine(pen, 0, y - 3, 0, y + 2);
                    e.Graphics.DrawLine(pen, x - 1, y - 3, x - 1, y + 2);
                }
                using (Pen pen = new Pen(Color.FromArgb(208, 208, 208)))
                {
                    e.Graphics.DrawLine(pen, 1, y - 2, 1, y + 1);
                    e.Graphics.DrawLine(pen, x - 2, y - 2, x - 2, y + 1);
                }

            }
        }

        public void BeginDragDrop(MouseEventArgs e)
        {
            if (this.SelectedIndex < 0)
                return;

            int itemIndex = this.IndexFromPoint(IconListBox.mousePoint);
            Rectangle rect = this.GetItemRectangle(itemIndex);

            IconListBox.imageList = new ImageList();
            IconListBox.imageList.ImageSize = this.SelectedItemImage.Size;
            IconListBox.imageList.Images.Add(this.SelectedItemImage);
            Win32.ImageList_BeginDrag(IconListBox.imageList.Handle, 0, e.X - rect.Left, e.Y - rect.Top);

            this.Status = 2;
            this.DoDragDrop(this.SelectedIndex, DragDropEffects.Move);

            Win32.ImageList_EndDrag();
        }

        protected override void OnDrawItem(DrawItemEventArgs e)
        {
            if (Items == null || Items.Count == 0 || e.Index < 0 || (e.Index - this.TopIndex) > this.showItemCount || e.Index >= Items.Count)
                return;

            if (this.backBuffer == null)
            {
                this.backBuffer = new Bitmap(this.Size.Width, this.ItemHeight, PixelFormat.Format32bppArgb);
            }
            else if (this.backBuffer.Size != this.Size)
            {
                this.backBuffer.Dispose();
                this.backBuffer = new Bitmap(this.Size.Width, this.ItemHeight, PixelFormat.Format32bppArgb);
            }

            using (Graphics gBack = Graphics.FromImage(this.backBuffer))
            {
                int x = e.Bounds.X;
                Image iconImage = null;
                string text = "";

                object o = Items[e.Index];
                if (ViewMode.IconText == this.viewMode)
                {
                    if (o.GetType() == typeof(IconListBoxItem))
                    {
                        iconImage = ((IconListBoxItem)o).IconImage;
                        text = ((IconListBoxItem)o).Text;
                    }
                    x += 18;
                }
                else
                {
                    if (o.GetType() == typeof(IconListBoxItem))
                        text = ((IconListBoxItem)o).Text;
                    else
                        text = o.ToString();
                }

                Rectangle destRect = new Rectangle();
                if ((e.State & DrawItemState.Selected) == DrawItemState.Selected)
                {
                    this.DrawBackground(gBack, e, this.selectedBackgroundImage);
                }
                else if (e.Index == this.preSelectedIndex && ((this.status == 0 || this.status == 1) || (this.status == 2 && Control.MouseButtons == MouseButtons.None)))
                {
                    this.DrawBackground(gBack, e, this.preSelectedBackgroundImage);
                }
                else
                {
                    gBack.Clear(SystemColors.Window);
                }

                if (this.ViewMode == ViewMode.IconText)
                {
                    if (iconImage != null)
                        gBack.DrawImage(iconImage, new Rectangle(1, 1, 16, 16));

                    destRect = new Rectangle(18, 0, e.Bounds.Width - 19, e.Bounds.Height);
                    using (SolidBrush brush = new SolidBrush(SystemColors.WindowText))
                    {
                        //TextRenderer.DrawText(gBack, text, this.Font, new Point(18, 1), brush.Color, Color.Transparent);
                        gBack.DrawString(text, this.Font, brush, 18, 1, StringFormat.GenericDefault);
                    }
                }
                else
                {
                    destRect = new Rectangle(1, 1, e.Bounds.Width, e.Bounds.Height);
                    using (SolidBrush brush = new SolidBrush(SystemColors.WindowText))
                    {
                        //TextRenderer.DrawText(gBack, text, this.Font, new Point(1, 1), brush.Color, TextFormatFlags.NoPrefix);
                        gBack.DrawString(text, this.Font, brush, 1, 1, StringFormat.GenericDefault);
                    }
                }

                e.Graphics.DrawImage(this.backBuffer, 0, e.Bounds.Y);

                if ((e.State & DrawItemState.Selected) == DrawItemState.Selected)
                {
                    if (this.selectedItemImage == null)
                    {
                        this.selectedItemImage = new Bitmap(e.Bounds.Width, e.Bounds.Height, PixelFormat.Format32bppArgb);
                    }
                    else if (this.selectedItemImage.Width != e.Bounds.Width || this.selectedItemImage.Height != e.Bounds.Height)
                    {
                        this.selectedItemImage.Dispose();
                        this.selectedItemImage = new Bitmap(e.Bounds.Width, e.Bounds.Height, PixelFormat.Format32bppArgb);
                    }
                    using (Graphics gSel = Graphics.FromImage(this.selectedItemImage))
                    {
                        gSel.Clear(Color.Transparent);
                        if (this.ViewMode == ViewMode.IconText)
                        {
                            if (iconImage != null)
                                gSel.DrawImage(iconImage, new Rectangle(1, 1, 16, 16));

                            destRect = new Rectangle(18, 0, e.Bounds.Width - 19, e.Bounds.Height);
                            using (SolidBrush brush = new SolidBrush(Color.FromArgb(192, SystemColors.WindowText)))
                            {
                                gSel.SmoothingMode = System.Drawing.Drawing2D.SmoothingMode.AntiAlias;
                                gSel.TextRenderingHint = System.Drawing.Text.TextRenderingHint.SingleBitPerPixelGridFit;
                                //TextRenderer.DrawText(gSel, text, this.Font, new Point(18, 0), brush.Color, TextFormatFlags.NoPrefix);
                                gSel.DrawString(text, this.Font, brush, 18, 0);
                            }
                        }
                        else
                        {
                            destRect = new Rectangle(1, 1, e.Bounds.Width, e.Bounds.Height);
                            //TextRenderer.DrawText(gBack, text, this.Font, destRect, SystemColors.WindowText, SystemColors.Window, TextFormatFlags.Left);
                        }

                        //gSel.DrawImage(this.backBuffer, 0, 0);
                    }
                }
            }
        }

        public void SearchItem(char c)
        {
            if (DateTime.Now.Ticks - lastSearchTime > 10000000)
            {
                searchText.Length = 0;
                searchTextSameChar = false;
            }

            lastSearchTime = DateTime.Now.Ticks;
            searchText.Append(c);

            if (!searchTextSameChar && searchText.Length > 1)
                searchTextSameChar = searchText[searchText.Length - 1] != searchText[searchText.Length - 2];

            IconListBoxItem item;
            if (searchTextSameChar)
            {
                //検索文字が違う
                for (int i = this.SelectedIndex; i < this.Items.Count; i++)
                {
                    item = (IconListBoxItem)this.Items[i];
                    if (Ime.StartsWith(item.Regex, searchText.ToString()))
                    {
                        this.SelectedIndex = i;
                        return;
                    }
                }
                if (this.SelectedIndex > 0)
                {
                    for (int i = 0; i < this.SelectedIndex - 1; i++)
                    {
                        item = (IconListBoxItem)this.Items[i];
                        if (Ime.StartsWith(item.Regex, searchText.ToString()))
                        {
                            this.SelectedIndex = i;
                            this.preSelectedIndex = -1;
                            return;
                        }
                    }
                }
            }
            else
            {
                //検索文字が全て同じ
                for (int i = this.SelectedIndex + 1; i < this.Items.Count; i++)
                {
                    item = (IconListBoxItem)this.Items[i];
                    if (Ime.StartsWith(item.Regex, searchText[searchText.Length - 1].ToString()))
                    {
                        this.SelectedIndex = i;
                        this.preSelectedIndex = -1;
                        return;
                    }
                }
                if (this.SelectedIndex > -1)
                {
                    for (int i = 0; i < this.SelectedIndex; i++)
                    {
                        item = (IconListBoxItem)this.Items[i];
                        if (Ime.StartsWith(item.Regex, searchText[searchText.Length - 1].ToString()))
                        {
                            this.SelectedIndex = i;
                            this.preSelectedIndex = -1;
                            return;
                        }
                    }
                }
            }
        }

        protected override void OnVisibleChanged(EventArgs e)
        {
            base.OnVisibleChanged(e);
            if (!this.Visible)
            {
                searchText.Length = 0;
                searchTextSameChar = false;
            }
        }

        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                case 0x115: //WM_VSCROLL
                    Win32.PostMessage(this.Parent.Handle, m.Msg, m.WParam, m.LParam);
                    break;

                case 0x0102: //WM_CHAR
                    if (this.searchItemMode && this.Items.Count > 0)
                        this.SearchItem((char)m.WParam);
                    break;
            }
            base.WndProc(ref m);
        }
    }
}
