package jp.sf.pal.webparts;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.collections.map.LRUMap;
import org.apache.commons.lang.StringUtils;

public class ProxyServlet extends HttpServlet {
    private static final String NEW_VALUE = "new";

    private static final String OLD_VALUE = "old";

    private static final long serialVersionUID = -5314394726940661335L;

    private static final int BLOCK_SIZE = 4096;

    private static final int DEFAULT_CACHE_SIZE = 20;

    private static final long DEFAULT_CACHE_TIMEOUT = 60 * 60 * 1000; // 1 hour

    private static final String CONTENT_TYPE = "Content-Type";

    private static final String DEFAULT_HEADER_NAMES = "Content-Type";

    public static final String URL_KEY = "url";

    public static final String MAX_CACHE_SIZE_KEY = "maxCacheSize";

    public static final String CACHE_TIMEOUT_KEY = "cacheTimeout";

    public static final String HEADER_NAMES_KEY = "headerNames";

    public static final String ACCEPT_URLS_KEY = "acceptUrls";

    public static final String REPLACE_CONTEXT_TYPE_LIST_KEY = "replaceContextTypeList";

    protected long cacheTimeout = DEFAULT_CACHE_TIMEOUT;

    protected int maxCacheSize = DEFAULT_CACHE_SIZE;

    protected String[] headerNames;

    protected String[] acceptUrls = null;

    protected List replaceContextTypeList = null;

    protected LRUMap contentCacheMap;

    public void init() throws ServletException {
        String cacheTimeoutValue = getServletConfig().getInitParameter(
                CACHE_TIMEOUT_KEY);
        if (cacheTimeoutValue != null) {
            try {
                cacheTimeout = Long.parseLong(cacheTimeoutValue);
            } catch (NumberFormatException e) {
            }
        }

        String maxCacheSizeValue = getServletConfig().getInitParameter(
                MAX_CACHE_SIZE_KEY);
        if (maxCacheSizeValue != null) {
            try {
                maxCacheSize = Integer.parseInt(maxCacheSizeValue);
            } catch (NumberFormatException e) {
            }
        }

        String headerNamesValue = getServletConfig().getInitParameter(
                HEADER_NAMES_KEY);
        if (headerNamesValue == null) {
            headerNamesValue = DEFAULT_HEADER_NAMES;
        }
        headerNames = headerNamesValue.split(",");

        String acceptUrlsValue = getServletConfig().getInitParameter(
                ACCEPT_URLS_KEY);
        if (acceptUrlsValue != null) {
            acceptUrls = acceptUrlsValue.split(",");
        }

        String replacedContextTypeListValue = getServletConfig()
                .getInitParameter(REPLACE_CONTEXT_TYPE_LIST_KEY);
        if (replacedContextTypeListValue != null) {
            replaceContextTypeList = new ArrayList();
            String[] list = replacedContextTypeListValue.split(",");
            for (int i = 0; i < list.length; i++) {
                String[] pair = list[i].split("=");
                if (pair.length == 2) {
                    Map map = new HashMap(2);
                    map.put(OLD_VALUE, pair[0]);
                    map.put(NEW_VALUE, pair[1]);
                    replaceContextTypeList.add(map);
                }
            }
        }

        contentCacheMap = new LRUMap(maxCacheSize);
    }

    public void destroy() {
        contentCacheMap = null;
    }

    protected void doGet(HttpServletRequest req, HttpServletResponse resp)
            throws ServletException, IOException {
        String u = req.getParameter(URL_KEY);
        if (u == null) {
            resp.sendError(HttpServletResponse.SC_BAD_REQUEST, "url is null.");
            return;
        }

        // check url
        if (acceptUrls != null) {
            boolean valid = false;
            for (int i = 0; i < acceptUrls.length; i++) {
                if (u.startsWith(acceptUrls[i])) {
                    valid = true;
                }
            }
            if (!valid) {
                resp.sendError(HttpServletResponse.SC_BAD_REQUEST,
                        "url is not accepted.");
                return;
            }
        }

        ContentCache contentCache = getContentCache(u);
        if (contentCache == null) {
            // load a content
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            try {
                URL url = new URL(u);
                URLConnection uc = url.openConnection();
                // store a content to a byte array
                drain(uc.getInputStream(), baos);
                // copy header fields
                Map headers = new HashMap();
                Map headerFields = uc.getHeaderFields();
                for (int i = 0; i < headerNames.length; i++) {
                    if (CONTENT_TYPE.equals(headerNames[i])
                            && replaceContextTypeList != null) {
                        Collection value = (Collection) headerFields
                                .get(headerNames[i]);
                        // replace content type
                        for (int j = 0; j < replaceContextTypeList.size(); j++) {
                            Map map = (Map) replaceContextTypeList.get(j);
                            String contentType = uc.getContentType();
                            if (contentType != null) {
                                contentType = StringUtils.replace(contentType,
                                        (String) map.get(OLD_VALUE),
                                        (String) map.get(NEW_VALUE));
                                value = new ArrayList(1);
                                value.add(contentType);
                            }
                        }
                        headers.put(headerNames[i], value);
                    } else {
                        headers.put(headerNames[i], headerFields
                                .get(headerNames[i]));
                    }
                }
                contentCache = new ContentCache(headers, baos.toByteArray(),
                        System.currentTimeMillis());
                contentCacheMap.put(u, contentCache);
            } catch (Exception e) {
                resp.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR, e
                        .getMessage());
                return;
            }
        }

        // store headers
        Iterator ite = contentCache.getHeaders().entrySet().iterator();
        while (ite.hasNext()) {
            Map.Entry entry = (Map.Entry) ite.next();
            if (entry.getValue() instanceof Collection) {
                resp.addHeader((String) entry.getKey(), StringUtils.join(
                        (Collection) entry.getValue(), ","));
            }
        }

        // store a content
        resp.getOutputStream().write(contentCache.getContent());
        resp.flushBuffer();

    }

    protected void doPost(HttpServletRequest req, HttpServletResponse resp)
            throws ServletException, IOException {
        doGet(req, resp);
    }

    protected ContentCache getContentCache(String url) {
        ContentCache cache = (ContentCache) contentCacheMap.get(url);
        if (cache != null) {
            if (System.currentTimeMillis() - cache.getCreatedTime() > cacheTimeout) {
                // timeout
                return null;
            }
            return cache;
        }
        return null;
    }

    public static void drain(InputStream r, OutputStream w) throws IOException {
        byte[] bytes = new byte[BLOCK_SIZE];
        try {
            int length = r.read(bytes);
            while (length != -1) {
                if (length != 0) {
                    w.write(bytes, 0, length);
                }
                length = r.read(bytes);
            }
        } finally {
            bytes = null;
        }
    }

    private static class ContentCache {
        private byte[] content;

        private long createdTime;

        private Map headers;

        public ContentCache(Map headers, byte[] content, long createdTime) {
            this.headers = headers;
            this.content = content;
            this.createdTime = createdTime;
        }

        public Map getHeaders() {
            return headers;
        }

        public void setHeaders(Map headers) {
            this.headers = headers;
        }

        public byte[] getContent() {
            return content;
        }

        public void setContent(byte[] content) {
            this.content = content;
        }

        public long getCreatedTime() {
            return createdTime;
        }

        public void setCreatedTime(long createdTime) {
            this.createdTime = createdTime;
        }

    }

}
