<?php

require_once "Date.php";
require_once "Date/Calc.php";
require_once "Date/Span.php";
require_once "Date/TimeZone.php";
Zend_Loader::loadClass("CFW_Util_String");
Zend_Loader::loadClass("CFW_Constants_Regex");
Zend_Loader::loadClass("CFW_Constants_DateTime");
Zend_Loader::loadClass("Zend_Date");
/**
 * 日付関係ユーティリティ
 * @author okada
 * @package CFW_Util
 */
class CFW_Util_Date{

	/**
	 * 年齢を月表示にするための年数
	 * @var integer
	 */
	public static $yearsForAgeByMonth = 2;

	/**
	 * 曜日インデックスの開始
	 * @var integer
	 */
	public static $weekDayBase = 0;

	/**
	 * US月表示を大文字にする
	 * @var bool
	 */
	public static $UseUpperCase = true;

	/**
	 * 年を示唆するための日数
	 * @var integer
	 */
	public static $DaysToSuggestYear = -3;

	/**
	 * Dateが期待される値をDateTimeに
	 * @param $value 変換する対象の文字列型の式
	 * @return Date 変換した日付
	 */
	public static function toDate($value){
	    //nullはnullのまま
		if ($value == null) return null;
		//DateTimeならそのままで良い
        if ($value instanceof DateTime) return $value;
        if($value instanceof Date){
            $date = new DateTime($value->format("%Y/%m/%d %H:%M:%S"));
            return $date;
        }
        if($value instanceof Zend_Date){
            $date = new DateTime($value->toString("yyyy/MM/dd HH:mm:ss"));
            return $date;
        }

        $stringValue = CFW_Util_String::normalize( CFW_Util_String::toString($value) );

		if ($stringValue == "") {
			return null;
		}
		return self::parse($stringValue);
	}
	/**
	 * 今日(時刻を含まない= 0時0分)
	 * @return  今日(時刻を含まない= 0時0分)
	 */
	public static function today(){
		$now= new DateTime();

		$result = self::truncateTime($now);
		return $result;
	}

	/**
	 * 現在(時刻を含む)
	 * @return 現在(時刻を含む)
	 */
	public static function now(){
		$result = new DateTime();
		return $result;
	}

	/**
	 * 年推論の基準にする日付
	 * @return object 年推論の基準にする日付
	 */
	public static function dateToSuggestYear(){
		return self::addDays(self::today(), self::$DaysToSuggestYear);
	}
	/**
	 * 年推論
	 * @param object $d
	 * @return object 年推論の基準にする日付以降でもっとも現在に近い日付
	 */
	public static function inferYear($d){
		$d = new Date($d);
		if ($d->compare($d, self::dateToSuggestYear()) < 0) {
			return self::addYears($d, 1);
		}
		return $d;
	}

	/**
	 * Date型をString型にして出力
	 * @param datetime $date 変換対象DateTime型
	 * @param string $formatString フォーマット方式
	 * @return string 変換したString型
	 */
	public static function format($date, $formatString = CFW_Constants_DateTime::FORMAT_DATE_TIME){
	    if($date == null) return "";
        $target = self::toDate($date);
        $result = $target->format($formatString);
        return $result;
	}

	/**
	 * 年、月、日の要素から日付生成
	 * @param integer $y 年をあらわす数値
	 * @param integer $m 月をあらわす数値
	 * @param integer $d 日をあらわす数値
	 * @return 指定年月日から成る日付
	 */
	public static function create($y, $m , $d){

		$y = CFW_Util_Number::formatZero($y, 4);
		$m = CFW_Util_Number::formatZero($m, 2);
		$d = CFW_Util_Number::formatZero($d, 2);
		return new DateTime($y."/".$m."/".$d);
	}
	/**
	 * 年、月の要素から日付生成
	 * 日は1日を仮定
	 * @param integer $y 年をあらわす数値
	 * @param integer $m 月をあらわす数値
	 * @return 指定年月＋1日の日付
	 */
	public static function createFromYearMonth($y, $m){
		$y = CFW_Util_Number::formatZero($y, 4);
		$m = CFW_Util_Number::formatZero($m, 2);
		return self::create($y,$m,1);
	}
	/**
	 * 月日の要素から日付生成
	 * 年は基準日以降の最初の年を仮定
	 * @param integer $m 月をあらわす数値
	 * @param integer $d 日をあらわす数値
	 * @return object 指定月日でかつ基準日以降の日付
	 */
	public static function createFromMonthDay($m, $d){
		$m = CFW_Util_Number::formatZero($m, 2);
		$d = CFW_Util_Number::formatZero($d, 2);
		//とりあえず現在の年を取得して使ってみる
		$now = self::today();
		$year =  intval($now->format("yyyy"));
		$newDate = self::create($year.$m.$d);

		return self::inferYear($newDate);
	}

	/**
	 * 時刻を捨てる
	 * @param object $object
	 * @return 指定日付の0時0分を表す日付
	 */
	public static function truncateTime($object){
		if ($object == null) return null;

		return new DateTime($object->format(CFW_Constants_DateTime::FORMAT_DATE_TIME));
	}
	 /**
	  * 日数計算
	  * @param object $d1 基準日
	  * @param object $d2 比較対象日
	  * @return $intDiff 基準日から比較対象日までの日数
	  */
	public static function diffDays($d1, $d2){
        if($d1 == null) return null;
        if($d2 == null) return null;

        //pear::dateを利用する
		$standardDay = new Date($d1->format("Ymdhis"));
		$comparisonDay = new Date($d2->format("Ymdhis"));

		$span = new Date_Calc();
		$intDiff = $span->dateToDays($comparisonDay->getDay(),$comparisonDay->getMonth(),$comparisonDay->getYear()) -
							$span->dateToDays($standardDay->getDay(),$standardDay->getMonth(),$standardDay->getYear());

		return $intDiff;
	}

	/**
	 * 年齢
	 * @param object $birthday 誕生日
	 * @param object $baseDate 基準日
	 * @return integer 年齢
	 */
	public static function ageYear($birthday, $baseDate){
		if ($birthday == null) return 0;
		if ($baseDate == null) return 0;
		$d = self::toDate($birthday);
		$x = self::toDate($baseDate);

		//計算式: 基準日の年 - 誕生日の年 - (基準日の月日が誕生日の月日を過ぎてなければ1)
		return self::year($x) - self::year($d) -
				((( self::month( $d ) * 100 + self::day($d)) > (self::month( $x ) * 100 + self::day($x))) ? 1 : 0);
	}

	/**
	 * 年齢(月)
	 * @param object $birthday 誕生日
	 * @param object $baseDate 基準日
	 * @return integer 年齢(月単位)
	 */
	public static function ageMonth($birthday, $baseDate){
		if ($birthday == null) return 0;
		if ($baseDate == null) return 0;
        $d = self::toDate($birthday);
        $x = self::toDate($baseDate);
		return ((self::year($x) - self::year($d)) * 12) + self::month($x) - self::month($d) -
				((self::day($d) > self::day($x)) ? 1 : 0);
	}

	/**
	 * 基準日と生年月日から99Y（24ヶ月以上）または99M（２４ヶ月未満）の文字列を得る
	 * @param object $birthday 誕生日
	 * @param object $baseDate 基準日
	 * @return string
	 */
	public static function ageString($birthday, $baseDate){
		$result = "";
		$age = self::ageYear($birthday, $baseDate);
		if ($age < self::$yearsForAgeByMonth) {
			$result = strval(self::ageMonth($birthday, $baseDate))."M";
		} else {
			$result = strval($age)."Y";
		}
		return $result;
	}

	/**
	 * 基準日と生年月日から99歳または99ヶ月（２４ヶ月未満）の文字列を得る
	 * @param object $birthday 誕生日
	 * @param object $baseDate 基準日
	 * @return string
	 */
	public static function ageStringJP($birthday, $baseDate){
		$result = "";
		$age = self::ageYear($birthday, $baseDate);
		if ($age < self::$yearsForAgeByMonth) {
			$result = strval(self::ageMonth($birthday, $baseDate))."ヶ月";
		} else {
			$result = strval($age)."歳";
		}
		return $result;
	}

	/**
	 * 曜日のインデックス(weekDayBaseから始まる).
	 * 曜日の基準(weekDayBase)により1-7または0-6
	 * @param object $object 指定日付
	 * @return integer 曜日のインデックス.曜日の基準(weekDayBase)により1-7または0-6
	 */
	public static function dayOfWeekIndex($object){
		if ($object == null) return self::$weekDayBase - 1;
		$date = self::toDate($object);

		$index = self::dayOfWeek($date);
		return $index + self::$weekDayBase;
	}

	/**
	 * 曜日文字列
	 * @param object $object 指定日付
	 * @return string 指定日付の曜日
	 */
	public static function dayOfWeekString($object){
		if ($object == null) return "";
		$date = self::toDate($object);
		$result = $date->format("D");
		return (self::$UseUpperCase) ? mb_strtoupper($result) : $result;
	}

	/**
	 * 曜日文字列
	 * @param object $object 指定日付
	 * @return string 指定日付の曜日
	 */
	public static function dayOfWeekStringJP($object){
		if ($object == null) return "";
		$weekDay = array("日","月","火","水","木","金","土");
		$date = self::toDate($object);
		$result = $weekDay[self::dayOfWeek($date)];
		return $result;

	}

	/**
	 * 範囲の正当性を検証
	 * @param object $object 検証する対象
	 * @param object $min 範囲(最小値)
	 * @param object $max 範囲(最大値)
	 * @return boolean 範囲が正当ならばtrue,でなければfalse
	 */
	public static function isInRange($object, $min, $max){
		if ($object == null) return false;
		if (($min == null) && ($max == null)) return false;
        //PEAR::Dateを利用する
		$target = new Date(self::toDate($object)->format("Ymdhis"));
		$maxDate = new Date(self::toDate($max)->format("Ymdhis"));
		$minDate = new Date(self::toDate($min)->format("Ymdhis"));
		if ($min == null) {
			if ($target->compare($target,$maxDate) > 0) return false;
		}
		elseif ($max == null) {
			if ($target->compare($target,$minDate) < 0) return false;
		}
		else {
			if ($target->compare($target,$minDate) < 0) return false;
			if ($target->compare($target,$maxDate) > 0) return false;
		}
		return true;
	}

	/**
	 * 指定日より前か
	 * @param object $d1 対象日
	 * @param object $d2 基準日
	 * @return boolean 対象日が基準日より前ならtrue
	 */
	public static function isBefore($d1, $d2){
		if ($d1 == null) { return false; }
		if ($d2 == null) { return false; }
		$objectDate = new Date(self::toDate($d1)->format("Ymdhis"));
		$baseDate = new Date(self::toDate($d2)->format("Ymdhis"));
		if ($objectDate->compare($objectDate, $baseDate) < 0){
			return true;
		}
		else {
			return false;
		}
	}
	/**
	 * 指定日より前か同じか
	 * @param object $d1 対象日
	 * @param object $d2 基準日
	 * @return boolean 対象日が基準日より前か同じならtrue
	 */
	public static function isBeforeOrSame($d1, $d2){
		if ($d1 == null) { return false; }
		if ($d2 == null) { return false; }
        $objectDate = new Date(self::toDate($d1)->format("Ymdhis"));
        $baseDate = new Date(self::toDate($d2)->format("Ymdhis"));
		if ($objectDate->compare($objectDate, $baseDate) <= 0){
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * 指定日より後か
	 * @param object $d1 対象日
	 * @param object $d2 基準日
	 * @return boolean 対象日が基準日より後ならtrue
	 */
	public static function isAfter($d1, $d2){
		if ($d1 == null) { return false; }
		if ($d2 == null) { return false; }
        $objectDate = new Date(self::toDate($d1)->format("Ymdhis"));
        $baseDate = new Date(self::toDate($d2)->format("Ymdhis"));
		if ($objectDate->compare($objectDate, $baseDate) > 0){
			return true;
		}
		else {
			return false;
		}
	}
	/**
	 * 指定日より後か同じか
	 * @param object $d1 対象日
	 * @param object $d2 基準日
	 * @return boolean 対象日が基準日より後か同じならtrue
	 */
	public static function isAfterOrSame($d1, $d2){
		if ($d1 == null) { return false; }
		if ($d2 == null) { return false; }
        $objectDate = new Date(self::toDate($d1)->format("Ymdhis"));
        $baseDate = new Date(self::toDate($d2)->format("Ymdhis"));
		if ($objectDate->compare($objectDate, $baseDate) >= 0){
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * 指定日と同じか
	 * @param object $d1 対象日
	 * @param object $d2 基準日
	 * @return boolean 対象日が基準日と同じならtrue
	 */
	public static function isSame($d1, $d2){
		if ($d1 == null) { return false; }
		if ($d2 == null) { return false; }
        $objectDate = new Date(self::toDate($d1)->format("Ymdhis"));
        $baseDate = new Date(self::toDate($d2)->format("Ymdhis"));
		if ($objectDate->compare($objectDate, $baseDate) == 0){
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * 文字列を解析して日付型に変換
	 * @param string $s 日付形式の文字列
	 * @return object 解析結果の日付。解析エラーの場合null
	 */
	public static function parse($s){
		if ($s == "") return null;
		$date = $s;
		//区切りあり
        //yyyy/M/d H:m:s
        if (mb_ereg_match(CFW_Constants_Regex::DATETIME_LONG, $s)) {
        	$date = $year.$month.$day.$hour.$minute.$second;
        	return new Date($date);
        }
        //yyyy/M/d H:m
        if (mb_ereg_match(CFW_Constants_Regex::DATETIME_SMALL, $s)) {
            return new DateTime($s);
        }
        //yyyy/M/d
        if (mb_ereg_match(CFW_Constants_Regex::DATE_LONG, $s)) {
            return new DateTime($s);
        }
        //yy/M/d
        if (mb_ereg_match(CFW_Constants_Regex::DATE_SHORT, $s)) {
            return new DateTime($s);
        }
        //区切りなし
        //yyyyMMddHHmmss
        if (mb_ereg_match(CFW_Constants_Regex::DATETIME_LONG_NO_DELIMITER, $s)) {
            return new DateTime(
                substr($s,0,4)."/".substr($s,4,2)."/".substr($s,6,2)." ".
                substr($s,8,2).":".substr($s,10,2).":".substr($s,12,2)
            );
        }
        //yyyyMMddHHmm
        if (mb_ereg_match(CFW_Constants_Regex::DATETIME_SMALL_NO_DELIMITER, $s)) {
            return new DateTime($s);
        }
        //yyyyMMdd
        if (mb_ereg_match(CFW_Constants_Regex::DATE_LONG_NO_DELIMITER, $s)) {
            return new DateTime(
                substr($s,0,4)."/".substr($s,4,2)."/".substr($s,6,2)
            );
        }
        //yyMMdd
        if (mb_ereg_match(CFW_Constants_Regex::DATE_SHORT_NO_DELIMITER, $s)) {
            return new DateTime(
                substr($s,0,2)."/".substr($s,2,2)."/".substr($s,4,2)
            );
        }
        return null;
	}

	/**
	 * 日付計算
	 * @param object $date 計算の基準日
	 * @param integer $diff 加算する日数、月数、年数
	 * @param string $unit 単位をあらわす文字 y,m,d
	 * @return object 加算結果
	 */
	public static function addDate($date, $diff, $unit){
	    if($date == null) return null;
		$d = new Zend_Date($date->format("Y/m/d h:i:s"));
		if (mb_strtolower($unit) == "y") {
		    $d->add($diff,Zend_Date::YEAR);
		}
		elseif (mb_strtolower($unit) == "m") {
            $d->add($diff,Zend_Date::MONTH);
		}
		elseif (mb_strtolower($unit) == "d") {
            $d->add($diff,Zend_Date::DAY);
		}

		return new DateTime($d->toString("yyyy/MM/dd HH:mm:ss"));
	}

	/**
	 * 何ヶ月か後
	 * @param integer $year 年
	 * @param integer $month 月
	 * @param integer $after 加算する月数
	 * @return object 指定年月のafterヶ月後の月初
	 */
	public static function monthAfter($year, $month, $after){
		if ($year < 0) return null;
		if ($month < 1) return null;
		if ($month > 12) return null;

		$baseDate = self::create($year, $month, 1);
		return self::addDate($baseDate, $after, "m");
	}
	/**
	 * 何ヶ月か後の月末
	 * @param integer $year 年
	 * @param integer $month 月
	 * @param integer $after 加算する月数
	 * @return object 指定年月のafterヶ月後の月末
	 */
	public static function endOfMonthAfter($year, $month, $after){
		return self::endOfMonth(self::monthAfter($year, $month, $after));
	}
	/**
	 * 指定日付の月の月末
	 * @param object $d 指定日付
	 * @return object 指定日付の月の月末
	 */
	public static function endOfMonth($d){
		if ($d == null) return null;
		$next = self::addDate($d, 1, "m");
		$days = self::day( $next );
		return self::addDate($next, -$days, "d");
	}
	/**
	 * 指定月の月末
	 * @param integer $year 指定年
	 * @param integer $month 指定月
	 * @return object 指定年月の月末
	 */
	public static function endOfYearMonth($year, $month){
		$next = self::addDate(self::createFromYearMonth($year, $month), 1, "m");
		$days = self::day($next);
		return self::addDate($next, -$days, "d");
	}
	/**
	 * 月末チェック
	 * @param object $d 指定日付
	 * @return boolean 月末ならtrue
	 */
	public static function isEndOfMonth($d){
		if ($d == null) return false;
		$date = self::toDate($d);
		$end = self::endOfMonth($date);
		return self::isSame($date, $end);
	}

	/**
	 * 日数加算
	 * @param object $d 日付
	 * @param integer $days 加算日数
	 * @return object 加算後の日付
	 */
	public static function addDays($d, $days){
		if ($d == null) return null;
		$next = self::addDate($d, $days, "d");

		return $next;
	}
	/**
	 * 月数加算
	 * @param object $d 日付
	 * @param integer $months 加算月数
	 * @return object 加算後の日付
	 */
	public static function addMonths($d, $months){
		if ($d == null) return null;
		$next = self::addDate($d, $months, "m");

		return $next;
	}
	/**
	 * 年加算
	 * @param object $d 日付
	 * @param integer $years 加算年数
	 * @return object 加算後の日付
	 */
	public static function addYears($d, $years){
		if ($d == null) return null;
		$next = self::addDate($d, $years, "y");

		return $next;
	}

	/**
	 * 次の月
	 * @param integer $year 年
	 * @param integer $month 月
	 * @return object 指定年月の翌月の１日
	 */
	public static function nextMonth($year, $month){
		return self::addMonths(self::createFromYearMonth($year, $month), 1);
	}
	/**
	 * 前の月
	 * @param integer $year 年
	 * @param integer $month 月
	 * @return object 指定年月の昨月の１日
	 */
	public static function previousMonth($year, $month){
		return self::addMonths(self::createFromYearMonth($year, $month), -1);
	}
	public static function year($date){
	   return intval( $date->format("Y"));
	}
    public static function month($date){
       return intval( $date->format("m"));
    }
    public static function day($date){
       return intval( $date->format("d"));
    }
    public static function hour($date){
       return intval( $date->format("h"));
    }
    public static function minute($date){
       return intval( $date->format("i"));
    }
    public static function second($date){
       return intval( $date->format("s"));
    }
    public static function dayOfWeek($date){
       return intval( $date->format("w"));
    }
}