<?php
Zend_Loader::loadClass("CFW_Util_String");

/**
 * 数値関連の便利メソッド
 * @author okada
 * @package CFW_Util
 */
class CFW_Util_Number{
	/**
	 * 整数値に変換
	 * @param unknown_type $o
	 * @return number|number
	 */
	public static function toInteger($o){
		if($o == null) return 0;
		if(is_int($o) ) return $o;

		$s = strval($o);
		$sn = CFW_Util_String::normalizeNumeric($s);

		return intval($sn);
	}

	/**
	 * byte型が期待される値を0～255内のint型に変換する
	 * @param object $o
	 * @return integer
	 */
	public static function toByte($o){
		if($o == null) return 0;
		$result = intval($o);
		if ($result < 0 || $result > 255) {
			return 0;
		}
		else {
			return $result;
		}
	}
	/**
	 * 数値型に変換(精度指定付き)
	 * @param object $o
	 * @param integer $precision
	 * @param integer $scale
	 * @return 数値型
	 */
	public static function toNumber($o,$precision = 20,$scale = 0){
		$result = floatval($o);
		$integer = strval(intval($result));
		$decimal = mb_substr(strval($result), mb_strlen($integer) + 1);

		if (mb_strlen($integer) > $precision - $scale) {
			return 0;
		}
		if (mb_strlen($decimal) > $scale) {
			return 0;
		}
		return $result;
	}
	public static function toFloat($o){
		return floatval($o);
	}

	/**
	 * Booleanが期待される値をBooleanに変換
	 * @param unknown_type $value 入力値
	 * @return unknown_type 入力値のBoolean表現
	 */
	public static function boolValue($value){
		if($value == null ) return false;
		if ($value == "") return false;
		if(is_bool($value)) return $value;
		try {
			if (mb_strtolower(strval($value)) == "false") return false;
			if (mb_strtolower(strval($value)) == "true") return true;
			if (mb_strtolower(strval($value)) == "no") return false;
			if (mb_strtolower(strval($value)) == "yes") return true;
			if (mb_strtolower(strval($value)) == "off") return false;
			if (mb_strtolower(strval($value)) == "on") return true;

			$i = self::toInteger($value);
			if ($i == 0) return false;
			else return true;
		} catch (Exception $e) {
			return false;
		}
	}

	/**
	 * Booleanを1/0にして返す。
	 * trueなら1,falseなら0
	 * @param Boolean $value 入力値
	 * @return Integer 変換結果
	 */
	public static function boolToInt($value){
		if ($value) return 1;
		else return 0;
	}

	/**
	 * デフォルトの書式で書式化
	 * @param object $value 入力値
	 * @return string $formated 書式化結果文字列
	 */
	public static function format($value){
		if ($value == null) return "";

		$formated = CFW_Util_String::toString($value);
		return $formated;
	}
	 /**
	  * デフォルトの書式で書式化
	  * @param unknown_type $value 入力値
	  * @return string 書式化結果文字列
	  */
	public static function defaultFormat($value){
		if ($value == null) return "";

		$formated = strval($value);
		return $formated;
	}

	/**
	 * boolの真偽に対し文字列を割り当てて返す
	 * @param Boolean $value 入力値
	 * @param unknown_type $trueExpression 値がtrueの場合の出力
	 * @param unknown_type $falseExpression 値がfalseの場合の出力
	 * @return string 書式化結果文字列
	 */
	public static function formatBool($value, $trueExpression, $falseExpression){
		if (is_bool($value)){
			if ($value) return $trueExpression;
			else return $falseExpression;
		}
	}

	/**
	 * boolの真偽に対し文字列(YかN)を割り当てて返す
	 * @param Boolean $value 入力値
	 * @return string 書式化結果文字列
	 */
	public static function formatBoolYN($value){
		return self::formatBool($value, "Y", "N");
	}

	/**
	 * boolの真偽に対し文字列(YesかNo)を割り当てて返す
	 * @param Boolean $value 入力値
	 * @return string 書式化結果文字列
	 */
	public static function formatBoolYesNo($value){
		return self::formatBool($value, "Yes", "No");
	}

	/**
	 * 桁区切り書式化
	 * @param decimal $value 入力値
	 * @param bool $useThousandsSeparator 桁区切りを使うか
	 * @param int $numDecimals 小数点以下の桁数
	 * @return string $result 書式化結果文字列
	 */
	public static function formatNumber($value,$useThousandsSeparator = true,$numDecimals = -1){
	    $result = "";
	    if($numDecimals < 0){
	        if($useThousandsSeparator){
                $result = number_format($value);
	        }
	        else{
                $result = strval($value);
	        }
	    }
	    else{
            if($useThousandsSeparator){
                $result = number_format($value,$numDecimals,".",",");
            }
            else{
                $result = number_format($value,$numDecimals,".","");
            }
	    }
		return $result;
	}

	/**
	 * 通貨形式書式化
	 * @param decimal $value 入力値
	 * @return string $result 書式化結果文字列
	 */
	public static function formatCurrency($value){
		$result = self::formatNumber($value);
		return "//".$result;
	}

	/**
	 * 文字列を解析してBooleanを返す
	 * @param string $s 解析対象文字列
	 * @return Boolean 解析結果
	 */
	public static function parseBool($s){
		if ($s == "1") return true;
		if (mb_strtolower($s) == "y") return true;
		if (mb_strtolower($s) == "yes") return true;
		if (mb_strtolower($s) == "true") return true;

		return false;
	}

	/**
	 * 2値の距離
	 *
	 * たとえば 0から2の距離は 2
	 * @param $origin 基準位置
	 * @param $position 位置
	 * @return unknown_type 距離。位置が基準位置の左側の場合負
	 */
	public static function distance($origin, $position){
		return $position - $origin;
	}

	/**
	 * 指定数値が範囲内にあるかチェック
	 * @param Number $target 入力値
	 * @param Number $min 最小値
	 * @param Number $max 最大値
	 * @return Boolean 入力値が最小値から最大値の範囲にあればtrue
	 */
	public static function isInRange($target, $min, $max){
		if ($target >= $min && $target <= $max) return true;
		else return false;
	}

	/**
	 * 0埋め書式化
	 * @param Number $value 入力値
	 * @param Number $width 出力文字数
	 * @return string $result 書式化結果文字列
	 */
	public static function formatZero($value, $width){
		$result = "";
		$valueToString = strval($value);
		if (mb_strlen($value) >= $width) {
			$result = $valueToString;
		}
		else {
			$repeatZero = $width - self::toInteger(mb_strlen($valueToString));
			$formatString = CFW_Util_String::repeat("0", $repeatZero);
			$result = $formatString.$valueToString;
		}
		return $result;
	}

	/**
	 * decimalのべき乗。
	 * 有理数乗はわからないので保留。
	 * @param decimal $x 基数
	 * @param int $y べき数
	 * @return Number $result decimalのべき乗。
	 */
	public static function power($x, $y){
		$result = 1;
		if ($y >= 0) {
			for ($d = 0; $d < $y; $d++) {
				$result *= $x;
			};
		}else {
			for ($d = 0; $d > $y; $d--) {
				$result /= $x;
			}
		}
		return $result;
	}

	/**
	 * 指定精度で四捨五入
	 * @param double $value 入力値
	 * @param int $scale 精度
	 * @return $result 四捨五入結果
	 */
	public static function round($value, $scale){
		return $result = round($value, $scale);
	}

	/**
	 * 指定精度で切り上げ
	 * @param double $value 入力値
	 * @param int $scale 精度
	 * @return $result 四捨五入結果
	 */
	public static function roundUp($value, $scale){
		$coefficient = pow(10 ,$scale);
		$value = $value * $coefficient;
		$value = ceil($value);
		$result = $value / $coefficient;
		return $result;
	}

	/**
	 * 指定精度で切り捨て
	 * @param double $value 入力値
	 * @param int $scale 精度
	 * @return $result 四捨五入結果
	 */
	public static function roundDown($value, $scale){
		$coefficient = pow(10 ,$scale);
		$value = $value * $coefficient;
		$value = floor($value);
		$result = $value / $coefficient;
		return $result;
	}
}