/*
 Copyright (c) 2009, hkrn All rights reserved.
 
 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions are met:
 
 Redistributions of source code must retain the above copyright notice, this
 list of conditions and the following disclaimer. Redistributions in binary
 form must reproduce the above copyright notice, this list of conditions and
 the following disclaimer in the documentation and/or other materials
 provided with the distribution. Neither the name of the hkrn nor
 the names of its contributors may be used to endorse or promote products
 derived from this software without specific prior written permission. 
 
 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
 ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 DAMAGE.
 */

//
//  MMLFilter.m
//  OCMML
//
//  Created by hkrn on 09/02/05.
//  Copyright 2009 hkrn. All rights reserved.
//
//  $Id: MMLFilter.m 65 2009-04-25 12:07:49Z hikarin $
//

#import "MMLFilter.h"

@implementation MMLFilter

- (id)init
{
    self = [super init];
    if (self != nil) {
        [self setSwitch:0];
    }
    return self;
}

- (void)reset
{
    t1 = t2 = b0 = b1 = b2 = b3 = b4 = 0.0;
}

- (void)setSwitch:(enum MMLFilterType)aSwitch
{
    [self reset];
    sw = aSwitch;
}

static inline void UpdateCut(double *cut)
{
    if (*cut < (1.0 / 127.0))
        *cut = 0.0;
    *cut = MIN(*cut, 1.0 - 0.0001);
}

static inline void UpdateCutAndFb(double *cut, double *fb, double resonance)
{
    UpdateCut(cut);
    *fb = resonance + resonance / (1.0 - *cut);
}

static inline double GetKeyValue(double key)
{
    return key * (2.0 * M_PI / (44100.0 * 440.0));
}

- (void)updateSamplesForHPF1:(double *)aSamples
                       index:(int)i
                         cut:(double)cut
                          fb:(double)fb
{
    double input = aSamples[i];
    b0 = b0 + cut * (input - b0 + fb * (b0 - b1));
    b1 = b1 + cut * (b0 - b1);
    aSamples[i] = input - b0;
}

- (void)updateSamplesForLPF1:(double *)aSamples
                       index:(int)i
                         cut:(double)cut
                          fb:(double)fb
{
    b0 = b0 + cut * (aSamples[i] - b0 + fb * (b0 - b1));
    aSamples[i] = b1 = b1 + cut * (b0 - b1);
}

- (void)LPF1WithSamples:(double *)samples
                  start:(int)start
                    end:(int)end
               envelope:(MMLEnvelope *)envelope
              frequency:(double)frequency
                 amount:(double)amount
             resonance:(double)resonance
                    key:(double)key
{
    double k = GetKeyValue(key);
    double fb = 0;
    if (amount > 0.0001 || amount < -0.0001) {
        for (int i = start; i < end; i++) {
            double cut = [MMLChannel frequencyWithNumber:(int)(frequency + amount * envelope.nextAmplitudeLinear)] * k;
            UpdateCutAndFb(&cut, &fb, resonance);
            [self updateSamplesForLPF1:samples
                                 index:i
                                   cut:cut
                                    fb:fb];
        }
    }
    else {
        double cut = [MMLChannel frequencyWithNumber:(int)frequency] * k;
        UpdateCutAndFb(&cut, &fb, resonance);
        for (int i = start; i < end; i++) {
            [self updateSamplesForLPF1:samples
                                 index:i
                                   cut:cut
                                    fb:fb];
        }
    }
}

- (void)LPF2WithSamples:(double *)samples
                  start:(int)start
                    end:(int)end
               envelope:(MMLEnvelope *)envelope
              frequency:(double)frequency
                 amount:(double)amount
              resonance:(double)resonance
                    key:(double)key
{
    double k = GetKeyValue(key);
    for (int i = start; i < end; i++) {
        double cut = [MMLChannel frequencyWithNumber:(int)(frequency + amount * envelope.nextAmplitudeLinear)] * k;
        UpdateCut(&cut);
        double q = 1.0 - cut;
        double p = cut + 0.8 * cut * q;
        double f = p + p - 1.0;
        q = resonance * (1.0 + 0.5 * q * (1.0 - q + 5.6 * q * q));
        double input = samples[i];
        input -= q * b4;
        t1 = b1;
        b1 = (input + b0) * p - b1 * f;
        t2 = b2;
        b2 = (b1 + t1) * p - b2 * f;
        t1 = b3;
        b3 = (b2 + t2) * p - b3 * f;
        b4 = (b3 + t1) * p - b4 * f;
        b4 = b4 - b4 * b4 * b4 * 0.166667;
        b0 = input;
        samples[i] = b4;
    }
}

- (void)HPF1WithSamples:(double *)samples
                  start:(int)start
                    end:(int)end
               envelope:(MMLEnvelope *)envelope
              frequency:(double)frequency
                 amount:(double)amount
              resonance:(double)resonance
                    key:(double)key
{
    double k = GetKeyValue(key);
    double fb = 0;
    if (amount > 0.0001 || amount < -0.0001) {
        for (int i = start; i < end; i++) {
            double cut = [MMLChannel frequencyWithNumber:(int)(frequency + amount * envelope.nextAmplitudeLinear)] * k;
            UpdateCutAndFb(&cut, &fb, resonance);
            [self updateSamplesForHPF1:samples
                                 index:i
                                   cut:cut
                                    fb:fb];
        }
    }
    else {
        double cut = [MMLChannel frequencyWithNumber:(int)frequency] * k;
        UpdateCutAndFb(&cut, &fb, resonance);
        for (int i = start; i < end; i++) {
            [self updateSamplesForHPF1:samples
                                 index:i
                                   cut:cut
                                    fb:fb];
        }
    }
}

- (void)HPF2WithSamples:(double *)samples
                  start:(int)start
                    end:(int)end
               envelope:(MMLEnvelope *)envelope
              frequency:(double)frequency
                 amount:(double)amount
              resonance:(double)resonance
                    key:(double)key
{
    double k = GetKeyValue(key);
    for (int i = start; i < end; i++) {
        double cut = [MMLChannel frequencyWithNumber:(int)(frequency + amount * envelope.nextAmplitudeLinear)] * k;
        UpdateCut(&cut);
        double q = 1.0 - cut;
        double p = cut + 0.8 * cut * q;
        double f = p + p - 1.0;
        q = resonance * (1.0 + 0.5 * q * (1.0 - q + 5.6 * q * q));
        double input = samples[i];
        input -= q * b4;
        t1 = b1;
        b1 = (input + b0) * p - b1 * f;
        t2 = b2;
        b2 = (b1 + t1) * p - b2 * f;
        t1 = b3;
        b3 = (b2 + t2) * p - b3 * f;
        b4 = (b3 + t1) * p - b4 * f;
        b4 = b4 - b4 * b4 * b4 * 0.166667;
        b0 = input;
        samples[i] = input - b4;
    }
}

- (void)runWithSamples:(double *)samples
                 start:(int)start
                   end:(int)end
              envelope:(MMLEnvelope *)envelope
             frequency:(double)frequeycy
                amount:(double)amount
             resonance:(double)resonance
                   key:(double)key
{
    switch (sw) {
        case kMMLFilterHPFFast:
            [self HPF2WithSamples:samples
                            start:start
                              end:end
                         envelope:envelope
                        frequency:frequeycy
                           amount:amount
                        resonance:resonance
                              key:key];
            break;
        case kMMLFilterHPFQuality:
            [self HPF1WithSamples:samples
                            start:start
                              end:end
                         envelope:envelope
                        frequency:frequeycy
                           amount:amount
                        resonance:resonance
                              key:key];
            break;
        case kMMLFilterLPFQuality:
            [self LPF1WithSamples:samples
                            start:start
                              end:end
                         envelope:envelope
                        frequency:frequeycy
                           amount:amount
                        resonance:resonance
                              key:key];
            break;
        case kMMLFilterLPFFast:
            [self LPF2WithSamples:samples
                            start:start
                              end:end
                         envelope:envelope
                        frequency:frequeycy
                           amount:amount
                        resonance:resonance
                              key:key];
            break;
        case kMMLFilterNone:
        default:
            break;
    }
}

@end
