package jp.sf.nikonikofw;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.util.Properties;
import java.util.logging.Logger;

import javax.servlet.RequestDispatcher;
import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import jp.sf.nikonikofw.action.IAction;
import jp.sf.nikonikofw.annotation.Authentication;
import jp.sf.nikonikofw.annotation.Request;
import jp.sf.nikonikofw.exception.AuthenticationException;
import jp.sf.nikonikofw.exception.InitializeException;
import jp.sf.nikonikofw.exception.ValidationException;
import jp.sf.nikonikofw.util.IOUtils;
import jp.sf.nikonikofw.util.StringUtils;
import jp.sf.nikonikofw.validation.DefaultValidator;
import jp.sf.nikonikofw.validation.IValidator;
import jp.sf.nikonikofw.validation.Required;

import org.apache.commons.fileupload.servlet.ServletFileUpload;

public class ControllerServlet extends HttpServlet {
	
	private static final Logger logger = Logger.getLogger(ControllerServlet.class.getName());
	
	public static final String REQUEST_ATTR_ACTION = ControllerServlet.class.getName() + "#action";
	
	private static final String CONFIG_FILE_NAME = "framework.properties";
	
	private static final long serialVersionUID = 1L;

	/**
	 * フレームワークの初期化処理を行います。
	 */
	@Override
	public void init(ServletConfig config) throws ServletException {
		super.init(config);
		
		ClassLoader loader = Thread.currentThread().getContextClassLoader();
		InputStream in = loader.getResourceAsStream(CONFIG_FILE_NAME);
		if(in == null){
			throw new InitializeException("設定ファイル" + CONFIG_FILE_NAME + "が見つかりません。");
		}
		
		Properties props = new Properties();
		try {
			props.load(in);
			
		} catch(Exception ex){
			throw new InitializeException("設定ファイルの読み込みに失敗しました。", ex);
			
		} finally {
			IOUtils.closeQuietly(in);
		}
		
		Config.initConfig(props);
	}

	@Override
	protected void service(HttpServletRequest request, HttpServletResponse response) 
			throws ServletException, IOException {
		
		if(ServletFileUpload.isMultipartContent(request)){
			request = new MultipartRequest(request);
		}
		
		beginRequest(request, response);
		
		request.setCharacterEncoding("UTF-8");
		request.setAttribute("context", getServletContext().getContextPath());
		
		try {
			// アクションのインスタンスを生成
			String path = request.getRequestURI().substring(request.getContextPath().length() + 1);
			IAction action = Config.getActionMapping().getAction(path);
			if(action == null){
				// TODO 例外はこれでいいのかなー
				throw new RuntimeException(path + "に対応するアクションが見つかりません。");
			}
			
			request.setAttribute(REQUEST_ATTR_ACTION, action);
			
			// 認証チェック
			authentication(action);
			
			// リクエストパラメータのバインド
			bindRequestParameters(action);
			
			// アクションの実行
			String result = action.execute(request, response);
			
			// null以外の場合は戻り値のパスにフォワード
			if(result != null){
				RequestDispatcher dispatcher = request.getRequestDispatcher(result);
				dispatcher.forward(request, response);
			}
			
			// トランザクションをコミット
			Config.getPersistenceManager().commit();
			
		} catch(Throwable t){
			try {
				processError(request, response, t);
				
			} catch(ServletException ex){
				throw (ServletException) ex;
			} catch(IOException ex){
				throw (IOException) ex;
			} catch(RuntimeException ex){
				throw (RuntimeException) ex;
			} catch(Throwable ex){
				throw new RuntimeException(ex);
			}
		} finally {
			endRequest(request, response);
		}
	}
	
	/**
	 * リクエスト処理の開始時に呼び出されます。
	 * 
	 * @param request リクエスト
	 * @param response レスポンス
	 */
	protected void beginRequest(HttpServletRequest request, HttpServletResponse response){
		Config.persistenceManager.begin();
		ServletUtil.init(request, response);
	}
	
	/**
	 * リクエスト処理の終了時に呼び出されます。
	 * 
	 * @param request リクエスト
	 * @param response レスポンス
	 */
	protected void endRequest(HttpServletRequest request, HttpServletResponse response){
		Config.persistenceManager.close();
		ServletUtil.release();
	}
	
	/**
	 * リクエスト処理中に例外が発生した場合に呼び出されます。
	 * <p>
	 * 本実装ではトランザクションのロールバック処理のみ行い、例外を再スローします。
	 * 
	 * @param request リクエスト
	 * @param response レスポンス
	 * @param t 発生した例外
	 */
	protected void processError(HttpServletRequest request, HttpServletResponse response, 
			Throwable t) throws Throwable {
		
		Config.persistenceManager.rollback();
		
		throw t;
	}

	/**
	 * 認証チェックを行います。
	 * <p>
	 * 本実装では{@link Authentication}アノテーションの有無によって認証チェックを行い、
	 * エラーの場合は{@link AuthenticationException}が発生します。
	 * 
	 * @param action アクション
	 * @throws AuthenticationException 認証エラーの場合
	 */
	protected void authentication(IAction action) throws AuthenticationException {
		Authentication auth = action.getClass().getAnnotation(Authentication.class);
		if(auth != null){
			if(!Config.getAuthenticationMananger().isLogin()){
				throw new AuthenticationException();
			}
		}
	}
	
	/**
	 * アクションクラスのRequestアノテーションがついたpublicフィールドに
	 * リクエストパラメータをバインドします。
	 */
	protected void bindRequestParameters(IAction action){
		HttpServletRequest req = ServletUtil.getRequest();
		
		DefaultValidator validator = new DefaultValidator();
		
		for(Field field: action.getClass().getFields()){
			Request ann = field.getAnnotation(Request.class);
			if(ann != null){
				String name = ann.name();
				if(field.getType().isArray()){
					String[] values = req.getParameterValues(name);
					validator.validateValues(ann.label(), field, values);
					
					if(values != null){
						try {
							field.set(action, values);
						} catch(Exception ex){
							logger.warning(StringUtils.getStackTrace(ex));
						}
					}
				} else {
					if(StringUtils.isEmpty(name)){
						name = field.getName();
					}
					
					if(ServletFileUpload.isMultipartContent(req) && field.getType().equals(FileItem.class)){
						// マルチパートかつファイルフィールドの場合
						FileItem value = ((MultipartRequest) req).getFileItem(name);
						if(field.getAnnotation(Required.class) != null && (value==null || value.getFieldValue()==null || value.getFieldValue().length==0)){
							if(StringUtils.isEmpty(ann.label())){
								throw new ValidationException();
							} else {
								throw new ValidationException(ann.label() + "は入力必須です。");
							}
						}
						if(value!=null && value.getFieldValue()!=null && value.getFieldValue().length!=0){
							try {
								field.set(action, value);
							} catch(Exception ex){
								logger.info("リクエストパラメータのバインドに失敗しました。" +
										" フィールド名：" + field.getName() +
										" 値：" + value + 
										" 例外：" + ex.toString());
							}
						}
						
					} else {
						// ファイルフィールド以外の場合
						String value = req.getParameter(name);
						validator.validateValue(ann.label(), field, value);
						
						if(value != null){
							try {
								field.set(action, StringUtils.convert(value, field.getType()));
							} catch(Exception ex){
								logger.info("リクエストパラメータのバインドに失敗しました。" +
										" フィールド名：" + field.getName() +
										" 値：" + value + 
										" 例外：" + ex.toString());
							}
						}
					}
				}
			}
		}
	}
	
	protected IValidator createValidator(){
		return new DefaultValidator();
	}

}
