package net.sf.amateras.nikocale;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.util.logging.Logger;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import jp.sf.nikonikofw.Config;
import jp.sf.nikonikofw.ControllerServlet;
import jp.sf.nikonikofw.persistence.jdbc.JdbcPersistenceManager;
import jp.sf.nikonikofw.persistence.jdbc.JdbcUtil;
import jp.sf.nikonikofw.util.IOUtils;
import net.sf.amateras.nikocale.entity.SystemInfo;
import net.sf.amateras.nikocale.util.Utils;
import net.sf.amateras.nikocale.util.Version;

/**
 * ニコカレ用のコントローラサーブレットです。
 * 
 * @author Naoki Takezoe
 */
@SuppressWarnings("serial")
public class NikocaleServlet extends ControllerServlet {
	
	private static Logger logger = Logger.getLogger(NikocaleServlet.class.getName());
	
	/**
	 * サーブレットの初期化処理を行います。
	 * <p>
	 * HSQLDBのデータベースファイルのパスをJDBC接続のURLに含める必要があるため、
	 * ここでデータベース接続用のURLを上書きします。
	 * また、データベースのマイグレーション（DBの作成とバージョンアップ時の自動最新化）を行います。
	 * </p>
	 */
	@Override
	public void init(ServletConfig config) throws ServletException {
		super.init(config);
		try {
			File data = Utils.getDataDir();
			
			JdbcPersistenceManager pm = (JdbcPersistenceManager) Config.getPersistenceManager();
			Field field = JdbcPersistenceManager.class.getDeclaredField("url");
			field.setAccessible(true);
			field.set(pm, "jdbc:hsqldb:file:" + data.getAbsolutePath() + "/nikocale");
			
			migration(pm, data);
			
			config.getServletContext().setAttribute("version", Version.VERSION);
			
		} catch(Exception ex){
			throw new ServletException(ex);
		}
	}
	
	/**
	 * データベースの作成・マイグレーションを行います。
	 * <p>
	 * データベースディレクトリが存在しない場合はデータベースを作成します。
	 * その後、未反映のSQLマイグレーションファイルをバージョンの順番で実行します。
	 */
	public static void migration(JdbcPersistenceManager pm, File dir) throws Exception {
		logger.info("データベースのマイグレーションを開始します。");
		
		Class.forName("org.hsqldb.jdbcDriver");
		
		boolean firstTime = !new File(dir, "nikocale.properties").exists();
		
		pm.begin();
		
		try {
			if(firstTime){
				String sql = IOUtils.read(Version.PREFIX + "nikocale_1.0.0.sql");
				logger.info("nikocale_1.0.0.sqlを実行します。");
				
				JdbcUtil.execute(sql, new Object[0]);
				pm.commit();
			}
			
			pm.begin();
			try {
				SystemInfo systemInfo = JdbcUtil.getSingleResult(SystemInfo.class, "SELECT VERSION FROM SYSTEM_INFO");
				String appVersion = systemInfo.getVersion();
				boolean flag = false;
				
				for(Version version: Version.getVersions()){
					if(version.getVersion().equals(appVersion)){
						flag = true;
						continue;
					}
					if(flag == true){
						if(version.getSql() != null){
							logger.info(version.getSql() + "を実行します。");
							
							String sql = IOUtils.read(Version.PREFIX + version.getSql());
							JdbcUtil.execute(sql, new Object[0]);
						}
					}
				}
				if(!appVersion.equals(Version.VERSION)){
					JdbcUtil.execute("UPDATE SYSTEM_INFO SET VERSION = ?", Version.VERSION);
				}
				pm.commit();
			} catch(Exception ex){
				pm.rollback();
				throw ex;
			}
		} finally {
			pm.close();
		}
		
		logger.info("データベースのマイグレーションを終了します。");
	}

	@Override
	protected void service(HttpServletRequest req, HttpServletResponse res) 
			throws ServletException, IOException {
		
		req.setCharacterEncoding("UTF-8");
		req.setAttribute("context", getServletContext().getContextPath());
		
		Cookie[] cookies = req.getCookies();
		if(cookies != null){
			for(Cookie cookie: cookies){
				if(cookie.getName().equals("NIKOCALE")){
					String value = cookie.getValue();
					req.setAttribute("setting_noAjax", String.valueOf(value.charAt(0)));
					req.setAttribute("setting_textarea", String.valueOf(value.charAt(1)));
				}
			}
		}
		
		super.service(req, res);
		
	}
	
//	/**
//	 * エラー時の処理を行います。通常はエラー画面へ遷移しますが、
//	 * Ajaxアクションの場合はエラーメッセージをレスポンスボディとして返却します。
//	 */
//	private void processError(Exception ex, Class<?> actionClass, 
//			HttpServletRequest req, HttpServletResponse res) throws Exception {
//		
//		if(actionClass.getAnnotation(Ajax.class) != null){
//			res.setStatus(500);
//			res.setContentType("text/plain; charset=UTF-8");
//			res.getOutputStream().write(ex.getMessage().getBytes("UTF-8"));
//			
//		} else {
//			req.setAttribute("title", "エラー");
//			
//			String error = ex.getMessage();
//			if(StringUtils.isEmpty(error)){
//				error = ex.toString();
//			}
//			req.setAttribute("error", error);
//			
//			RequestDispatcher dispatcher = req.getRequestDispatcher("error.jsp");
//			dispatcher.forward(req, res);
//		}
//	}

}
