/********************************************************************************/
/* FileName: RX62N_driver_GPIO.cpp											  */
/* Function: RX62N GPIO Functions											   */
/********************************************************************************/

/*
 *	  Copyright (C) 2009-2010 Renesas Electronics America Inc. All rights reserved.
 */
 
#include <tinyhal.h>
#include "RX62N_GPIO.h"

RX62N_GPIO_Driver g_RX62N_GPIO_Driver;

void RX62N_GPIO_Driver::PIN_ISR_DESCRIPTOR::Fire( void* arg )
{
	NATIVE_PROFILE_HAL_PROCESSOR_GPIO();
	PIN_ISR_DESCRIPTOR* desc = (PIN_ISR_DESCRIPTOR*)arg;
		
	desc->m_isr( desc->m_pin, (desc->m_status & c_Status_AllowHighEdge) != 0, desc->m_param );

	// we need to clear (invert) the status only for actual dual edge pins
	UINT8 mask = c_Flags_RequireHighEdge | c_Flags_RequireLowEdge;
	
	if((desc->m_flags & mask) == mask)
	{
		desc->m_status ^= PIN_ISR_DESCRIPTOR::c_Status_AllowLowEdge | PIN_ISR_DESCRIPTOR::c_Status_AllowHighEdge;
	}
}

void RX62N_GPIO_Driver::STUB_GPIOISRVector( GPIO_PIN Pin, BOOL PinState, void* Param )
{
//	  lcd_printf( "\fSTUB_GPI %04x\r\n", (size_t)Param );
	 hal_printf( "\fSTUB_GPI %04x %04x\r\n", Pin, (size_t)Param );
//	  DEBUG_HARD_BREAKPOINT();
}

BOOL RX62N_GPIO_Driver::Initialize()
{
	NATIVE_PROFILE_HAL_PROCESSOR_GPIO();
	UINT32 i;

	// initialize the interrupt information
	{
		PIN_ISR_DESCRIPTOR* pinIsr = g_RX62N_GPIO_Driver.m_PinIsr;

		for(i = 0; i < c_MaxPins; i++)
		{
			pinIsr->m_pin	  = i;
			pinIsr->m_intEdge = GPIO_INT_NONE;
			pinIsr->m_isr	  = STUB_GPIOISRVector;
			pinIsr->m_param	  = NULL;

			pinIsr->m_completion.Initialize( );
			pinIsr->m_completion.InitializeForISR( &PIN_ISR_DESCRIPTOR::Fire, pinIsr );

			pinIsr++;
		}
	}

//	  for(i = 0; i < c_MaxPins; i++)
	for(i = 0; i < c_MaxPorts; i++)
	{
		// initialize pins as free
		g_RX62N_GPIO_Driver.m_PinReservationInfo[i] = 0;
	}	
	
	return TRUE;
}

BOOL RX62N_GPIO_Driver::Uninitialize()
{
	NATIVE_PROFILE_HAL_PROCESSOR_GPIO();
	GLOBAL_LOCK(irq);

	// uninitialize the interrupt information
	{
		PIN_ISR_DESCRIPTOR* pinIsr = g_RX62N_GPIO_Driver.m_PinIsr;
		for(int i = 0; i < c_MaxPins; i++)
		{
			pinIsr->m_completion.Abort();
			pinIsr++;
		}
	}

	return TRUE;
}

UINT32 RX62N_GPIO_Driver::Attributes( GPIO_PIN Pin )
{
	NATIVE_PROFILE_HAL_PROCESSOR_GPIO();
	UINT32 Port = PinToPort(Pin);
	UINT32 Bit = PinToBit(Pin);

	return BitAttributes(Bit, Port);
}

void RX62N_GPIO_Driver::DisablePin( GPIO_PIN Pin, GPIO_RESISTOR ResistorState, UINT32 direction, GPIO_ALT_MODE AltFunction )
{
	NATIVE_PROFILE_HAL_PROCESSOR_GPIO();
	UINT32	 bitMask  = 1 << PinToBit( Pin );
	UINT32 Port = PinToPort(Pin);
	UINT32 Bit = PinToBit(Pin);	  
	
	GLOBAL_LOCK(irq);
	
	// Set pin to specified alternate function
	GPIO_IOReg_Write(Port, (GPIO_IOReg_Read(Port) | (0x1 << Bit)));
	
	if(direction)
		GPIO_IOReg_Write(Port, (GPIO_IOReg_Read(Port) | (0x1 << Bit))); // Set pin as output
	else
		GPIO_IOReg_Write(Port, (GPIO_IOReg_Read(Port) & ~(0x1 << Bit))); // Set pin as input

	PIN_ISR_DESCRIPTOR& pinIsr = g_RX62N_GPIO_Driver.m_PinIsr[Pin];

	pinIsr.m_intEdge = GPIO_INT_NONE;
	pinIsr.m_isr	 = STUB_GPIOISRVector;
	pinIsr.m_param	 = NULL;
}

void RX62N_GPIO_Driver::EnableOutputPin( GPIO_PIN Pin, BOOL InitialState )
{
	NATIVE_PROFILE_HAL_PROCESSOR_GPIO();

	unsigned short Port, Bit;
	
	GLOBAL_LOCK(irq);

	Port = PinToPort(Pin);
	Bit = PinToBit(Pin);
	
	GPIO_ControlReg_Write(Port, (GPIO_ControlReg_Read(Port) & ~(0x1 << Bit)));
	GPIO_IOReg_Write(Port, (GPIO_IOReg_Read(Port) | (0x1 << Bit)));
}

BOOL RX62N_GPIO_Driver::EnableInputPin( GPIO_PIN pin, BOOL GlitchFilterEnable, GPIO_INTERRUPT_SERVICE_ROUTINE ISR, void* pinIsrParam, GPIO_INT_EDGE intEdge, GPIO_RESISTOR resistorState )
{
	NATIVE_PROFILE_HAL_PROCESSOR_GPIO();

	unsigned short Port, Bit;
	
	GLOBAL_LOCK(irq);

	Port = PinToPort(pin);
	Bit = PinToBit(pin);

	GPIO_ControlReg_Write(Port, (GPIO_ControlReg_Read(Port) & ~(0x1 << Bit)));
	GPIO_IOReg_Write(Port, (GPIO_IOReg_Read(Port) & ~(0x1 << Bit)));
	
	return TRUE;
}

BOOL RX62N_GPIO_Driver::GetPinState( GPIO_PIN Pin )
{
	NATIVE_PROFILE_HAL_PROCESSOR_GPIO();

	UINT32 Port, Bit;
   
	GLOBAL_LOCK(irq);

	Port = PinToPort(Pin);
	Bit = PinToBit(Pin);

	return (GPIO_DataReg_Read(Port) & (0x1 << Bit));
}

void RX62N_GPIO_Driver::SetPinState( GPIO_PIN Pin, BOOL PinState )
{
	NATIVE_PROFILE_HAL_PROCESSOR_GPIO();
	ASSERT(Pin < c_MaxPins);
	
	UINT32 Port, Bit;
	
	GLOBAL_LOCK(irq);

	Port = PinToPort(Pin);
	Bit = PinToBit(Pin);

	if(PinState)
		GPIO_DataReg_Write(Port, (GPIO_DataReg_Read(Port) | (0x1 << Bit)));
	else
		GPIO_DataReg_Write(Port, (GPIO_DataReg_Read(Port) & ~(0x1 << Bit)));
}

void RX62N_GPIO_Driver::ISR()
{
	NATIVE_PROFILE_HAL_PROCESSOR_GPIO();

}

void RX62N_GPIO_Driver::SetResistor( GPIO_PIN Pin, GPIO_RESISTOR ResistorState )
{
	NATIVE_PROFILE_HAL_PROCESSOR_GPIO();	
	return; 
}

BOOL RX62N_GPIO_Driver::PinIsBusy( GPIO_PIN Pin )
{
	NATIVE_PROFILE_HAL_PROCESSOR_GPIO();
	UINT32	port	 = PinToPort(Pin);
	UINT32	bitMask	 = 1 << PinToBit(Pin);
	UINT32& res		 = g_RX62N_GPIO_Driver.m_PinReservationInfo[port];

	return (res & bitMask) != 0 ? TRUE : FALSE;
}

BOOL RX62N_GPIO_Driver::ReservePin( GPIO_PIN Pin, BOOL fReserve )
{
	NATIVE_PROFILE_HAL_PROCESSOR_GPIO();
	GLOBAL_LOCK(irq);

	UINT32	port = PinToPort(Pin);
	UINT32	bit	 = PinToBit(Pin);
	UINT32& res	 = g_RX62N_GPIO_Driver.m_PinReservationInfo[port];
	UINT32	mask = (fReserve ? 1 : 0) << bit;

	if((res & (1 << bit)) == mask) 
		return FALSE;

	if(fReserve)
	{
		res |= (1 << bit);
	}
	else
	{
		res &= ~(1 << bit);
	}

	return TRUE;
}

UINT32 RX62N_GPIO_Driver::GetDebounce()
{
	NATIVE_PROFILE_HAL_PROCESSOR_GPIO();
	return CPU_TicksToTime( g_RX62N_GPIO_Driver.m_DebounceTicks ) / 10000;
}

BOOL RX62N_GPIO_Driver::SetDebounce( INT64 debounceTimeMilliseconds )
{
	NATIVE_PROFILE_HAL_PROCESSOR_GPIO();
	if(debounceTimeMilliseconds < c_MinDebounceTimeMs || c_MaxDebounceTimeMs < debounceTimeMilliseconds )
	{
		return FALSE;
	}

	g_RX62N_GPIO_Driver.m_DebounceTicks = CPU_MillisecondsToTicks( (UINT32)debounceTimeMilliseconds );
	
	return TRUE;
}

void RX62N_GPIO_Driver::GetPinsMap( UINT8* pins, size_t size )
{
	NATIVE_PROFILE_HAL_PROCESSOR_GPIO();

	UINT8* dst = pins;	 
	UINT8 attr;

	if ( size == 0 ) return;

	for ( UINT32 port = 0; port < 17; port++ ) {
		for ( UINT32 bit = 0; bit < 8; bit++ ) {
			if((attr = BitAttributes(bit, port)) != GPIO_ATTRIBUTE_NONE){
				*dst = attr;
				dst++;
				size--;
				if ( size == 0 ) return;
			}
		}
	}
}

/*
void RX62N_GPIO_Driver::SetPinData( GPIO_PIN Pin, BOOL Value )
{
	NATIVE_PROFILE_HAL_PROCESSOR_GPIO();

	unsigned short Port, Bit;	 
	
	GLOBAL_LOCK(irq);

	Port = PinToPort(Pin);
	Bit = PinToBit(Pin);
	
	if (Value == 1)	 // Pin High
	{		
		GPIO_DataReg_Write(Port, (GPIO_DataReg_Read(Port) | (0x1 << Bit)));
	}
	else			// Pin Low
	{
		GPIO_DataReg_Write(Port, (GPIO_DataReg_Read(Port) & ~(0x1 << Bit)));
	}
}

unsigned short RX62N_GPIO_Driver::GetPinData( GPIO_PIN Pin)
{
	NATIVE_PROFILE_HAL_PROCESSOR_GPIO();

	unsigned short Port, Bit;	
	
	GLOBAL_LOCK(irq);

	Port = PinToPort(Pin);
	Bit = PinToBit(Pin);	  
			
	return (GPIO_DataReg_Read(Port) & (0x1 << Bit));
}
*/
