#
# Cookbook Name:: openldap
# Recipe Spec:: nss-ldapd_spec
#
# Copyright 2015-2018, whitestar
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

require_relative '../spec_helper'

describe 'openldap::nss-ldapd' do
  let(:chef_run_on_debian) {
    ChefSpec::SoloRunner.new(platform: 'debian', version: '7.6') {|node|
      node.override['openldap']['nss-ldapd']['base'] = 'dc=example,dc=net'
      node.override['openldap']['nss-ldapd']['invalid_key'] = 'value'
    }.converge(described_recipe)
  }

  it 'installs libnss-ldapd and nscd packages on debian family.' do
    expected = expect(chef_run_on_debian)
    expected.to install_package('libnss-ldapd')
    expected.to enable_service('nslcd')
    expected.to install_package('nscd')
    expected.to enable_service('nscd')
  end

  it 'renders /etc/nslcd.conf on debian family.' do
    expected = expect(chef_run_on_debian)
    expected.to render_file('/etc/nslcd.conf')
      .with_content(/^base\s+dc=example,dc=net$/)
    expected.to_not render_file('/etc/nslcd.conf')
      .with_content(/^invalid_key\s+value$/)
  end

  it 'notifies nslcd and nscd service\'s restart after rendering on debian family.' do
    expected = expect(chef_run_on_debian.template('/etc/nslcd.conf'))
    expected.to notify('service[nslcd]').to(:restart).delayed
    expected.to notify('service[nscd]').to(:restart).delayed
  end

  it 'creates ruby_block on debian family.' do
    expected = expect(chef_run_on_debian)
    expected.to run_ruby_block('configuring_nameservices')
  end

  it 'notifies nscd service\'s restart after creating ruby_block on debian family.' do
    expected = expect(chef_run_on_debian.ruby_block('configuring_nameservices'))
    expected.to notify('service[nscd]').to(:restart).delayed
  end

  let(:chef_run_on_rhel) {
    ChefSpec::SoloRunner.new(platform: 'centos', version: '7.0') {|node|
      node.override['openldap']['nss-ldapd']['base'] = 'dc=example,dc=net'
      node.override['openldap']['nss-ldapd']['invalid_key'] = 'value'
    }.converge(described_recipe)
  }

  it 'installs nss-pam-ldapd on rhel family.' do
    expected = expect(chef_run_on_rhel)
    expected.to install_package('nss-pam-ldapd')
    expected.to enable_service('nslcd')
    expected.to enable_service('nscd')
  end

  it 'renders /etc/nslcd.conf on rhel family.' do
    expected = expect(chef_run_on_rhel)
    expected.to create_template('/etc/nslcd.conf').with(mode: '0600')
    expected.to render_file('/etc/nslcd.conf')
      .with_content(/^base\s+dc=example,dc=net$/)
    expected.to_not render_file('/etc/nslcd.conf')
      .with_content(/^invalid_key\s+value$/)
  end

  it 'notifies nslcd and nscd service\'s restart after rendering on rhel family.' do
    expected = expect(chef_run_on_rhel.template('/etc/nslcd.conf'))
    expected.to notify('service[nslcd]').to(:restart).delayed
    expected.to notify('service[nscd]').to(:restart).delayed
  end

  it 'creates ruby_block on rhel family.' do
    expected = expect(chef_run_on_rhel)
    expected.to run_ruby_block('configuring_nameservices')
  end

  it 'notifies nscd service\'s restart after creating ruby_block on rhel family.' do
    expected = expect(chef_run_on_rhel.ruby_block('configuring_nameservices'))
    expected.to notify('service[nscd]').to(:restart).delayed
  end
end
