#
# Cookbook Name:: docker-grid
# Recipe:: compose
#
# Copyright 2016-2018, whitestar
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

# See: https://docs.docker.com/compose/install/

install_flavor = node['docker-grid']['compose']['install_flavor']
auto_upgrade = node['docker-grid']['compose']['auto_upgrade']

if node['docker-grid']['compose']['skip_setup']
  log 'Skip the Docker Compose setup.'
  return
end

include_recipe 'docker-grid::engine'

home_dir = node['docker-grid']['compose']['home_dir']
app_dir = node['docker-grid']['compose']['app_dir']
[
  home_dir,
  app_dir,
].each {|dir|
  resources(directory: dir) rescue directory dir do
    owner 'root'
    group 'root'
    mode '0755'
    recursive true
  end
}

docker_compose_path = '/usr/local/bin/docker-compose'
docker_compose_link = '/usr/sbin/docker-compose'

# clean up installed docker-compose(s)
file docker_compose_path do
  action :delete
  not_if { install_flavor == 'dockerproject' || install_flavor == 'pypi' }
end

link "delete #{docker_compose_link}" do
  target_file docker_compose_link
  action :delete
  not_if { install_flavor == 'dockerproject' }
end

package 'delete docker-compose' do
  package_name 'docker-compose'
  action :remove
  not_if { install_flavor == 'os-repository' }
end

execute 'pip uninstall docker-compose' do
  user 'root'
  environment 'HOME' => '/root'
  command 'pip uninstall docker-compose'
  action :run
  only_if 'pip list | grep docker-compose'
  not_if { install_flavor == 'pypi' }
end

case install_flavor
when 'dockerproject'
  [
    'curl',
  ].each {|pkg|
    resources(package: pkg) rescue package pkg do
      action :install
    end
  }

  release_url = node['docker-grid']['compose']['release_url']
  expected_ver = release_url.match(/(\d+\.\d+.\d+)/)
  expected_ver = expected_ver[1] unless expected_ver.nil?
  # latest version
  if expected_ver.nil?
    pkgs = ['jq']
    pkgs.unshift('epel-release') if node['platform_family'] == 'rhel'
    pkgs.each {|pkg|
      resources(package: pkg) rescue package pkg do
        action :install
      end.run_action(:install)
    }

    result = shell_out!('curl -s https://api.github.com/repos/docker/compose/releases/latest | jq -r .tag_name')
    expected_ver = result.stdout.chomp if result.exitstatus.zero? && !result.stdout.chomp.empty?
    release_url = "https://github.com/docker/compose/releases/download/#{expected_ver}/docker-compose-#{node['kernel']['name']}-#{node['kernel']['machine']}"
  end

  execute 'install_docker_compose' do
    user 'root'
    command "curl -L \"#{release_url}\" -o #{docker_compose_path} && chmod +x #{docker_compose_path}"
    action :run
    not_if "#{docker_compose_path} -v | grep #{expected_ver},"
    not_if { ::File.exist?(docker_compose_path) } unless auto_upgrade
  end

  # for sudo secure_path on RHEL family OS
  link docker_compose_link do
    to docker_compose_path
    only_if { node['platform_family'] == 'rhel' }
  end
when 'os-repository'
  act = auto_upgrade ? :upgrade : :install
  package 'docker-compose' do
    action act
  end
when 'pypi'
  expected_ver = node['docker-grid']['compose']['version']
  expected_ver = '' if expected_ver.nil?
  pin_ver = expected_ver.empty? ? '' : "==#{expected_ver}"

  include_recipe 'platform_utils::pip'

  # Note: installed to /usr/local/bin/docker-compose
  execute 'pip_install_docker_compose' do
    user 'root'
    environment 'HOME' => '/root'  # = sudo -H
    command "pip install docker-compose#{pin_ver}"
    action :run
    not_if "pip list | grep docker-compose | grep '(#{expected_ver})'" unless expected_ver.empty?
    not_if 'pip list | grep docker-compose' unless auto_upgrade
  end
end
