#
# Cookbook Name:: platform_utils
# Library:: Helper
#
# Copyright 2016-2017, whitestar
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

require 'shellwords'

module PlatformUtils
  # Helper methods
  module Helper
    def validate_shellwords(words)
      invalid_word = words.find {|word|
        word =~ /[;\|&<>`]/
      }
      unless invalid_word.nil?
        Chef::Log.fatal("Command string includes the invalid character (;|&<>`): #{invalid_word}")
        raise
      end
    end

    def touch_subid_files
      subid_files = [
        '/etc/subuid',
        '/etc/subgid',
      ]

      subid_files.each {|subid_file|
        resources(file: subid_file) rescue file subid_file do
          owner 'root'
          group 'root'
          mode '0644'
          action :touch
          not_if { File.exist?(subid_file) }
        end
      }

      subid_files
    end

    def append_subusers(users, notifies_hash = nil)
      subid_files = touch_subid_files

      this_recipe = self
      users.each {|uname|
        blk_name = "adds_subid_entries_#{uname}"
        resources(ruby_block: blk_name) rescue ruby_block blk_name do
          action :run
          not_if "cat /etc/subuid | grep #{uname}"
          not_if "cat /etc/subgid | grep #{uname}"
          notifies notifies_hash['action'], notifies_hash['resource'], notifies_hash['timer'] unless notifies_hash.nil?
          block do
            subid_files.each {|subid_file|
              max_start_id = 100_000
              offset = 0
              already_exist = false

              begin
                File.open(subid_file) {|file|
                  file.each_line {|line|
                    entry = line.split(':')
                    if entry[0] == uname
                      already_exist = true
                      break
                    end
                    if entry[1].to_i >= max_start_id
                      max_start_id = entry[1].to_i
                      offset = entry[2].to_i
                    end
                  }
                }

                if already_exist
                  this_recipe.log "#{uname} already exists in #{subid_file}"
                else
                  File.open(subid_file, 'a') {|file|
                    entry_str = "#{uname}:#{max_start_id + offset}:65536"
                    this_recipe.log "#{uname} (#{entry_str}) is added in #{subid_file}"
                    file.puts entry_str
                  }
                end
              rescue IOError => e
                puts e
              end
            }
          end
        end
      }
    end
  end
end
