/*
 * Copyright (C) 2004-2013 L2J DataPack
 * 
 * This file is part of L2J DataPack.
 * 
 * L2J DataPack is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * L2J DataPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package handlers.effecthandlers;

import com.l2jserver.gameserver.datatables.SkillTable;
import com.l2jserver.gameserver.enums.InstanceType;
import com.l2jserver.gameserver.handler.ITargetTypeHandler;
import com.l2jserver.gameserver.handler.TargetHandler;
import com.l2jserver.gameserver.model.L2Object;
import com.l2jserver.gameserver.model.StatsSet;
import com.l2jserver.gameserver.model.actor.L2Character;
import com.l2jserver.gameserver.model.actor.events.listeners.IDamageReceivedEventListener;
import com.l2jserver.gameserver.model.conditions.Condition;
import com.l2jserver.gameserver.model.effects.AbstractEffect;
import com.l2jserver.gameserver.model.skills.BuffInfo;
import com.l2jserver.gameserver.model.skills.L2Skill;
import com.l2jserver.gameserver.model.skills.targets.L2TargetType;
import com.l2jserver.util.Rnd;

/**
 * Trigger Skill By Damage effect implementation.
 * @author UnAfraid
 */
public class TriggerSkillByDamage extends AbstractEffect implements IDamageReceivedEventListener
{
	private final int _minAttackerLevel;
	private final int _maxAttackerLevel;
	private final int _minDamage;
	private final int _chance;
	private final int _skill;
	private final L2TargetType _targetType;
	private final InstanceType _attackerType;
	
	public TriggerSkillByDamage(Condition attachCond, Condition applyCond, StatsSet set, StatsSet params)
	{
		super(attachCond, applyCond, set, params);
		_minAttackerLevel = getParameters().getInt("minAttackerLevel", 1);
		_maxAttackerLevel = getParameters().getInt("maxAttackerLevel", 100);
		_minDamage = getParameters().getInt("minDamage", 1);
		_chance = getParameters().getInt("chance", 100);
		_skill = SkillTable.getSkillHashCode(getParameters().getInt("skillId"), getParameters().getInt("skillLevel", 1));
		_targetType = getParameters().getEnum("targetType", L2TargetType.class, L2TargetType.SELF);
		_attackerType = getParameters().getEnum("attackerType", InstanceType.class, InstanceType.L2Character);
	}
	
	@Override
	public void onDamageReceivedEvent(L2Character attacker, L2Character target, double damage, L2Skill skill, boolean crit)
	{
		if ((_chance == 0) || (SkillTable.getSkillLevel(_skill) == 0))
		{
			return;
		}
		
		final ITargetTypeHandler targetHandler = TargetHandler.getInstance().getHandler(_targetType);
		if (targetHandler == null)
		{
			_log.warning("Handler for target type: " + _targetType + " does not exist.");
			return;
		}
		
		if (attacker == target)
		{
			return;
		}
		
		if ((attacker.getLevel() < _minAttackerLevel) || (attacker.getLevel() > _maxAttackerLevel))
		{
			return;
		}
		
		if ((damage < _minDamage) || (Rnd.get(100) > _chance) || !attacker.getInstanceType().isType(_attackerType))
		{
			return;
		}
		
		final L2Skill triggerSkill = SkillTable.getSkill(_skill);
		final L2Object[] targets = targetHandler.getTargetList(triggerSkill, target, false, attacker);
		for (L2Object triggerTarget : targets)
		{
			if ((triggerTarget == null) || !triggerTarget.isCharacter())
			{
				continue;
			}
			
			final L2Character targetChar = (L2Character) triggerTarget;
			if (!targetChar.isInvul())
			{
				target.makeTriggerCast(triggerSkill, targetChar);
			}
		}
	}
	
	@Override
	public void onExit(BuffInfo info)
	{
		info.getEffected().getEvents().unregisterListener(this);
	}
	
	@Override
	public void onStart(BuffInfo info)
	{
		info.getEffected().getEvents().registerListener(this);
	}
}
