package jp.kirikiri.tvp2.base;

import java.util.HashMap;

import jp.kirikiri.tjs2.BinaryStream;
import jp.kirikiri.tjs2.TJSException;
import jp.kirikiri.tvp2.msg.Message;

public abstract class Archive {

	private HashMap<String, Integer> mHash;
	private boolean mInit;
	private String mArchiveName;

	//-- constructor
	public Archive( final String name ) {
		mArchiveName = name;
		mInit = false;
	}

	public abstract int getCount();

	/**
	 * @return name must be already normalized using NormalizeInArchiveStorageName
	 * and the index must be sorted by its name.
	 * this is needed by fast directory search.
	 */
	public abstract String getName( int idx);

	public abstract BinaryStream createStreamByIndex( int idx );


	public static String normalizeInArchiveStorageName( final String name ) {
		// normalization of in-archive storage name does :
		if( name == null || name.length() == 0 ) return null;

		// make all characters small
		// change '\\' to '/'
		String tmp = name.toLowerCase();
		tmp = tmp.replace('\\','/');

		// eliminate duplicated slashes
		char[] ptr = tmp.toCharArray();
		final int len = ptr.length;
		int dest = 0;
		for( int i = 0; i < len; ) {
			if( ptr[i] != '/' ) {
				ptr[dest] = ptr[i];
				i++;
				dest++;
			} else {
				if( i != 0 ) {
					ptr[dest] = ptr[i];
					i++;
					dest++;
				}
				while( i < len && ptr[i] == '/' ) i++;
			}
		}
		return new String( ptr, 0, dest );
	}

	private void addToHash() {
		// enter all names to the hash table
		final int count = getCount();
		for( int i = 0; i < count; i++ ) {
			String name = getName(i);
			name = normalizeInArchiveStorageName(name);
			mHash.put( name, i );
		}
	}
	public BinaryStream createStream( final String name ) throws TJSException {
		if( name == null || name.length() == 0 ) return null;

		if( !mInit ) {
			mInit = true;
			addToHash();
		}

		Integer p = mHash.get(name);
		if( p == null ) Message.throwExceptionMessage( Message.StorageInArchiveNotFound, name, mArchiveName );
		return createStreamByIndex(p);

	}
	public boolean isExistent( final String name ) {
		if( name == null | name.length() == 0 ) return false;
		if( !mInit ) {
			mInit = true;
			addToHash();
		}
		return mHash.get(name) != null;
	}

	/**
	 * the item must be sorted by operator < , otherwise this function
	 * will not work propertly.
	 * @return first index which have 'prefix' at start of the name.
	 * @return -1 if the target is not found.
	 */
	public int getFirstIndexStartsWith( final String prefix ) {
		int total_count = getCount();
		int s = 0, e = total_count;
		while( e - s > 1 ) {
			int m = (e + s) / 2;
			if( !(getName(m).compareTo(prefix) < 0) ) {
				// m is after or at the target
				e = m;
			} else {
				// m is before the target
				s = m;
			}
		}

		// at this point, s or s+1 should point the target.
		// be certain.
		if( s >= total_count) return -1; // out of the index
		if( getName(s).startsWith(prefix) ) return s;
		s++;
		if( s >= total_count ) return -1; // out of the index
		if( getName(s).startsWith(prefix) ) return s;
		return -1;
	}
}
