package jp.kirikiri.tvp2.base;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;

import jp.kirikiri.tjs2.BinaryStream;
import jp.kirikiri.tjs2.Error;
import jp.kirikiri.tjs2.TJSException;
import jp.kirikiri.tvp2.msg.Message;
import jp.kirikiri.tvp2.utils.Random;

public class LocalFileStream extends BinaryStream {

	private RandomAccessFile mRandomAccessFile;
	private FileChannel mFilcChannel;

	public LocalFileStream( final String origname, final String localname, int flag ) throws TJSException {
		try {
			int access = flag & ACCESS_MASK;
			switch( access ) {
			case READ:
				mRandomAccessFile= new RandomAccessFile(localname,"r");
				mFilcChannel = mRandomAccessFile.getChannel();
				break;
			case WRITE: {
				File file = new File(localname);
				if( file.exists() ) {
					if( file.isFile() ) {
						file.delete();
					}
				}
				mRandomAccessFile = new RandomAccessFile(localname,"w");
				mFilcChannel = mRandomAccessFile.getChannel();
				break;
			}
			case APPEND:
				mRandomAccessFile = new RandomAccessFile(localname,"w");
				mFilcChannel = mRandomAccessFile.getChannel();
				break;
			case UPDATE: {
				File file = new File(localname);
				if( file.exists() == false ) {
					Message.throwExceptionMessage( Message.CannotOpenStorage, origname );
				}
				mRandomAccessFile = new RandomAccessFile(localname,"rw");
				mFilcChannel = mRandomAccessFile.getChannel();
				break;
			}
			}
			if( access == APPEND ) {
				mFilcChannel.position(mFilcChannel.size());
			}
		} catch (FileNotFoundException e) {
			Message.throwExceptionMessage( Message.CannotOpenStorage, origname );
		} catch (IOException e) {
			Message.throwExceptionMessage( Message.CannotOpenStorage, origname );
		}
		// push current tick as an environment noise
		Random.updateEnvironNoiseForTick();
	}

	protected void finalize() {
		if( mRandomAccessFile != null ) {
			close();
		}
		// push current tick as an environment noise
		// (timing information from file accesses may be good noises)
		Random.updateEnvironNoiseForTick();
	}
	@Override
	public long seek(long offset, int whence) throws TJSException {
		try {
			if( offset == 0 ) {
				return mFilcChannel.position();
			} else {
				switch(whence){
				case SEEK_SET:
					mFilcChannel.position( offset );
					break;
				case SEEK_CUR:
					mFilcChannel.position( mFilcChannel.position() + offset );
					break;
				case SEEK_END:
					mFilcChannel.position( mFilcChannel.size() + offset );
					break;
				}
				return mFilcChannel.position();
			}
		} catch (IOException e) {
			throw new TJSException( Error.SeekError );
		}
	}

	@Override
	public int read(ByteBuffer buffer) {
		try {
			return mFilcChannel.read(buffer);
		} catch (IOException e) {
			return 0;
		}
	}

	@Override
	public int write(ByteBuffer buffer) {
		try {
			return mFilcChannel.write(buffer);
		} catch (IOException e) {
			return 0;
		}
	}
	@Override
	public long getPosition() {
		try {
			return mFilcChannel.position();
		} catch (IOException e) {
			return 0;
		}
	}

	@Override
	public void setPosition( long pos ) throws TJSException {
		try {
			mFilcChannel.position(pos);
		} catch (IOException e) {
			throw new TJSException( Error.SeekError );
		}
	}

	@Override
	public void close() {
		if( mRandomAccessFile == null ) return;
		try {
			mRandomAccessFile.close();
			mRandomAccessFile = null;
			mFilcChannel = null;
		} catch (IOException e) {
		}
	}

	public void setEndOfStorage() throws TJSException {
		Message.throwExceptionMessage(Message.WriteError);
	}

	public long getSize() {
		try {
			return mFilcChannel.size();
		} catch (IOException e) {
			return 0;
		}
	}

	FileChannel getHandle() { return mFilcChannel; }
}
