package jp.kirikiri.tvp2.visual;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.CharBuffer;
import java.nio.IntBuffer;
import java.util.Arrays;

import jp.kirikiri.tjs2.BinaryStream;
import jp.kirikiri.tjs2.TJSException;
import jp.kirikiri.tvp2.msg.Message;

/**
 * プリレンダリング済みフォントクラス
 */
public class PrerenderedFont {

	private static final byte[] HEADER = {
		'T', 'V', 'P', ' ', 'p', 'r', 'e', '-', 'r', 'e', 'n', 'd', 'e', 'r', 'e', 'd', ' ', 'f', 'o', 'n', 't', 0x1a
	};
	private static final int SKIP_SIZE = 24;

	private int mVersion;
	private int mIndexCount;

	private char[] mChIndex;
	//private byte[] mData;
	private int mDataOffset;
	private ByteBuffer mDataBuffer;

	static private final int CHARACTER_ITEM_SIZE = 20;
	static public class CharacterItem {
		public int Offset;
		public short Width;
		public short Height;
		public short OriginX;
		public short OriginY;
		public short IncX;
		public short IncY;
		public short Inc;
		public short Reserved;
	}
	private CharacterItem mWork;
	public PrerenderedFont( BinaryStream input ) throws TJSException {
		mWork = new CharacterItem();
		load( input );
	}
	private void load( BinaryStream src ) throws TJSException {
		byte[] buff = new byte[22];
		src.setPosition(0);
		src.read(buff);
		if( Arrays.equals( HEADER, buff ) == false ) {
			Message.throwExceptionMessage(Message.PrerenderedFontMappingFailed, "Signature not found or invalid pre-rendered font file.");
		}
		src.read(buff,0,2);

		if(buff[1] != 2) {
			Message.throwExceptionMessage(Message.PrerenderedFontMappingFailed,"Not a 16-bit UNICODE font file.");
		}
		mVersion = buff[0];
		if( mVersion != 0 && mVersion != 1 ) {
			Message.throwExceptionMessage(Message.PrerenderedFontMappingFailed, "Invalid header version.");
		}
		int length = (int) (src.getSize() - SKIP_SIZE);
		mDataBuffer = ByteBuffer.allocateDirect(length);
		src.read(mDataBuffer);
		ByteBuffer bb = mDataBuffer;
		bb.order(ByteOrder.LITTLE_ENDIAN);
		bb.clear();
		mDataBuffer = bb;
		mIndexCount = bb.getInt();
		int chOffset = bb.getInt();
		mDataOffset = bb.getInt();

		bb.position(chOffset - SKIP_SIZE);
		CharBuffer cb = bb.asCharBuffer();
		mChIndex = new char[mIndexCount];
		cb.get(mChIndex);
	}
	public CharacterItem getCharData( char ch ) {
		int idx = findCh( ch );
		if( idx < 0 ) return null;

		int offset = mDataOffset + idx * CHARACTER_ITEM_SIZE - SKIP_SIZE;
		ByteBuffer bb = mDataBuffer;
		bb.position(offset);
		CharacterItem ret = mWork;
		ret.Offset = bb.getInt();
		ret.Width = bb.getShort();
		ret.Height = bb.getShort();
		ret.OriginX = bb.getShort();
		ret.OriginY = bb.getShort();
		ret.IncX = bb.getShort();
		ret.IncY = bb.getShort();
		ret.Inc = bb.getShort();
		ret.Reserved = bb.getShort();

		return ret;
	}
	private int findCh( char ch ) {
		// search through ChIndex
		int s = 0;
		int e = mIndexCount;
		final char[] chindex = mChIndex;
		while( true ) {
			int d = e-s;
			if( d <= 1 ) {
				if(chindex[s] == ch)
					return s;
				else
					return -1;
			}
			int m = s + (d>>1);
			int c = ((int)chindex[m])&0xffff;
			if( c > ch) e = m; else s = m;
		}
	}
	public void retrieve( final CharacterItem item, byte[] buffer, int bufferpitch ) {
		// retrieve font data and store to buffer
		// bufferpitch must be larger then or equal to item.Width
		if(item.Width == 0 || item.Height == 0) return;

		final int w = item.Width;
		int ptr = item.Offset - SKIP_SIZE;
		int dest = 0;
		int destlim = w * item.Height;

		ByteBuffer bb = mDataBuffer;
		// expand compressed character bitmap data
		if( mVersion == 0 ) {
			// version 0 decompressor
			while( dest < destlim ) {
				if( bb.get(ptr) == 0x41) { // running
					ptr++;
					byte last = buffer[dest-1];
					int len = bb.get(ptr);
					ptr++;
					while(len > 0) {
						len--;
						buffer[dest] = last;
						dest++;
					}
				} else {
					buffer[dest] = bb.get(ptr);
					dest++;
					ptr++;
				}
			}
		} else if( mVersion >= 1 ) {
			// version 1+ decompressor
			while( dest < destlim ) {
				int p = ((int)bb.get(ptr))&0xff;
				if( p >= 0x41 ) { // running
					int len = p - 0x40;
					ptr++;
					byte last = buffer[dest-1];
					while(len > 0) {
						len--;
						buffer[dest] = last;
						dest++;
					}
				} else {
					buffer[dest] = bb.get(ptr);
					dest++;
					ptr++;
				}
			}
		}

		// expand to each pitch
		ptr = destlim - w;
		dest = bufferpitch * item.Height - bufferpitch;
		byte[] work = new byte[w];
		while( dest >= 0 ) {
			if( dest != ptr ) {
				System.arraycopy( buffer, ptr, work, 0, w );
				System.arraycopy( work, 0, buffer, dest, w );
			}
			dest -= bufferpitch;
			ptr -= w;
		}
	}
}
