package jp.kirikiri.tvp2.env;

import java.awt.AlphaComposite;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;

import jp.kirikiri.tvp2.visual.DivisibleData;

/**
 * 画面の中心で回転しながらズームイン、あるいはズームアウトするトランジション
 * (KAGでは)裏画面が常に回転してズームイン、あるいはズームアウトする
 */
public class RotateZoomTransHandler extends BaseRotateTransHandler {

	private double mFactor; // 初期ズーム拡大率
	private double mTargetFactor; // 最終ズーム拡大率
	private double mAccel; // 加速度的な動きを行わせるかどうか ( 0 = 行わない )
	private double mTwist; // 初期回転位置
	private double mTwistAccel; // 回転に加速度的な動きを行わせるかどうか
	private int mCenterX; // 回転中心 X 位置
	private int mCenterY; // 回転中心 Y 位置
	private boolean mFixSrc1; // src1 を固定するか

	private AffineTransform mSrc1Mat;
	private AffineTransform mSrc2Mat;

	public RotateZoomTransHandler(long time, int width, int height, double factor, double targetfactor, double accel, double twist, double twistaccel, int centerx, int centery, boolean fixsrc1 ) {
		super(time, width, height, 0);
		mFactor = factor;
		mTargetFactor = targetfactor;
		mAccel = accel;
		mTwist = twist;
		mTwistAccel = twistaccel;
		mCenterX = centerx;
		mCenterY = centery;
		mFixSrc1 = fixsrc1;
		mSrc1Mat = new AffineTransform();
	}

	@Override
	void calcPosition() {
		// src1, src2 の画面位置を設定する

		// src1
		// src1 は常に画面全体固定

		// src2
		float zm = (float)(int)mCurTime / (float)(int)mTime;
		float tm = zm;
		if(mAccel < 0) {
			// 上弦 ( 最初が動きが早く、徐々に遅くなる )
			zm = 1.0f - zm;
			zm = (float) Math.pow(zm, -mAccel);
			zm = 1.0f - zm;
		} else if(mAccel > 0) {
			// 下弦 ( 最初は動きが遅く、徐々に早くなる )
			zm = (float) Math.pow(zm, mAccel);
		}

		float scx = mWidth/2.0f;
		float scy = mHeight/2.0f;
		float cx = ((scx - mCenterX) * zm + mCenterX);
		float cy = ((scy - mCenterY) * zm + mCenterY);

		if( mTwistAccel < 0 ) {
			// 上弦 ( 最初が動きが早く、徐々に遅くなる )
			tm = 1.0f - tm;
			tm = (float) Math.pow(tm, -mTwistAccel);
			tm = 1.0f - tm;
		} else if( mTwistAccel > 0 ) {
			// 下弦 ( 最初は動きが遅く、徐々に早くなる )
			tm = (float) Math.pow(tm, mTwistAccel);
		}

		float rad = (float) (mCurTime == mTime ? 0 : 2 * Math.PI * mTwist * tm);
		zm = (float) ((mTargetFactor - mFactor) * zm + mFactor);

		AffineTransform rot = AffineTransform.getRotateInstance( -rad, cx, cy );
		AffineTransform scale = AffineTransform.getScaleInstance(zm, zm);
		scale.concatenate(rot);
		if( zm != 0.0f && zm != 1.0f ) {
			cx = cx - cx * zm;
			cy = cy - cy * zm;
			AffineTransform trans = AffineTransform.getTranslateInstance(cx, cy);
			trans.concatenate(scale);
			mSrc2Mat = trans;
		} else {
			mSrc2Mat = scale;
		}
	}

	@Override
	public int process(DivisibleData data) {
		BufferedImage dest = (BufferedImage)data.Dest.getScanLineForWrite().getImage();
		BufferedImage src1 = (BufferedImage)data.Src1.getScanLine().getImage();
		BufferedImage src2 = (BufferedImage)data.Src2.getScanLine().getImage();
		int destLeft = data.DestLeft;
		int src1Left = data.Src1Left;
		int src2Left = data.Src2Left;
		int h = data.Height;
		int w = data.Width;
		int destTop = data.DestTop;
		int src1Top = data.Src1Top;
		int src2Top = data.Src2Top;

		Graphics2D g = (Graphics2D)dest.getGraphics();
		g.setRenderingHint(RenderingHints.KEY_INTERPOLATION , RenderingHints.VALUE_INTERPOLATION_NEAREST_NEIGHBOR ); // ニアレストネイバー
		g.setComposite( AlphaComposite.Src );
		g.setTransform(mSrc1Mat);
		if( mFixSrc1 ) {
			g.drawImage( src1, destLeft, destTop, destLeft+w, destTop+h, src1Left, src1Top, src1Left+w, src1Top+h, null );
		} else {
			g.drawImage( src2, destLeft, destTop, destLeft+w, destTop+h, src1Left, src1Top, src1Left+w, src1Top+h, null );
		}

		g.setTransform(mSrc2Mat);
		if( mFixSrc1 ) {
			g.drawImage( src2, destLeft, destTop, destLeft+w, destTop+h, src2Left, src2Top, src2Left+w, src2Top+h, null );
		} else {
			g.drawImage( src1, destLeft, destTop, destLeft+w, destTop+h, src2Left, src2Top, src2Left+w, src2Top+h, null );
		}
		g.dispose();

		return S_OK;
	}
}
