package jp.kirikiri.tjs2;

import java.nio.ByteBuffer;
import java.util.ArrayList;

// a class for managing the script block
public class ScriptBlock implements SourceCodeAccessor {
	private static final boolean D_IS_DISASSEMBLE = false;

	private static final String NO_SCRIPT = "no script";

	private TJS mOwner;

	private InterCodeObject mTopLevelObject;
	private ArrayList<InterCodeObject> mInterCodeObjectList;

	// 以下の4つは実行時にいるかな、名前以外はエラー発生時に必要になるだけだろうけど。
	private String mName;
	private int mLineOffset;
	private String mScript;
	private ScriptLineData mLineData;

	public ScriptBlock( TJS owner, final String name, int lineoffset, final String script, ScriptLineData linedata ) {
		mOwner = owner;
		mName = name;
		mLineOffset = lineoffset;
		mScript = script;
		mLineData = linedata;

		mOwner.addScriptBlock(this);
	}
	public void setObjects( InterCodeObject toplevel, ArrayList<InterCodeObject> objs ) {
		mTopLevelObject = toplevel;
		mInterCodeObjectList = objs;
	}

	public ScriptBlock( TJS owner ) {
		mOwner = owner;

		// Java で初期値となる初期化は省略
		//mScript = null;
		//mName = null;
		//mInterCodeContext = null;
		//mTopLevelContext = null;
		//mLexicalAnalyzer = null;
		//mUsingPreProcessor = false;
		//mLineOffset = 0;
		//mCompileErrorCount = 0;
		//mNode = null;

		mOwner.addScriptBlock(this);
	}
	protected void finalize() {
		if( mTopLevelObject != null ) mTopLevelObject = null;
		mOwner.removeScriptBlock(this);

		if( mScript != null ) mScript = null;
		if( mName != null ) mName = null;;

		try {
			super.finalize();
		} catch (Throwable e) {
		}
	}

	public int srcPosToLine( int pos ) {
		if( mLineData == null ) return 0;
		return mLineData.getSrcPosToLine(pos);
	}

	public TJS getTJS() {
		return mOwner;
	}

	public String getName() { return mName; }
	/*
	public void setName( String name, int lineofs ) {
		mName = null;
		if( name != null ) {
			mLineOffset = lineofs;
			mName = new String(name);
		}
	}
	*/
	public int getLineOffset() { return mLineOffset; }

	public void executeTopLevel( Variant result, Dispatch2 context ) throws VariantException, TJSException, CompileException {
		// compiles text and executes its global level scripts.
		// the script will be compiled as an expression if isexpressn is true.


		// 逆アセンブル
		if( D_IS_DISASSEMBLE ) {
			final int count = mInterCodeObjectList.size();
			for( int i = 0; i < count; i++ ) {
				InterCodeObject v = mInterCodeObjectList.get(i);
				TJS.outputToConsole( v.getName() );
				v.disassemble( this, 0, 0 );
			}
		}

		// execute global level script
		executeTopLevelScript(result, context);

		final int context_count = mInterCodeObjectList.size();
		if( context_count != 1 ) {
			// this is not a single-context script block
			// (may hook itself)
			// release all contexts and global at this time
			InterCodeObject toplevel = mTopLevelObject;
			if( mTopLevelObject != null ) mTopLevelObject = null;
			remove(toplevel);
		}
	}



	public void executeTopLevelScript(Variant result, Dispatch2 context) throws VariantException, TJSException {
		if( mTopLevelObject != null ) {
			mTopLevelObject.funcCall( 0, null, result, null, context );
		}
	}

	public String getLineDescriptionString(int pos) {
		// get short description, like "mainwindow.tjs(321)"
		// pos is in character count from the first of the script
		StringBuilder builer = new StringBuilder(512);
		int line = srcPosToLine(pos)+1;
		if( mName != null ) {
			builer.append( mName );
		} else {
			builer.append("anonymous@");
			builer.append( toString() );
		}
		builer.append('(');
		builer.append( String.valueOf(line) );
		builer.append(')');
		return builer.toString();
	}

	public int lineToSrcPos(int line) {
		if( mLineData == null ) return 0;
		// assumes line is added by LineOffset
		line -= mLineOffset;
		return mLineData.getLineToSrcPos(line);
	}

	public String getLine(int line ) {
		if( mLineData == null ) return NO_SCRIPT;
		// note that this function DOES matter LineOffset
		line -= mLineOffset;
		return mLineData.getLine(line);
	}

	/*
	public boolean isReusable() {
		return getContextCount() == 1 && mTopLevelObject != null && !mUsingPreProcessor;
	}
	*/
	public boolean isReusable() {
		return getContextCount() == 1 && mTopLevelObject != null;
	}

	public int getContextCount() {
		return mInterCodeObjectList.size();
	}

	public void add(InterCodeObject obj) {
		mInterCodeObjectList.add(obj);
	}
	public void remove(InterCodeObject obj) {
		mInterCodeObjectList.remove(obj);
	}

	public int getObjectIndex( InterCodeObject obj ) {
		return mInterCodeObjectList.indexOf(obj);
	}
	public InterCodeObject getCodeObject( int index ) {
		if( index >= 0 && index < mInterCodeObjectList.size() ) {
			return mInterCodeObjectList.get(index);
		} else {
			return null;
		}
	}

	private void exportByteCode( BinaryStream output ) throws TJSException {
		//BinaryStream output = TJS.mStorage.createBinaryWriteStream(filename);
		byte[] filetag = { 'T', 'J', 'S', '2' };
		byte[] objtag = { 'O', 'B', 'J', 'S' };
		byte[] datatag = { 'D', 'A', 'T', 'A' };

		final int count = mInterCodeObjectList.size();
		ArrayList<ByteBuffer> objarray = new ArrayList<ByteBuffer>(count*2);
		ConstArrayData constarray = new ConstArrayData();
		int objsize = 0;
		for( int i = 0; i < count; i++ ) {
			InterCodeObject obj = mInterCodeObjectList.get(i);
			ByteBuffer buf = obj.exportByteCode(this,constarray);
			objarray.add(buf);
			objsize += buf.capacity() + 4 + 4; // tag + size
		}
		objsize += 4 + 4 + 4; // OBJS tag + size + count
		ByteBuffer dataarea = constarray.exportBuffer();
		int datasize = dataarea.capacity() + 4 + 4; // DATA tag + size
		int filesize = objsize + datasize + 4 + 4; // TJS2 tag + file size
		byte[] filesizearray = { (byte) (filesize&0xff), (byte) ((filesize>>>8)&0xff), (byte) ((filesize>>>16)&0xff), (byte) ((filesize>>>24)&0xff) };
		byte[] datasizearray = { (byte) (datasize&0xff), (byte) ((datasize>>>8)&0xff), (byte) ((datasize>>>16)&0xff), (byte) ((datasize>>>24)&0xff) };
		byte[] objsizearray = { (byte) (objsize&0xff), (byte) ((objsize>>>8)&0xff), (byte) ((objsize>>>16)&0xff), (byte) ((objsize>>>24)&0xff) };
		byte[] objcountarray = { (byte) (count&0xff), (byte) ((count>>>8)&0xff), (byte) ((count>>>16)&0xff), (byte) ((count>>>24)&0xff) };

		output.write(filetag);
		output.write(filesizearray);
		output.write(datatag);
		output.write(datasizearray);
		output.write(dataarea);
		output.write(objtag);
		output.write(objsizearray);
		output.write(objcountarray);
		for( int i = 0; i < count; i++ ) {
			ByteBuffer buf = objarray.get(i);
			int size = buf.capacity();
			byte[] bufsizearray = { (byte) (size&0xff), (byte) ((size>>>8)&0xff), (byte) ((size>>>16)&0xff), (byte) ((size>>>24)&0xff) };
			output.write(filetag);
			output.write(bufsizearray);
			output.write(buf);
		}
		output.close();
		output = null;
		objarray.clear();
		objarray = null;
		constarray = null;
		dataarea = null;
	}


	public String getNameInfo() {
		if( mLineOffset == 0 ) {
			return new String(mName);
		} else {
			return mName + "(line +" + String.valueOf(mLineOffset) + ")";
		}
	}

	public int getTotalVMCodeSize() {
		int size = 0;
		final int count = mInterCodeObjectList.size();
		for( int i = 0; i < count; i++ ) {
			size += mInterCodeObjectList.get(i).getCodeSize();
		}
		return size;
	}

	public int getTotalVMDataSize() {
		int size = 0;
		final int count = mInterCodeObjectList.size();
		for( int i = 0; i < count; i++ ) {
			size += mInterCodeObjectList.get(i).getDataSize();
		}
		return size;
	}

	static public void consoleOutput( final String msg, ScriptBlock blk  ) {
		TJS.outputToConsole(msg);
	}
	public void dump() throws VariantException {
		final int count = mInterCodeObjectList.size();
		for( int i = 0; i < count; i++ ) {
			InterCodeObject v = mInterCodeObjectList.get(i);
			consoleOutput( "", this );
			String ptr = String.format( " 0x%08X", v.hashCode() );
			consoleOutput( "(" + v.getContextTypeName() + ") " + v.getName() + ptr, this );
			v.disassemble( this, 0, 0 );
		}
	}

	public String getScript() {
		if( mScript != null ) return mScript;
		else return NO_SCRIPT;
	}
	@Override
	public int codePosToSrcPos(int codepos) {
		return 0; // allways 0, 基本的に使われない
	}
}
