package jp.kirikiri.tvp2.base;

import java.io.File;

import jp.kirikiri.tjs2.BinaryStream;
import jp.kirikiri.tjs2.TJSException;
import jp.kirikiri.tvp2.msg.Message;


public class FileMedia implements StorageMedia {

	private static final String SCHEME = "file";

	/**
	 * returns media name like "file", "http" etc.
	 */
	public String getName() { return SCHEME; }

	/**
	 * normalize domain name according with the media's rule
	 */
	public String normalizeDomainName( final String name ) {
		// make all characters small
		return name.toLowerCase();
	}

	/**
	 * normalize path name according with the media's rule
	 * "name" below is normalized but does not contain media, eg.
	 * not "media://domain/path" but "domain/path"
	 */
	public String normalizePathName( final String name ) {
		// make all characters small
		return name.toLowerCase();
	}


	/**
	 * check file existence
	 * @throws TJSException
	 */
	public boolean checkExistentStorage( final String name ) throws TJSException {
		if( name.length() == 0 ) return false;
		String _name = getLocalName(name);
		return Storage.checkExistentLocalFile(_name);
	}

	/**
	 * open a storage and return a tTJSBinaryStream instance.
	 * @param name does not contain in-archive storage name but is normalized.
	 * @throws TJSException
	 */
	public BinaryStream open( final String name, int flags ) throws TJSException {
		// open storage named "name".
		// currently only local/network(by OS) storage systems are supported.
		if( name == null | name.length() == 0 )
			Message.throwExceptionMessage( Message.CannotOpenStorage, "\"\"" );

		String origname = name;
		String _name = getLocalName(name);
		return new LocalFileStream( origname, _name, flags );
	}

	/**
	 * list files at given place
	 * @throws TJSException
	 */
	public void getListAt( final String name, StorageLister lister ) throws TJSException {
		String localName = getLocalName(name);
		File file = new File(localName);
		if( file.isDirectory() ) {
			File[] files = file.listFiles();
			for( File f : files ) {
				if( f.isFile() ) {
					lister.add( f.getName().toLowerCase() );
				}
			}
		}
	}

	/**
	 * basically the same as above,
	 * check wether given name is easily accessible from local OS filesystem.
	 * if true, returns local OS native name. otherwise returns an empty string.
	 */
	public String getLocallyAccessibleName( final String name ) {
		StringBuilder newname = new StringBuilder(256);
		if( name.startsWith("./") == false ) {
			// differs from "./",
			// this may be a UNC file name.
			// UNC first two chars must be "\\\\" ?
			// AFAIK 32-bit version of Windows assumes that '/' can be used as a path
			// delimiter. Can UNC "\\\\" be replaced by "//" though ?

			// UNIX 系の場合は、絶対パスとみなして'/'をつける
			newname.append( File.separatorChar );
			if( File.separatorChar == '\\' ) {
				newname.append( File.separatorChar );
			}
			newname.append( name );
		} else {
			if( name.length() == 2 ) {
				// newname.append( "" );
			} else {
				final char ch = name.charAt(2);
				if( ch < 'a' || ch > 'z' ) {
					// newname.append( "" );
				} else {
					char ch3 = name.charAt(3);
					if( ch3 != '/' ) {
						// newname.append( "" );
					} else {
						newname.append( ch );
						newname.append( ':' );
						newname.append( name.substring(3) );
					}
				}
			}
		}
		if( newname.length() == 0 ) {
			return null;
		}

		// Win では '/' を '\\' へ
		// その他では、'\\' を '/' へ
		char replaceCh = '\\';
		if( File.separatorChar == '\\' ) {
			replaceCh = '/';
		}
		String pp = newname.toString();
		return pp.replace( replaceCh, File.separatorChar );
	}

	public String getLocalName( final String name ) throws TJSException {
		String tmp = getLocallyAccessibleName(name);
		if( tmp == null | tmp.length() == 0 ) Message.throwExceptionMessage( Message.CannotGetLocalName, name );
		return tmp;
	}
}
