package jp.kirikiri.tvp2.base;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;

import jp.kirikiri.tjs2.BinaryStream;
import jp.kirikiri.tjs2.Error;
import jp.kirikiri.tjs2.LexBase;
import jp.kirikiri.tjs2.TJSException;
import jp.kirikiri.tvp2.TVP;
import jp.kirikiri.tvp2.msg.Message;


public class Storage {

	public static final char ArchiveDelimiter = '>';

	public static boolean checkExistentLocalFile( final String name ) {
		File file = new File(name);
		if( file.exists() ) {
			if( file.isFile() ) {
				return true;
			}
		}
		file = null;
		return false;
	}

	public static String extractStorageExt( final String name ) {
		// extract an extension from name.
		// returned string will contain extension delimiter ( '.' ), except for
		// missing extension of the input string.
		// ( returns null string when input string does not have an extension )

		final int slen = name.length();
		int p = slen - 1;
		while( p >= 0 ){
			char c = name.charAt(p);
			if(c == '\\') break;
			if(c == '/') break;
			if(c == ArchiveDelimiter) break;
			if(c == '.') {
				// found extension delimiter
				//int extlen = slen - p;
				return name.substring( p );
			}
			p--;
		}
		// not found
		return new String();
	}
	public static String extractStorageName( final String name ) {
		// extract "name"'s storage name ( excluding path ) and return it.
		final int slen = name.length();
		int p = slen - 1;
		while( p >= 0 ){
			char c = name.charAt(p);
			if(c == '\\') break;
			if(c == '/') break;
			if(c == ArchiveDelimiter) break;
			p--;
		}
		p++;
		if( p <= 0 ) {
			return name;
		} else {
			return name.substring(p);
		}
	}
	public static String extractStoragePath( final String name ) {
		// extract "name"'s path ( including last delimiter ) and return it.
		final int slen = name.length();
		int p = slen;
		p--;
		while( p >= 0 ) {
			char c = name.charAt(p);
			if( c == '\\' ) break;
			if( c == '/' ) break;
			if( c == ArchiveDelimiter ) break;
			p--;
		}
		p++;
		return name.substring(0,p);
	}
	public static String chopStorageExt(String name) {
		// chop storage's extension and return it.
		int slen = name.length();
		int p = slen;
		p--;
		while( p >= 0 ) {
			char c = name.charAt(p);
			if( c == '\\' ) break;
			if( c == '/' ) break;
			if( c == ArchiveDelimiter) break;
			if( c == '.' ) {
				// found extension delimiter
				return name.substring(0,p);
			}
			p--;
		}
		// not found
		return name;
	}

	public static String searchPlacedPath( final String name ) throws TJSException {
		String place = getPlacedPath(name);
		if( place == null || place.length() == 0) Message.throwExceptionMessage( Message.CannotFindStorage, name);
		return place;
	}

	/**
	 * search path and return the path which the "name" is placed.
	 * returned name is normalized. returns empty string if the storage is not
	 * found.
	 * @throws TJSException
	 */
	public static String getPlacedPath( final String name ) throws TJSException {
		return TVP.AutoPath.getPlacedPath(name);
	}
	public static boolean isExistentStorage( final String name ) throws TJSException {
		String path = TVP.AutoPath.getPlacedPath(name);
		return( path != null && path.length() > 0 );
	}

	public static boolean isExistentStorageNoSearch(String name) throws TJSException {
		return isExistentStorageNoSearchNoNormalize( TVP.StorageMediaManager.normalizeStorageName(name,null) );
	}

	public static boolean isExistentStorageNoSearchNoNormalize( String name  ) throws TJSException {
		// does name contain > ?
		synchronized (TVP.CreateStreamCS) {
			final int sharp_pos = name.indexOf( ArchiveDelimiter );
			if( sharp_pos != -1 ) {
				// this storagename indicates a file in an archive
				String arcname = name.substring(0, sharp_pos);
				Archive arc = TVP.ArchiveCache.get(arcname);
				boolean ret;
				try {
					String in_arc_name = name.substring(sharp_pos + 1);
					in_arc_name = Archive.normalizeInArchiveStorageName(in_arc_name);
					ret = arc.isExistent(in_arc_name);
				} finally {
					arc = null;
				}
				return ret;
			}

			return TVP.StorageMediaManager.checkExistentStorage(name);
		}
	}
	public static BinaryStream createStream( final String _name, int flags ) throws TJSException {
		try {
			synchronized (TVP.CreateStreamCS) {
				String name;
				int access = flags & BinaryStream.ACCESS_MASK;
				if( access == BinaryStream.WRITE )
					name = TVP.StorageMediaManager.normalizeStorageName(_name,null);
				else
					name = TVP.AutoPath.getPlacedPath(_name); // file must exist

				if( name == null || name.length() == 0 ) Message.throwExceptionMessage(Message.CannotOpenStorage, _name);

				// does name contain > ?
				int sharp_pos = name.indexOf(ArchiveDelimiter);
				if( sharp_pos != -1 ) {
					// this storagename indicates a file in an archive
					if( access != BinaryStream.READ )
						Message.throwExceptionMessage(Message.CannotWriteToArchive);

					String arcname = name.substring( 0, sharp_pos );

					Archive arc;
					BinaryStream stream;
					arc = TVP.ArchiveCache.get(arcname);
					try {
						String in_arc_name = name.substring(sharp_pos + 1);
						in_arc_name = Archive.normalizeInArchiveStorageName(in_arc_name);
						stream = arc.createStream(in_arc_name);
					} finally {
						arc = null;
						if(access >= 1) clearStorageCaches();
					}
					return stream;
				}

				BinaryStream stream;
				try {
					stream = TVP.StorageMediaManager.open(name, flags);
				} finally {
					if(access >= 1) clearStorageCaches();
				}
				return stream;
			}
		} catch( TJSException e ) {
			if( _name.indexOf('#') != -1 ) {
				String tmp = Message.FilenameContainsSharpWarn.replace( "%1", _name );
				StringBuilder builder = new StringBuilder(128);
				builder.append(e.getMessage());
				builder.append('[');
				builder.append(tmp);
				builder.append(']');
				throw new TJSException(builder.toString());
			}
			throw e;
		}
	}
	public static void clearStorageCaches() {
		// clear all storage related caches
		// TVPClearXP3SegmentCache(); TODO XP3 書いたら追加すること
		TVP.AutoPath.clearAutoPathCache();
	}
	public static void setCurrentDirectory( final String _name ) throws TJSException {
		TVP.StorageMediaManager.setCurrentDirectory(_name);
		clearStorageCaches();
	}
	public static String readText( String name , String modestr ) throws TJSException {
		BinaryStream stream = null;
		try {
			String encodeType = null;
			int CryptMode = -1;
			boolean DirectLoad = false;
			stream = createStream(name,BinaryStream.READ);
			int o_ofs = -1;
			if( modestr != null ) o_ofs = modestr.indexOf('o');
			int ofs = 0;
			if( o_ofs != -1 ) {
				// seek to offset
				StringBuilder builder = new StringBuilder(256);
				int i = o_ofs + 1;
				final int count = modestr.length();
				for( ; i < count; i++ ) {
					int ch = modestr.codePointAt(i);
					if( ch >= '0' && ch <= '9' ) {
						builder.append(ch);
					} else {
						break;
					}
				}
				LexBase lex = new LexBase( builder.toString() );
				Number num = lex.parseNumber();
				if( num != null ) {
					ofs = num.intValue();
					stream.setPosition(ofs);
				}
			}
			// check first of the file - whether the file is unicode
			ByteBuffer mark = ByteBuffer.allocate(2);
			stream.read( mark );
			if( mark.get(0) == 0xff && mark.get(1) == 0xfe ) {
				// unicode
				DirectLoad = true;
				encodeType = "UTF-16LE";
			} else if( mark.get(0) == 0xfe && mark.get(1) == 0xfe ) {
				// ciphered text or compressed
				ByteBuffer mode = ByteBuffer.allocate(1);
				stream.read( mode );
				byte m0 = mode.get(0);
				if( m0 != 0 && m0 != 1 && m0 != 2 ) {
					throw new TJSException( /*Error.UnsupportedCipherMode*/ ); // TODO エラーメッセージを作る
				}
				CryptMode = m0;
				boolean directLoad = CryptMode != 2;
				stream.read( mark ); // original bom code comes here (is not compressed)
				if(m0 != 0xff || m0 != 0xfe)
					throw new TJSException( /*Error.UnsupportedCipherMode*/ ); // TODO エラーメッセージを作る

				if( CryptMode == 2 ) {
					// compressed text stream
					// TODO 圧縮ストリームを展開する機能を実装する

				}
			} else {
				// 開き直してシークする
				stream.setPosition(ofs);
				encodeType = "MS932";
			}
			if( encodeType != null ) {
				long size = stream.getSize();
				ByteBuffer buf = ByteBuffer.allocate((int) size);
				int readSize = stream.read(buf);
				if( buf.hasArray() ) {
					String result = new String( buf.array(), 0, readSize, encodeType );
					buf = null;
					return result;
				}
			}
		} catch (IOException e) {
			throw new TJSException( Error.ReadError + e.toString() );
		} finally {
			if( stream != null ) {
				stream.close();
			}
		}
		return null;
	}


}
