<?php
	// Copyright (c) 2008, u6k.yu1@gmail.com
	// All rights reserved.
	// 
	// Redistribution and use in source and binary forms, with or without
	// modification, are permitted provided that the following conditions are met:
	//     * Redistributions of source code must retain the above copyright
	//       notice, this list of conditions and the following disclaimer.
	//     * Redistributions in binary form must reproduce the above copyright
	//       notice, this list of conditions and the following disclaimer in the
	//       documentation and/or other materials provided with the distribution.
	//     * Neither the name of the u6k nor the
	//       names of its contributors may be used to endorse or promote products
	//       derived from this software without specific prior written permission.
	// 
	// THIS SOFTWARE IS PROVIDED BY u6k.yu1@gmail.com ``AS IS'' AND ANY
	// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
	// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
	// DISCLAIMED. IN NO EVENT SHALL u6k.yu1@gmail.com BE LIABLE FOR ANY
	// DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
	// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
	// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
	// ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
	// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
	// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
	
	require_once(dirname(__FILE__) . "/../common/common.php");
	require_once("MDB2.php");
	
	/**
	 * 記事に関する機能を提供します。
	 * 
	 * @package logic
	 * @author u6k <u6k.yu1@gmail.com>
	 * @copyright Copyright (c) 2008 u6k.yu1@gmail.com, All rights reserved.
	 * @license BSD License
	 * @version $Id$
	 * @link http://hirameki-note.sourceforge.jp/ ひらめきノート
	 */
	class Hirameki_ArticleLogic {
		/**
		 * 記事を登録します。
		 * 
		 * <ol>
		 * <li>引数の確認を行います。
		 * <li>記事のIDを新規に作成します。
		 * <li>作成したID、現在日時を使用して、データベースに記事の情報を挿入します。
		 * </ol>
		 * 
		 * @param int $account_id 記事を登録するユーザのID。
		 * @param string $title 記事の件名。
		 * @param string $content 記事の内容。
		 * @param string $tag 記事のタグ。
		 * @return 登録した記事のID。
		 */
		public static function create($account_id, $title, $content, $tag) {
			/*
			 * 引数の確認を行います。
			 */
			if (!is_int($account_id)) {
				throw new ArgumentException("account_id引数の値が整数型ではありません。");
			}
			if (!is_string($title)) {
				throw new ArgumentException("title引数の値が文字列型ではありません。");
			}
			if (!empty($content)) {
				if (!is_string($content)) {
					throw new ArgumentException("content引数の値が文字列型ではありません。");
				}
			} else {
				$content = NULL;
			}
			if (!empty($tag)) {
				if (!is_string($tag)) {
					throw new ArgumentException("tag引数の値が文字列型ではありません。");
				}
			} else {
				$tag = NULL;
			}
			
			/*
			 * データベースに記事を登録します。
			 */
			global $config;
			$con = MDB2::connect($config["dsn"]);
			$con->loadModule("Extended");
			$con->beginTransaction();
			
			$id = $con->nextID("article_id");
			$now = Hirameki_DateTimeUtil::now();
			$sql = <<< SQL_END
insert into article (
	id,
	create_time,
	enable,
	create_account_id,
	title,
	content,
	tag
) values (
	:id,
	:create_time,
	:enable,
	:create_account_id,
	:title,
	:content,
	:tag
)
SQL_END;
			$param = array(
				"id" => $id,
				"create_time" => $now,
				"enable" => 1,
				"create_account_id" => $account_id,
				"title" => $title,
				"content" => $content,
				"tag" => $tag
			);
			$param_type = array(
				"id" => "integer",
				"create_time" => "integer",
				"enable" => "integer",
				"create_account_id" => "integer",
				"title" => "text",
				"content" => "text",
				"tag" => "text"
			);
			
			$result = $con->execParam($sql, $param, $param_type);
			if ($result !== 1) {
				$con->rollback();
				$con->disconnect();
				throw new SQLException("データが正しく登録されませんでした。");
			}
			
			$con->commit();
			$con->disconnect();
			
			/*
			 * 登録した記事のIDを返します。
			 */
			return $id;
		}
		
		/**
		 * 記事を更新します。
		 * 
		 * <ol>
		 * <li>引数を確認します。
		 * <li>更新する記事の最終更新の有効/無効が有効であることを確認します。
		 * <li>引数のID、現在日時を使用して、データベースに記事の情報を挿入します。
		 * </ol>
		 * 
		 * @param int $id 更新する記事のID。
		 * @param int $account_id 記事を更新するユーザのID。
		 * @param string $title 記事の件名。
		 * @param string $content 記事の内容。
		 * @param string $tag 記事のタグ。
		 * @return void
		 */
		public static function update($id, $account_id, $title, $content, $tag) {
			/*
			 * 引数を確認します。
			 */
			if (!is_int($id)) {
				throw new ArgumentException("id引数の値が整数型ではありません。");
			}
			if (!is_int($account_id)) {
				throw new ArgumentException("account_id引数の値が整数型ではありません。");
			}
			if (!is_string($title)) {
				throw new ArgumentException("title引数の値が文字列型ではありません。");
			}
			if (!empty($content)) {
				if (!is_string($content)) {
					throw new ArgumentException("content引数の値が文字列型ではありません。");
				}
			} else {
				$content = NULL;
			}
			if (!empty($tag)) {
				if (!is_string($tag)) {
					throw new ArgumentException("tag引数の値が文字列型ではありません。");
				}
			} else {
				$tag = NULL;
			}
			
			/*
			 * データベース関連を確認します。
			 */
			global $config;
			$con = MDB2::connect($config["dsn"]);
			$con->loadModule("Extended");
			
			$sql = <<< SQL_END
select
	count(article.id)
from
	article,
	(
		select
			id,
			max(create_time) as latest_time
		from
			article
		group by
			id
		having
			id = :id
	) as latest_article
where
	article.id = latest_article.id
	and article.create_time = latest_article.latest_time
	and article.enable = :enable
SQL_END;
			$type = "integer";
			$param = array("id" => $id, "enable" => 1);
			$param_type = array("id" => "integer", "enable" => "integer");
			
			$result = $con->getOne($sql, $type, $param, $param_type);
			if ($result !== 1) {
				$con->disconnect();
				throw new HiramekiException("記事「" . $id . "」は存在しないか、既に削除されています。");
			}
			
			/*
			 * 記事をデータベースに登録します。
			 */
			$con->beginTransaction();
			
			$now = Hirameki_DateTimeUtil::now();
			$sql = <<< SQL_END
insert into article (
	id,
	create_time,
	enable,
	create_account_id,
	title,
	content,
	tag
) values (
	:id,
	:create_time,
	:enable,
	:create_account_id,
	:title,
	:content,
	:tag
)
SQL_END;
			$param = array(
				"id" => $id,
				"create_time" => $now,
				"enable" => 1,
				"create_account_id" => $account_id,
				"title" => $title,
				"content" => $content,
				"tag" => $tag
			);
			$param_type = array(
				"id" => "integer",
				"create_time" => "integer",
				"enable" => "integer",
				"create_account_id" => "integer",
				"title" => "text",
				"content" => "text",
				"tag" => "text"
			);
			
			$result = $con->execParam($sql, $param, $param_type);
			if ($result !== 1) {
				$con->rollback();
				$con->disconnect();
				throw new SQLException("データが正しく登録されませんでした。");
			}
			
			$con->commit();
			$con->disconnect();
		}
		
		/**
		 * 記事を削除します。
		 * 
		 * <ol>
		 * <li>引数を確認します。
		 * <li>削除する記事の最終更新の有効/無効が有効であることを確認します。
		 * <li>引数のID、現在日時を使用して、データベースに記事の情報を挿入します(有効/無効 = 無効)。
		 * </ol>
		 * 
		 * @param int $id 削除する記事のID。
		 * @param int $account_id 記事を削除するユーザのID。
		 * @return void
		 */
		public static function delete($id, $account_id) {
			/*
			 * 引数を確認します。
			 */
			if (!is_int($id)) {
				throw new ArgumentException("id引数の値が整数型ではありません。");
			}
			if (!is_int($account_id)) {
				throw new ArgumentException("account_id引数の値が整数型ではありません。");
			}
			
			/*
			 * データベース関連を確認します。
			 */
			global $config;
			$con = MDB2::connect($config["dsn"]);
			$con->loadModule("Extended");
			
			$sql = <<< SQL_END
select
	article.id as id,
	article.title as title,
	article.content as content,
	article.tag as tag
from
	article,
	(
		select
			id,
			max(create_time) latest_time
		from
			article
		group by
			id
		having
			id = :id
	) as latest_article
where
	article.id = latest_article.id
	and article.enable = :enable
SQL_END;
			$type = array(
				"id" => "integer",
				"title" => "text",
				"content" => "text",
				"tag" => "text"
			);
			$param = array(
				"id" => $id,
				"enable" => 1
			);
			$param_type = array(
				"id" => "integer",
				"enable" => "integer"
			);
			
			$articles = $con->getAll($sql, $type, $param, $param_type, MDB2_FETCHMODE_ASSOC);
			if (count($articles) === 0) {
				$con->disconnect();
				throw new HiramekiException("記事「" . $id . "」は存在しないか、既に削除されています。");
			}
			$article = $articles[0];
			
			/*
			 * 記事をデータベースから削除します。
			 */
			$con->beginTransaction();
			
			$now = Hirameki_DateTimeUtil::now();
			$sql = <<< SQL_END
insert into article (
	id,
	create_time,
	enable,
	create_account_id,
	title,
	content,
	tag
) values (
	:id,
	:create_time,
	:enable,
	:create_account_id,
	:title,
	:content,
	:tag
)
SQL_END;
			$param = array(
				"id" => $article["id"],
				"create_time" => $now,
				"enable" => 0,
				"create_account_id" => $account_id,
				"title" => $article["title"],
				"content" => $article["content"],
				"tag" => $article["tag"]
			);
			$param_type = array(
				"id" => "integer",
				"create_time" => "integer",
				"enable" => "integer",
				"create_account_id" => "integer",
				"title" => "text",
				"content" => "text",
				"tag" => "text"
			);
			
			$result = $con->execParam($sql, $param, $param_type);
			if ($result !== 1) {
				$con->rollback();
				$con->disconnect();
				throw new SQLException("データが正しく登録されませんでした。");
			}
			
			$con->commit();
			$con->disconnect();
		}
		
		/**
		 * 記事の一覧を返します。
		 * 
		 * 次のような値を返します。
		 * <samp>
		 * array(
		 *     [] => array(
		 *         "id" => ID,
		 *         "title" => 件名,
		 *         "update_time" => 更新日時,
		 *         "update_account_id" => 更新者のID、
		 *         "update_account_name" => 更新者のユーザ名,
		 *         "update_account_real_name" => 更新者の名前,
		 *         "create_time" => 作成日時,
		 *         "create_account_id" => 登録者のID,
		 *         "create_account_name" => 登録者のユーザ名,
		 *         "create_account_real_name" => 登録者の名前,
		 *         "tag" => タグ
		 *     )
		 * )
		 * </samp>
		 * 
		 * @param void
		 * @return 記事の概要を格納した配列。
		 */
		public static function findByAll() {
			global $config;
			$con = MDB2::connect($config["dsn"]);
			$con->loadModule("Extended");
			
			$sql = <<< SQL_END
select
	latest_article.id as id,
	latest_article.title as title,
	latest_article.create_time as update_time,
	update_account.id as update_account_id,
	update_account.name as update_account_name,
	update_account.real_name as update_account_real_name,
	oldest_article.create_time as create_time,
	create_account.id as create_account_id,
	create_account.name as create_account_name,
	create_account.real_name as create_account_real_name,
	latest_article.tag as tag
from
	article as latest_article,
	(
		select
			id,
			max(create_time) as latest_time
		from
			article
		group by
			id
	) as latest_article_time,
	article as oldest_article,
	(
		select
			id,
			min(create_time) as oldest_time
		from
			article
		group by
			id
	) as oldest_article_time,
	account as update_account,
	account as create_account
where
	latest_article.id = latest_article_time.id
	and latest_article.create_time = latest_article_time.latest_time
	and oldest_article.id = oldest_article_time.id
	and oldest_article.create_time = oldest_article_time.oldest_time
	and latest_article.id = oldest_article.id
	and latest_article.create_account_id = update_account.id
	and oldest_article.create_account_id = create_account.id
	and latest_article.enable = :enable
order by
	latest_article.id desc
SQL_END;
			$type = array(
				"id" => "integer",
				"title" => "text",
				"update_time" => "integer",
				"update_account_id" => "integer",
				"update_account_name" => "text",
				"update_account_real_name" => "text",
				"create_time" => "integer",
				"create_account_id" => "integer",
				"create_account_name" => "text",
				"create_account_real_name" => "text",
				"tag" => "text"
			);
			$param = array("enable" => 1);
			$param_type = array("enable" => "integer");
			
			$articles = $con->getAll($sql, $type, $param, $param_type, MDB2_FETCHMODE_ASSOC);
			
			$con->disconnect();
			
			return $articles;
		}
		
		/**
		 * 記事の詳細情報を返します。
		 * 
		 * 次のような値を返します。
		 * <samp>
		 * array(
		 *     "id" => ID,
		 *     "title" => 件名,
		 *     "update_time" => 更新日時,
		 *     "update_account_id" => 更新者のID,
		 *     "update_account_name" => 更新者のユーザ名,
		 *     "update_account_real_name" => 更新者の名前,
		 *     "create_time" => 作成日時,
		 *     "create_account_id" => 登録者のID,
		 *     "create_account_name" => 登録者のユーザ名,
		 *     "create_account_real_name" => 登録者の名前,
		 *     "content" => 内容,
		 *     "tag" => タグ
		 * )
		 * </samp>
		 * 
		 * @param int $id 記事のID。
		 * @return 記事の詳細情報を格納した配列。
		 */
		public static function findByID($id) {
			/*
			 * 引数を確認します。
			 */
			if (!is_int($id)) {
				throw new ArgumentException("id引数が整数型ではありません。");
			}
			
			/*
			 * 記事を検索します。
			 */
			global $config;
			$con = MDB2::connect($config["dsn"]);
			$con->loadModule("Extended");
			
			$sql = <<< SQL_END
select
	latest_article.id as id,
	latest_article.title as title,
	latest_article.create_time as update_time,
	update_account.id as update_account_id,
	update_account.name as update_account_name,
	update_account.real_name as update_account_real_name,
	oldest_article.create_time as create_time,
	create_account.id as create_account_id,
	create_account.name as create_account_name,
	create_account.real_name as create_account_real_name,
	latest_article.content as content,
	latest_article.tag as tag
from
	article as latest_article,
	(
		select
			id,
			max(create_time) as latest_time
		from
			article
		group by
			id
	) as latest_article_time,
	article as oldest_article,
	(
		select
			id,
			min(create_time) as oldest_time
		from
			article
		group by
			id
	) as oldest_article_time,
	account as update_account,
	account as create_account
where
	latest_article.id = latest_article_time.id
	and latest_article.create_time = latest_article_time.latest_time
	and oldest_article.id = oldest_article_time.id
	and oldest_article.create_time = oldest_article_time.oldest_time
	and latest_article.id = oldest_article.id
	and latest_article.create_account_id = update_account.id
	and oldest_article.create_account_id = create_account.id
	and latest_article.id = :id
	and latest_article.enable = :enable
order by
	latest_article.id desc
SQL_END;
			$type = array(
				"id" => "integer",
				"title" => "text",
				"update_time" => "integer",
				"update_account_id" => "integer",
				"update_account_name" => "text",
				"update_account_real_name" => "text",
				"create_time" => "integer",
				"create_account_id" => "integer",
				"create_account_name" => "text",
				"create_account_real_name" => "text",
				"content" => "text",
				"tag" => "text"
			);
			$param = array("id" => $id, "enable" => 1);
			$param_type = array("id" => "integer", "enable" => "integer");
			
			$articles = $con->getAll($sql, $type, $param, $param_type, MDB2_FETCHMODE_ASSOC);
			
			$con->disconnect();
			
			if (count($articles) === 1) {
				return $articles[0];
			} else {
				return NULL;
			}
		}
	}
?>
