/*

Copyright (C) 2009 NTT DATA INTELLILINK CORPORATION

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.

 */

package jp.co.intellilink.hinemos.importtool.conf.notify;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;

import javax.ejb.DuplicateKeyException;

import jp.co.intellilink.hinemos.util.CheckString;
import jp.co.intellilink.hinemos.util.Config;
import jp.co.intellilink.hinemos.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.util.Messages;
import jp.co.intellilink.hinemos.util.ReadCsvFile;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.notify.mail.bean.MailTemplateTableDefine;
import com.clustercontrol.notify.mail.ejb.entity.MailTemplateInfoData;
import com.clustercontrol.notify.mail.ejb.session.MailTemplateController;

/**
 * メールテンプレート情報をインポートするクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class ImportMailTemplate {

	// ログ出力
	private static Log log = LogFactory.getLog(ImportMailTemplate.class);

	@SuppressWarnings("unchecked")
	private ArrayList mailTemplateList = null;

	private ArrayList<ArrayList<String>> mailTemplateInfos = null;

	/**
	 * メールテンプレート情報をインポートする。<br>
	 */
	public void importMailTemplate() {
		log.info(Messages.getMsg("MailTemplate.Start", new String[]{Messages.getMsg("ImportTool.Import")}));

		//メールテンプレート情報のCSVファイルを読み込む
		mailTemplateInfos = ReadCsvFile.readCsvFile(Config.getConfig("Import.Dir") + "/MAILTEMPLATE.csv");

		//メールテンプレートリスト取得
		mailTemplateList = getMailTemplateList();

		for(ArrayList<String> info : mailTemplateInfos){
			MailTemplateInfoData mailTemplateInfo = createMailTemplateInfo(info);
			registerMailTemplateInfo(mailTemplateInfo);
		}

		log.info(Messages.getMsg("MailTemplate.End", new String[]{Messages.getMsg("ImportTool.Import")}));
	}

	/**
	 * メールテンプレート情報を作成する。<br>
	 * 
	 * @param info メールテンプレート情報配列
	 * @return メールテンプレート情報
	 */
	public MailTemplateInfoData createMailTemplateInfo(ArrayList<String> info) {
		MailTemplateInfoData mailTemplateInfo = new MailTemplateInfoData();

		//メールテンプレートID
		String id = CheckString.checkLength(info.get(0), 64);
		if(!id.equals(info.get(0))){
			String[] args = { com.clustercontrol.util.Messages.getString("mail.template.id"), "64" };
			log.error(Messages.getMsg("ImportTool.Common.2", args));
			System.exit(20);
		}
		mailTemplateInfo.setMailTemplateId(id);
		//説明
		String description = CheckString.checkLength(info.get(1), 256);
		if(!description.equals(info.get(1))){
			String[] args = { com.clustercontrol.util.Messages.getString("description"), "256" };
			log.warn(Messages.getMsg("ImportTool.Common.2", args));
		}
		mailTemplateInfo.setDescription(description);
		//件名
		String subject = CheckString.checkLength(info.get(2), 256);
		if(!subject.equals(info.get(2))){
			String[] args = { com.clustercontrol.util.Messages.getString("subject"), "256" };
			log.warn(Messages.getMsg("ImportTool.Common.2", args));
		}
		mailTemplateInfo.setSubject(subject);
		//本文
		mailTemplateInfo.setBody(info.get(3));

		return mailTemplateInfo;
	}

	/**
	 * メールテンプレート一覧に指定したメールテンプレートIDが存在するかチェックする。<br>
	 * 
	 * @param mailTemplateId メールテンプレートID
	 * @return チェック結果
	 */
	@SuppressWarnings("unchecked")
	protected ArrayList checkMailTemplateList(String mailTemplateId) {

		if(mailTemplateList instanceof ArrayList){
			Iterator itr = mailTemplateList.iterator();
			while(itr.hasNext()){
				ArrayList line = (ArrayList)itr.next();

				String id = (String)line.get(MailTemplateTableDefine.MAIL_TEMPLATE_ID);
				if(id.compareTo(mailTemplateId) == 0)
					return line;
			}
		}

		return null;
	}

	/**
	 * メールテンプレート情報を登録する。<br>
	 * 
	 * @param mailTemplateInfo メールテンプレート情報
	 */
	protected void registerMailTemplateInfo(MailTemplateInfoData mailTemplateInfo) {

		MailTemplateController mail = EjbConnectionManager.getConnectionManager().getMailTemplateController();

		try {
			@SuppressWarnings("unchecked")
			ArrayList line = checkMailTemplateList(mailTemplateInfo.getMailTemplateId());
			if(line instanceof ArrayList){
				Date createTime = (Date)line.get(MailTemplateTableDefine.CREATE_TIME);
				String createUser = (String)line.get(MailTemplateTableDefine.CREATE_USER);
				mailTemplateInfo.setRegDate(new Timestamp(createTime.getTime()));
				mailTemplateInfo.setRegUser(createUser);

				Object[] args = {mailTemplateInfo.getMailTemplateId()};
				log.info(Messages.getMsg("MailTemplateID", args) + 
						" (" + Messages.getMsg("modify") + ")");

				mail.modifyMailTemplate(mailTemplateInfo);
			}
			else{
				Object[] args = {mailTemplateInfo.getMailTemplateId()};
				log.info(Messages.getMsg("MailTemplateID", args) + 
						" (" + Messages.getMsg("add") + ")");

				mail.addMailTemplate(mailTemplateInfo);
			}
		} catch (DuplicateKeyException e) {
			String args[] = {mailTemplateInfo.getMailTemplateId()};
			log.error(Messages.getMsg("ImportTool.Common.1", args));
			System.exit(20);
		} catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
		}
	}

	/**
	 * メールテンプレートリストを取得する。<br>
	 * 
	 * @return メールテンプレート一覧
	 */
	@SuppressWarnings("unchecked")
	protected ArrayList getMailTemplateList() {

		MailTemplateController mail = EjbConnectionManager.getConnectionManager().getMailTemplateController();

		ArrayList records = null;
		try {
			records = mail.getMailTemplateList();
		} catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
		}
		return records;
	}
}