/*

Copyright (C) 2009 NTT DATA INTELLILINK CORPORATION

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.

 */

package jp.co.intellilink.hinemos.importtool.conf.monitor;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.ejb.DuplicateKeyException;

import jp.co.intellilink.hinemos.util.CheckString;
import jp.co.intellilink.hinemos.util.Config;
import jp.co.intellilink.hinemos.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.util.Messages;
import jp.co.intellilink.hinemos.util.ReadCsvFile;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorTabelDefine;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.snmp.bean.MonitorSnmpInfo;
import com.clustercontrol.snmp.ejb.session.MonitorSnmpController;

/**
 * SNMP監視情報をインポートするクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class ImportMonitorSnmpString extends ImportMonitorString {

	// ログ出力
	private static Log log = LogFactory.getLog(ImportMonitorSnmpString.class);

	@SuppressWarnings("unchecked")
	private ArrayList monitorList = null;

	private ArrayList<ArrayList<String>> monitorInfos = null;

	/**
	 * SNMP監視情報をインポートする。<br>
	 */
	public void importSnmp() {
		log.info(Messages.getMsg("Snmp.Start", new String[]{Messages.getMsg("ImportTool.Import")}));

		//SNMP監視情報のCSVファイルを読み込む
		monitorInfos = ReadCsvFile.readCsvFile(Config.getConfig("Import.Dir") + "/SNMP_STR.csv");
		monitorRuleInfos = ReadCsvFile.readCsvFile(Config.getConfig("Import.Dir") + "/SNMP_STR_RULE.csv");

		//監視項目リスト取得
		monitorList = getMonitorList();

		for(ArrayList<String> info : monitorInfos){
			MonitorInfo monitorInfo = createMonitorInfo(info);
			registerMonitorInfo(monitorInfo);
		}

		log.info(Messages.getMsg("Snmp.End", new String[]{Messages.getMsg("ImportTool.Import")}));
	}

	/**
	 * 監視情報オブジェクトを作成する。<br>
	 * 
	 * @see jp.co.intellilink.hinemos.importtool.conf.monitor.ImportMonitor#createMonitorInfo(java.util.ArrayList)
	 */
	@Override
	protected MonitorInfo createMonitorInfo(ArrayList<String> info){
		MonitorInfo monitorInfo = new MonitorInfo();

		//監視基本情報
		createMonitorBase(info, monitorInfo);

		//監視条件情報
		createMonitorRule(info.subList(3, 5), monitorInfo);

		//監視情報
		createMonitorCheck(info.subList(5, 9), monitorInfo);

		//監視通知情報
		createMonitorNotify(info.subList(9, 14), monitorInfo);

		//アプリケーション
		createMonitorApplication(info.subList(14, 15), monitorInfo);

		//値取得失敗時
		if(info.get(15) instanceof String && info.get(15).length() > 0){
			monitorInfo.setFailurePriority(PriorityConstant.stringToType(info.get(15)));
		}

		//有効/無効
		createMonitorVaild(info.subList(16, 17), monitorInfo);

		//監視判定情報
		createMonitorJudgement(monitorInfo);

		return monitorInfo;
	}

	/**
	 * SNMP監視情報を設定する。<br>
	 * 
	 * @param info 監視情報配列
	 * @param monitorInfo 監視情報オブジェクト
	 * 
	 * @see jp.co.intellilink.hinemos.importtool.conf.monitor.ImportMonitor#createMonitorCheck(java.util.List, com.clustercontrol.monitor.run.bean.MonitorInfo)
	 */
	@Override
	protected void createMonitorCheck(List<String> info, MonitorInfo monitorInfo) {
		MonitorSnmpInfo snmpInfo = new MonitorSnmpInfo();

		//コミュニティ名
		String communityName = CheckString.checkLength(info.get(0), DataRangeConstant.VARCHAR_64);
		if(!communityName.equals(info.get(0))){
			String[] args = { com.clustercontrol.util.Messages.getString("community.name"), "64" };
			log.warn(Messages.getMsg("ImportTool.Common.2", args));
		}
		snmpInfo.setCommunityName(communityName);
		//ポート番号
		snmpInfo.setSnmpPort(Integer.parseInt(info.get(1)));
		//OID
		String oid = CheckString.checkLength(info.get(2), DataRangeConstant.VARCHAR_1024);
		if(!oid.equals(info.get(2))){
			String[] args = { com.clustercontrol.util.Messages.getString("oid"), "1024" };
			log.warn(Messages.getMsg("ImportTool.Common.2", args));
		}
		snmpInfo.setSnmpOid(oid);
		//バージョン
		snmpInfo.setSnmpVersion(info.get(3));

		monitorInfo.setCheckInfo(snmpInfo);

		monitorInfo.setMonitorType(MonitorTypeConstant.TYPE_STRING);

		super.createMonitorCheck(info, monitorInfo);
	}

	/**
	 * 監視情報を登録する。<br>
	 * 
	 * @param info 監視情報オブジェクト
	 */
	protected void registerMonitorInfo(MonitorInfo info) {
		MonitorSnmpController snmp = EjbConnectionManager.getConnectionManager().getMonitorSnmpController();

		try {
			@SuppressWarnings("unchecked")
			ArrayList list = checkMonitorList(info.getMonitorId(), monitorList);
			if(list instanceof ArrayList){
				Date createTime = (Date)list.get(MonitorTabelDefine.CREATE_TIME);
				String createUser = (String)list.get(MonitorTabelDefine.CREATE_USER);
				info.setRegDate(new Timestamp(createTime.getTime()));
				info.setRegUser(createUser);

				Object[] args = {info.getMonitorId()};
				log.info(Messages.getMsg("MonitorID", args) + 
						" (" + Messages.getMsg("modify") + ")");

				snmp.modifySnmp(info);
			}
			else{
				Object[] args = {info.getMonitorId()};
				log.info(Messages.getMsg("MonitorID", args) + 
						" (" + Messages.getMsg("add") + ")");

				snmp.addSnmp(info);
			}
		} catch (DuplicateKeyException e) {
			String args[] = {info.getMonitorId()};
			log.error(Messages.getMsg("ImportTool.Common.1", args));
			System.exit(20);
		} catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
		}
	}

	/**
	 * 監視項目リストを取得する。<br>
	 * 
	 * @return 監視項目一覧
	 */
	@SuppressWarnings("unchecked")
	protected ArrayList getMonitorList() {

		MonitorSnmpController snmp = EjbConnectionManager.getConnectionManager().getMonitorSnmpController();

		ArrayList records = null;
		try {
			records = snmp.getSnmpList();
		} catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
		}
		return records;
	}
}