/*

Copyright (C) 2009 NTT DATA INTELLILINK CORPORATION

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.

 */

package jp.co.intellilink.hinemos.importtool.conf.monitor;

import java.util.ArrayList;
import java.util.List;

import jp.co.intellilink.hinemos.util.CheckString;
import jp.co.intellilink.hinemos.util.Config;
import jp.co.intellilink.hinemos.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.util.Messages;
import jp.co.intellilink.hinemos.util.ReadCsvFile;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.ProcessConstant;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.syslogng.bean.LogFilterInfo;
import com.clustercontrol.syslogng.ejb.session.SyslogNGController;

/**
 * Syslog-ng監視情報を取得するクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class ImportMonitorSyslogng extends ImportMonitor {

	// ログ出力
	private static Log log = LogFactory.getLog(ImportMonitorSyslogng.class);

	@SuppressWarnings("unchecked")
	private ArrayList<LogFilterInfo> monitorList = null;

	private ArrayList<ArrayList<String>> monitorInfos = null;

	/**
	 * Syslog-ng監視情報をExportする。<br>
	 */
	public void importSyslogng() {
		log.info(Messages.getMsg("Syslogng.Start", new String[]{Messages.getMsg("ImportTool.Import")}));

		//Syslog-ng監視情報のCSVファイルを読み込む
		monitorInfos = ReadCsvFile.readCsvFile(Config.getConfig("Import.Dir") + "/SYSLOGNG.csv");

		//監視項目リスト取得
		monitorList = getMonitorList();

		ArrayList<LogFilterInfo> logFilterList = new ArrayList<LogFilterInfo>();

		for(ArrayList<String> info : monitorInfos){
			MonitorInfo monitorInfo = createMonitorInfo(info);

			LogFilterInfo list = checkMonitorList(monitorInfo.getMonitorId(), monitorList);
			if(list instanceof LogFilterInfo){
				monitorInfo.setRegDate(list.getRegDate());
				monitorInfo.setRegUser(list.getRegUser());
			}
			Object[] args = {monitorInfo.getMonitorId()};
			log.info(Messages.getMsg("MonitorID", args));

			logFilterList.add((LogFilterInfo)monitorInfo);
		}
		registerMonitorInfo(logFilterList);

		log.info(Messages.getMsg("Syslogng.End", new String[]{Messages.getMsg("ImportTool.Import")}));
	}

	/**
	 * 監視情報オブジェクトを作成する。<br>
	 * 
	 * @see jp.co.intellilink.hinemos.importtool.conf.monitor.ImportMonitor#createMonitorInfo(java.util.ArrayList)
	 */
	public MonitorInfo createMonitorInfo(ArrayList<String> info){
		LogFilterInfo monitorInfo = new LogFilterInfo();

		//監視基本情報
		createMonitorBase(info, monitorInfo);

		//監視条件情報
		createMonitorRule(info.subList(3, 5), monitorInfo);

		//監視情報
		createMonitorCheck(info.subList(5, 7), monitorInfo);

		//監視通知情報
		createMonitorNotify(info.subList(7, 12), monitorInfo);

		//アプリケーション
		createMonitorApplication(info.subList(12, 13), monitorInfo);

		//メッセージ
		createMonitorMessages(info.subList(13, 16), monitorInfo);

		//有効/無効
		createMonitorVaild(info.subList(16, 17), monitorInfo);

		return monitorInfo;
	}

	/**
	 * 監視基本情報を設定する。<br>
	 * 
	 * @param info 監視情報配列
	 * @param monitorInfo 監視情報オブジェクト
	 * 
	 * @see jp.co.intellilink.hinemos.importtool.conf.monitor.ImportMonitor#createMonitorBase(java.util.List, com.clustercontrol.monitor.run.bean.MonitorInfo)
	 */
	@Override
	protected void createMonitorBase(List<String> info, MonitorInfo monitorInfo) {
		LogFilterInfo filterInfo = (LogFilterInfo)monitorInfo;
		//順序
		filterInfo.setOrderNo(Integer.parseInt(info.get(0)));
		//監視項目ID
		String id = CheckString.checkLength(info.get(1), DataRangeConstant.VARCHAR_64);
		if(!id.equals(info.get(1))){
			String[] args = { com.clustercontrol.util.Messages.getString("monitor.id"), "64" };
			log.error(Messages.getMsg("ImportTool.Common.2", args));
			System.exit(20);
		}
		filterInfo.setMonitorId(id);
		//説明
		String description = CheckString.checkLength(info.get(2), DataRangeConstant.VARCHAR_256);
		if(!description.equals(info.get(2))){
			String[] args = { com.clustercontrol.util.Messages.getString("description"), "256" };
			log.warn(Messages.getMsg("ImportTool.Common.2", args));
		}
		filterInfo.setDescription(description);
	}

	/**
	 * 監視条件情報を設定する。<br>
	 * 
	 * @param info 監視情報配列
	 * @param monitorInfo 監視情報オブジェクト
	 * 
	 * @see jp.co.intellilink.hinemos.importtool.conf.monitor.ImportMonitor#createMonitorRule(java.util.List, com.clustercontrol.monitor.run.bean.MonitorInfo)
	 */
	@Override
	protected void createMonitorRule(List<String> info, MonitorInfo monitorInfo) {
		//カレンダID
		monitorInfo.setCalendarId(info.get(0));
		//ファシリティID
		monitorInfo.setFacilityId(info.get(1));
	}

	/**
	 * Syslog-ng監視情報を設定する。<br>
	 * 
	 * @param info 監視情報配列
	 * @param monitorInfo 監視情報オブジェクト
	 * 
	 * @see jp.co.intellilink.hinemos.importtool.conf.monitor.ImportMonitor#createMonitorCheck(java.util.List, com.clustercontrol.monitor.run.bean.MonitorInfo)
	 */
	@Override
	protected void createMonitorCheck(List<String> info, MonitorInfo monitorInfo) {
		LogFilterInfo filterInfo = (LogFilterInfo)monitorInfo;

		//パターンマッチ表現
		String pattern = CheckString.checkLength(info.get(0), DataRangeConstant.VARCHAR_1024);
		if(!pattern.equals(info.get(0))){
			String[] args = { com.clustercontrol.util.Messages.getString("pattern.matching.expression"), "1024" };
			log.warn(Messages.getMsg("ImportTool.Common.2", args));
		}
		filterInfo.setPattern(pattern);
		//処理有無
		filterInfo.setProcessType(ProcessConstant.stringToType(info.get(1)));
	}

	/**
	 * Syslog-ng監視固有のメッセージ情報を設定する。<br>
	 * 
	 * @param info 監視情報配列
	 * @param monitorInfo 監視情報オブジェクト
	 */
	@SuppressWarnings("unchecked")
	protected void createMonitorMessages(List<String> info, MonitorInfo monitorInfo) {
		LogFilterInfo filterInfo = (LogFilterInfo)monitorInfo;

		//重要度
		filterInfo.setPriority(PriorityConstant.stringToType(info.get(0)));
		//メッセージID
		String id = CheckString.checkLength(info.get(1), DataRangeConstant.VARCHAR_64);
		if(!id.equals(info.get(1))){
			String[] args = { com.clustercontrol.util.Messages.getString("message.id"), "64" };
			log.warn(Messages.getMsg("ImportTool.Common.2", args));
		}
		filterInfo.setMessageId(id);
		//メッセージ
		String message = CheckString.checkLength(info.get(2), DataRangeConstant.VARCHAR_256);
		if(!message.equals(info.get(2))){
			String[] args = { com.clustercontrol.util.Messages.getString("message"), "256" };
			log.warn(Messages.getMsg("ImportTool.Common.2", args));
		}
		filterInfo.setMessage(message);
	}

	/**
	 * 監視情報を登録する。<br>
	 * 
	 * @param logFilterInfos 監視情報オブジェクト
	 */
	protected void registerMonitorInfo(ArrayList<LogFilterInfo> logFilterInfos) {
		SyslogNGController syslog = EjbConnectionManager.getConnectionManager().getMonitorSyslogNGController();

		try {
			syslog.createMonitorRuleList(logFilterInfos);
		} catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
		}
	}

	/**
	 * 監視項目リストに指定した監視項目IDが存在するかチェックする。<br>
	 * 
	 * @param monitorId 監視項目ID
	 * @param monitorList 監視項目リスト
	 * @return 監視項目
	 */
	@SuppressWarnings("unchecked")
	protected LogFilterInfo checkMonitorList(String monitorId, ArrayList<LogFilterInfo> monitorList) {

		if(monitorList instanceof ArrayList){
			for(LogFilterInfo line : monitorList){
				if(line.getMonitorId().compareTo(monitorId) == 0)
					return line;
			}
		}

		return null;
	}

	/**
	 * 監視項目リストを取得する。<br>
	 * 
	 * @return 監視項目一覧
	 */
	@SuppressWarnings("unchecked")
	protected ArrayList<LogFilterInfo> getMonitorList() {

		SyslogNGController syslog = EjbConnectionManager.getConnectionManager().getMonitorSyslogNGController();

		ArrayList<LogFilterInfo> records = null;
		try {
			records = syslog.getFilterInfoList();
		} catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
		}
		return records;
	}
}