/*
 
Copyright (C) 2009 NTT DATA INTELLILINK CORPORATION
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package jp.co.intellilink.hinemos.export.conf.monitor;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;

import jp.co.intellilink.hinemos.util.Config;
import jp.co.intellilink.hinemos.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.util.IntegerListComparator;
import jp.co.intellilink.hinemos.util.Messages;
import jp.co.intellilink.hinemos.util.WriteCsvFile;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.ProcessConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.syslogng.bean.LogFilterInfo;
import com.clustercontrol.syslogng.ejb.session.SyslogNGController;

/**
 * Syslog-ng監視情報を取得するクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class ExportMonitorSyslogng extends ExportMonitor {

	// ログ出力
	private static Log log = LogFactory.getLog(ExportMonitorSyslogng.class);
	
	/**
     * Syslog-ng監視情報をExportする。<br>
     */
    public void exportSyslogng() {
    	log.info(Messages.getMsg("Syslogng.Start", new String[]{Messages.getMsg("ExportTool.Export")}));
    	
    	//CSV出力配列を用意
    	ArrayList<ArrayList<String>> csvOutput = new ArrayList<ArrayList<String>>();
    	
    	//監視項目リスト取得
    	ArrayList<LogFilterInfo> infoList = getSyslogngList();
    	
    	if(infoList instanceof ArrayList){
			for(LogFilterInfo info : infoList){
				Object[] args = {info.getMonitorId()};
				log.info(Messages.getMsg("MonitorID", args));
				
				csvOutput.add(createCsvFormat(info));
			}
			
			//順序IDでソート
			Collections.sort(csvOutput, new IntegerListComparator(0));
    	}
    	
		String timestamp = "";
		if(Config.getConfig("file.timestamp").equalsIgnoreCase("on")){
			SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd_HHmmss");
			timestamp = "_" + format.format(new Date());
		}
		WriteCsvFile.writeCsvFile(
				Config.getConfig("Export.Dir") + "/SYSLOGNG" + timestamp + ".csv", 
				csvOutput);
		
		log.info(Messages.getMsg("Syslogng.End", new String[]{Messages.getMsg("ExportTool.Export")}));
    }
    
	/**
	 * Syslog-ng監視情報をExportする。<br>
	 * 
	 * @param monitorInfo HTTP監視情報
	 * @return CSVフォーマット文字列
	 * 
     * @see jp.co.intellilink.hinemos.export.conf.monitor.ExportMonitorNumeric#createCsvFormat(com.clustercontrol.monitor.run.bean.MonitorInfo)
     */
    @Override
	public ArrayList<String> createCsvFormat(MonitorInfo monitorInfo) {
		ArrayList<String> csvFormat = new ArrayList<String>();
		
		//監視基本情報
		csvFormat.addAll(createMonitorBase(monitorInfo));
		
		//監視条件情報
		csvFormat.addAll(createMonitorRule(monitorInfo));
		
		//監視情報
		csvFormat.addAll(createMonitorInfo(monitorInfo));
		
		//監視通知情報
		csvFormat.addAll(createMonitorNotify(monitorInfo));

        //アプリケーション
        if (monitorInfo.getApplication() != null)
        	csvFormat.add(monitorInfo.getApplication());
        else
        	csvFormat.add("");
        
        LogFilterInfo filterInfo = (LogFilterInfo)monitorInfo; 
        
        //重要度
        csvFormat.add(PriorityConstant.typeToString(filterInfo.getPriority()));
        
        //メッセージID
        if (filterInfo.getMessageId() != null)
        	csvFormat.add(filterInfo.getMessageId());
        else
        	csvFormat.add("");
        
        //メッセージ
        if (filterInfo.getMessage() != null)
        	csvFormat.add(filterInfo.getMessage());
        else
        	csvFormat.add("");
        
        //有効/無効
        csvFormat.add(ValidConstant.typeToString(monitorInfo.getValid()));
		
		return csvFormat;
	}
    
    /**
	 * 監視基本情報をExportする。<br>
	 * 
	 * @param monitorInfo 監視情報
	 * @return CSVフォーマット文字列
	 */
    @Override
	protected ArrayList<String> createMonitorBase(MonitorInfo monitorInfo) {
		ArrayList<String> csvFormat = new ArrayList<String>();
		
		LogFilterInfo filterInfo = (LogFilterInfo)monitorInfo; 
		
		//順序
		csvFormat.add(String.valueOf(filterInfo.getOrderNo()));
		//監視項目ID
		csvFormat.add(monitorInfo.getMonitorId());
		//説明
		if (monitorInfo.getDescription() != null)
			csvFormat.add(monitorInfo.getDescription());
		else
			csvFormat.add("");

		return csvFormat;
	}
    
    /**
	 * 監視条件情報をExportする。<br>
	 * 
	 * @param monitorInfo 監視情報
	 * @return CSVフォーマット文字列
	 */
    @Override
	protected ArrayList<String> createMonitorRule(MonitorInfo monitorInfo) {
		ArrayList<String> csvFormat = new ArrayList<String>();

		//カレンダID
		if (monitorInfo.getCalendarId() != null)
			csvFormat.add(monitorInfo.getCalendarId());
		else
			csvFormat.add("");
		//ファシリティID
		csvFormat.add(monitorInfo.getFacilityId());
		
		return csvFormat;
	}
    
    /**
	 * Syslog-ng監視情報をExportする。<br>
	 * 
	 * @param monitorInfo SNMPTRAP監視情報
	 * @return CSVフォーマット文字列
	 */
    @Override
	public ArrayList<String> createMonitorInfo(MonitorInfo monitorInfo) {
		ArrayList<String> csvFormat = new ArrayList<String>();
		
		LogFilterInfo filterInfo = (LogFilterInfo)monitorInfo; 
        //パターンマッチ表現
        if (filterInfo.getPattern() != null)
        	csvFormat.add(filterInfo.getPattern());
        else
        	csvFormat.add("");
        //処理有無
        csvFormat.add(ProcessConstant.typeToString(filterInfo.getProcessType()));
        
		return csvFormat;
	}

    /**
     * 監視項目リストを取得する。<br>
     * 
     * @return 監視項目一覧
     */
    protected ArrayList<LogFilterInfo> getSyslogngList() {

    	SyslogNGController syslog = EjbConnectionManager.getConnectionManager().getMonitorSyslogNGController();

    	ArrayList<LogFilterInfo> records = null;
    	try {
    		records = syslog.getFilterInfoList();
        } catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
        }
        return records;
    }
}