﻿Option Explicit On
Option Strict On

Imports System.Windows.Media.Animation

''' <summary>アイコンを格納するコントロール。</summary>
''' <remarks>
''' 起動する要素（ショートカット／エグゼ）を格納する。
''' </remarks>
Public NotInheritable Class IconCell

#Region "events"

    ''' <summary>マウスがコントロールにホバー中であることを通知する。</summary>
    ''' <param name="sender">イベント発行元。</param>
    Public Event NotifyHovering(sender As Object)

    ''' <summary>マウスドラッグが開始されたことを外部に通知する。</summary>
    ''' <param name="sender">イベント発行元。</param>
    Public Event DragStart(sender As Object)

    ''' <summary>コントロールが左クリックされたことを外部に通知する。</summary>
    ''' <param name="sender">イベント発行元。</param>
    Public Event Clicked(sender As Object)

    ''' <summary>コントロールが右クリックされたことを外部に通知する。</summary>
    ''' <param name="sender">イベント発行元。</param>
    Public Event ClickedByRight(sender As Object)

    ''' <summary>アイコンが削除されたことを外部に通知する。</summary>
    ''' <param name="sender">イベント発行元。</param>
    Public Event Removed(sender As Object)

#End Region

    ''' <summary>このコントロールの列位置。</summary>
    Public ReadOnly Column As Integer

    ''' <summary>このコントロールの行位置。</summary>
    Public ReadOnly Row As Integer

    ' ホバーモードフラグ
    Private mHovering As Boolean

    ' クリックポイント
    Private mClickPos As Point

    ''' <summary>コンストラクタ。</summary>
    ''' <param name="row">行位置。</param>
    ''' <param name="column">列位置。</param>
    Public Sub New(row As Integer, column As Integer)
        InitializeComponent()

        Me.Hovering = False
        Me.Row = row
        Me.Column = column
        Me.mClickPos = New Point(-1, -1)

        AddHandler Me.MouseEnter, AddressOf Me.frameImageLayer_MouseEnter
        AddHandler Me.MouseLeave, AddressOf Me.frameImageLayer_MouseLeave
        AddHandler Me.PreviewMouseMove, AddressOf Me.frameImageLayer_PreviewMouseMove
        AddHandler Me.MouseMove, AddressOf Me.frameImageLayer_MouseMove
        AddHandler Me.MouseLeftButtonUp, AddressOf Me.frameImageLayer_MouseLeftButtonUp
        AddHandler Me.MouseRightButtonUp, AddressOf Me.frameImageLayer_MouseRightButtonUp
        AddHandler Me.MouseLeftButtonDown, AddressOf Me.frameImageLayer_MouseLeftButtonDown
    End Sub

#Region "コントロールのフォーカス処理"

    ''' <summary>マウス侵入イベントハンドラ。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="e">イベントオブジェクト。</param>
    Private Sub frameImageLayer_MouseEnter(sender As Object, e As MouseEventArgs)
        If Not Me.Hovering AndAlso Me.IsIconVisible Then
            RaiseEvent NotifyHovering(Me)
            DirectCast(Me.Resources("mouseEnterAnime"), Storyboard).Begin()
        End If
    End Sub

    ''' <summary>マウス離脱イベントハンドラ。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="e">イベントオブジェクト。</param>
    Private Sub frameImageLayer_MouseLeave(sender As Object, e As MouseEventArgs)
        If Not Me.Hovering AndAlso Me.IsIconVisible Then
            DirectCast(Me.Resources("mouseLeaveAnime"), Storyboard).Begin()
        End If
    End Sub

    ''' <summary>マウス移動イベントハンドラ。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="e">イベントオブジェクト。</param>
    Private Sub frameImageLayer_MouseMove(sender As Object, e As MouseEventArgs)
        If Not Me.Hovering AndAlso Me.IsIconVisible Then
            Dim pt As Windows.Point = Mouse.GetPosition(Me)
            Me.ellipse.Margin = New Thickness(pt.X - Me.ellipse.Width / 2, pt.Y - Me.ellipse.Height / 2, 0, 0)
            e.Handled = True
        End If
    End Sub

#End Region

    ''' <summary>マウスダウンイベントハンドラ。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="e">イベントオブジェクト。</param>
    ''' <remarks>マウスのクリック位置を記憶する。</remarks>
    Private Sub frameImageLayer_MouseLeftButtonDown(sender As Object, e As MouseButtonEventArgs)
        If Not Me.Hovering AndAlso Me.IsIconVisible Then
            Me.mClickPos = e.GetPosition(Me)
        End If
    End Sub

    ''' <summary>マウス移動イベント（直前）ハンドラ。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="e">イベントオブジェクト。</param>
    ''' <remarks>
    ''' マウスクリック位置からドラッグされた距離が適切ならドラッグ開始を外部に通知する。
    ''' </remarks>
    Private Sub frameImageLayer_PreviewMouseMove(sender As Object, e As MouseEventArgs)
        If Not Me.Hovering Then
            If e.LeftButton = MouseButtonState.Pressed AndAlso
               Me.mClickPos.X >= 0 AndAlso Me.mClickPos.Y >= 0 AndAlso
               Me.frameImageLayer.Visibility = Windows.Visibility.Visible Then
                Dim newPos As Point = e.GetPosition(Me)
                If Math.Abs(newPos.X - Me.mClickPos.X) > SystemParameters.MinimumHorizontalDragDistance OrElse
                   Math.Abs(newPos.Y - Me.mClickPos.Y) > SystemParameters.MinimumVerticalDragDistance Then
                    RaiseEvent DragStart(Me)
                End If
            End If
        End If
    End Sub

#Region "クリックイベントハンドラ"

    ''' <summary>左クリックイベントハンドラ。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="e">イベントオブジェクト。</param>
    Private Sub frameImageLayer_MouseLeftButtonUp(sender As Object, e As MouseButtonEventArgs)
        If Not Me.Hovering AndAlso Me.IsIconVisible Then
            DirectCast(Me.Resources("clickAnime"), Storyboard).Begin()
            e.Handled = True
        End If
        Me.mClickPos = New Point(-1, -1)
    End Sub

    ' 左クリックアニメーションが終了したことを外部に通知する
    Private Sub ClickAnime_Completed(sender As Object, e As EventArgs)
        RaiseEvent Clicked(Me)
    End Sub

    ''' <summary>右クリックイベントハンドラ。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="e">イベントオブジェクト。</param>
    Private Sub frameImageLayer_MouseRightButtonUp(sender As Object, e As MouseButtonEventArgs)
        If Not Me.Hovering AndAlso Me.IsIconVisible Then
            RaiseEvent ClickedByRight(Me)
            e.Handled = True
        End If
        Me.mClickPos = New Point(-1, -1)
    End Sub

#End Region

#Region "アイコン削除操作"

    ''' <summary>アイコン削除アニメーションを開始する。</summary>
    Public Sub StartRemoveAnime()
        DirectCast(Me.Resources("removeAnime"), Storyboard).Begin()
    End Sub

    ' 削除アニメーションが終了したことを外部に通知する
    Private Sub FinishRemoveAnime(sender As Object, e As EventArgs)
        RaiseEvent Removed(Me)
    End Sub

#End Region

#Region "properties"

    ''' <summary>アイコン画像。</summary>
    ''' <value>ImageSource。</value>
    Public Property IconImage() As ImageSource
        Get
            Return Me.iconImageLayer.Source
        End Get
        Set(value As ImageSource)
            Me.iconImageLayer.Source = value
            Me.frameImageLayer.Visibility = CType(IIf(value IsNot Nothing,
                    Windows.Visibility.Visible, Windows.Visibility.Hidden), Windows.Visibility)
        End Set
    End Property

    ''' <summary>アイコンフレームの表示を取得する。</summary>
    ''' <value>真偽値。</value>
    Public ReadOnly Property IsIconVisible() As Boolean
        Get
            Return (Me.frameImageLayer.Visibility = Windows.Visibility.Visible)
        End Get
    End Property

    ''' <summary>コントロールをホバーモードに変更する／しないを設定する。</summary>
    ''' <value>真偽値。</value>
    ''' <remarks>
    ''' コントロールをホバーモードに変更する、しないを設定する。
    ''' ホバーモードのときはアイコンを操作できない。
    ''' </remarks>
    Public Property Hovering() As Boolean
        Get
            Return Me.mHovering
        End Get
        Set(value As Boolean)
            Me.mHovering = value
        End Set
    End Property

#End Region

End Class
