/**
 *  X1_Limiter
 *
 *  Copyright (C) 2006-2014 Teru Kamogashira
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "X1_Limiter.hpp"
#include "X1_Limiter_Editor.hpp"

static char parameterName[KNumParams][kVstMaxParamStrLen+1] = {
  "Ceiling", "Threshol", "PreGain", "Release", "Attack", "Lookahea", "LookRati", "RMS", "Factor", "StLink", "Type",
};

static char parameterLabel[KNumParams][kVstMaxParamStrLen+1] = {
  "dB", "dB", "dB", "ms", "ms", "ms", "X", "ms", "X", "-", "-",
};

#define KNumPrograms 24
#define KPresetPrograms 9

static char presetProgramName[KPresetPrograms][kVstMaxProgNameLen] = {
  "Default Limiter",

  "Maximize +12dB",
  "Maximize +15dB",

  "SoftLimiter -5dB",
  "WarmLimiter -5dB",
  "HardLimiter -5dB",

  "SoftLimiter -9dB",
  "WarmLimiter -9dB",
  "HardLimiter -9dB",
};

static float presetProgram[KPresetPrograms][KNumParams] = {
  {  0, -2, 10,  100, 0, 1.0, 1, 0.0, 1,  1, 0, },

  {  0, -2, 12,   90, 0, 0.9, 1, 0.0, 1,  1, 0, },
  {  0, -2, 15,   80, 0, 0.8, 1, 0.0, 1,  1, 0, },

  { -5, -2,  0,   80, 0, 0.8, 1, 0.0, 1,  1, 0, },
  { -5, -1,  0,   40, 0, 0.4, 1, 0.0, 1,  1, 0, },
  { -5,  0,  0,   00, 0, 0.0, 1, 0.0, 1,  1, 0, },

  { -9, -2,  0,   60, 0, 0.8, 1, 2.0, 1,  1, 0, },
  { -9, -1,  0,   30, 0, 0.4, 1, 1.0, 1,  1, 0, },
  { -9,  0,  0,   00, 0, 0.0, 1, 0.0, 1,  1, 0, },
};

static float ParamConverter(int index, float value){ return X1Limiter::model2param(index, value); }

X1Limiter::X1Limiter(audioMasterCallback audioMaster)
  : AudioEffectX(audioMaster, KNumPrograms, KNumParams), ProcessBlock(), Locker(),
    currentFs(FV3_REVBASE_DEFAULT_FS)
{
  setNumInputs(2);
  setNumOutputs(2);
  setUniqueID(CCONST('W', 'n', 'L', '1'));
  canProcessReplacing();
#ifdef PLUGDOUBLE
  canDoubleReplacing();
#endif
  converter_type = FV3_SRC_LPF_IIR_2;
  limiterGain = 1;
  model = new REVMODEL;
  model->setSampleRate(FV3_REVBASE_DEFAULT_FS);

  programs = new X1LimiterProgram[numPrograms];
  for(int pc = 0;pc < KPresetPrograms;pc ++) programs[pc].setProgram(presetProgramName[pc], presetProgram[pc]);
  setProgram(0);
  X1LimiterEditor * _editor = new X1LimiterEditor(this);
  _editor->registerParamConverter(ParamConverter);
  editor = _editor;
}

X1Limiter::~X1Limiter()
{
  freeProcessBlock();
  delete[] programs;
  delete model;
}

bool X1Limiter::getEffectName (char* name)
{
  strcpy(name, EFFECT_NAME);
  return true;
}

bool X1Limiter::getVendorString (char* text)
{
  strcpy(text, VENDOR_STRING);
  return true;
}

bool X1Limiter::getProductString (char* text)
{
  strcpy(text, "Freeverb3");
  return true;
}

VstPlugCategory X1Limiter::getPlugCategory()
{
  return (kPlugCategRoomFx);
}

VstInt32 X1Limiter::canDo (char* text)
{
  if (!strcmp(text, "1in1out")) return 1;
  if (!strcmp(text, "2in2out")) return 1;
  if (!strcmp(text, "1in2out")) return 1;
  return -1;
}

bool X1Limiter::setBypass(bool onOff)
{
  byPass = onOff;
  return onOff;
}

VstInt32 X1Limiter::getProgram()
{
  return curProgram;
}

void X1Limiter::setProgram(VstInt32 program)
{
  if(program < numPrograms)
    {
      X1LimiterProgram * p = &programs[program];
      curProgram = program;
      setParameterM(KCeiling, p->fCeiling);
      setParameterM(KThreshold, p->fThreshold);
      setParameterM(KPreGain, p->fPreGain);
      setParameterM(KRelease, p->fRelease);
      setParameterM(KAttack, p->fAttack);
      setParameterM(KLookahead, p->fLookahead);
      setParameterM(KLookaheadRatio, p->fLookaheadRatio);
      setParameterM(KRMS, p->fRMS);
      setParameterM(KFactor, p->fFactor);
      setParameterM(KStLink, p->fStLink);
      setParameterM(KType, p->fType);
      lock();
      model->mute();
      ProcessBlock::mute();
      unlock();
    }
}

bool X1Limiter::getInputProperties(VstInt32 index, VstPinProperties* properties)
{
  bool returnCode = false;
  if(index == 0)
    {
      sprintf(properties->label, "%s Left Input", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  else if(index == 1)
    {
      sprintf(properties->label, "%s Right Input", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  return returnCode;
}

bool X1Limiter::getOutputProperties(VstInt32 index, VstPinProperties* properties)
{
  bool returnCode = false;
  if(index == 0)
    {
      sprintf(properties->label, "%s Left Output", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  else if(index == 1)
    {
      sprintf(properties->label, "%s Right Output", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  return returnCode;
}

bool X1Limiter::getProgramNameIndexed(VstInt32 category, VstInt32 index, char* text)
{
  if(index >= numPrograms) return false;
  strcpy(text, programs[index].name);
  return true;
}

void X1Limiter::setProgramName(char *name)
{
  strcpy(programs[curProgram].name, name);
}

void X1Limiter::getProgramName(char *name)
{
  strcpy(name, programs[curProgram].name);
}

void X1Limiter::setParameter(VstInt32 index, float value)
{
  setParameter(index, value, true);
  if(editor != NULL) ((AEffGUIEditor*)editor)->setParameter(index, value);
}

void X1Limiter::setParameterM(VstInt32 index, float value)
{
  setParameter(index, value, false);
  if(editor != NULL) ((AEffGUIEditor*)editor)->setParameter(index, model2param(index, value));
}

void X1Limiter::setParameter(VstInt32 index, float value, bool vstp)
{
  // The VST host stores parameters in float(0.0-1.0).
  if(vstp) value = param2model(index, value);
  // stores parameters to current Program.
  X1LimiterProgram * p = &programs[curProgram];
  switch (index)
    {
    case KCeiling:
      model->setCeiling(p->fCeiling = value);
      model->setThreshold(p->fCeiling + p->fThreshold);
      break;
    case KThreshold:
      model->setThreshold(p->fCeiling + (p->fThreshold = value));
      break;
    case KPreGain:
      limiterGain = UTILS::dB2R(p->fPreGain = value);
      break;
    case KRelease:
      model->setRelease(p->fRelease = value);
      break;
    case KAttack:
      model->setAttack(p->fAttack = value);
      break;
    case KLookahead:
      lock();
      model->setLookahead(p->fLookahead = value);
      model->mute();
      unlock();
      break;
    case KLookaheadRatio:
      model->setLookaheadRatio(p->fLookaheadRatio = value);
      break;
    case KRMS:
      lock();
      model->setRMS(p->fRMS = value);
      model->mute();
      unlock();
      break;
    case KFactor:
      break;
    case KStLink:
      p->fStLink = value;
      if(value > 0.5) model->setStereoLink(true);
      else model->setStereoLink(false);
      break;
    default:
      break;
    }
}

float X1Limiter::getParameter(VstInt32 index)
{
  float ret = 0;
  X1LimiterProgram * p = &programs[curProgram];
  switch (index)
    {
    case KCeiling:
      ret = p->fCeiling;
      break;
    case KThreshold:
      ret = p->fThreshold;
      break;
    case KPreGain:
      ret = p->fPreGain;
      break;
    case KRelease:
      ret = p->fRelease;
      break;
    case KAttack:
      ret = p->fAttack;
      break;
    case KLookahead:
      ret = p->fLookahead;
      break;
    case KLookaheadRatio:
      ret = p->fLookaheadRatio;
      break;
    case KRMS:
      ret = p->fRMS;
      break;
    case KFactor:
      ret = p->fFactor;
      break;
    case KStLink:
      ret = p->fStLink;
    default:
      break;
    }
  return model2param(index, ret);
}

void X1Limiter::getParameterName(VstInt32 index, char *label)
{
  if(index < KNumParams) strcpy(label, parameterName[index]);
}

void X1Limiter::getParameterDisplay(VstInt32 index, char *text)
{
  X1LimiterProgram * p = &programs[curProgram];
  switch (index)
    {
    case KCeiling:
      float2string(p->fCeiling, text, kVstMaxParamStrLen);
      break;
    case KThreshold:
      float2string(p->fThreshold, text, kVstMaxParamStrLen);
      break;
    case KPreGain:
      float2string(p->fPreGain, text, kVstMaxParamStrLen);
      break;
    case KRelease:
      float2string(p->fRelease, text, kVstMaxParamStrLen);
      break;
    case KAttack:
      float2string(p->fAttack, text, kVstMaxParamStrLen);
      break;
    case KLookahead:
      float2string(p->fLookahead, text, kVstMaxParamStrLen);
      break;
    case KLookaheadRatio:
      float2string(p->fLookaheadRatio, text, kVstMaxParamStrLen);
      break;
    case KRMS:
      float2string(p->fRMS, text, kVstMaxParamStrLen);
      break;
    case KFactor:
      float2string(p->fRMS, text, kVstMaxParamStrLen);
      break;
    case KStLink:
      if(p->fStLink > 0.5) snprintf(text, kVstMaxParamStrLen, "On");
      else snprintf(text, kVstMaxParamStrLen, "Off");
    default:
      break;
    }
}

#define LINCV(imin,imax,omin,omax,val) ((omin)+((val)-(imin))*((omax)-(omin))/((imax)-(imin)))

pfloat_t X1Limiter::pconv(int index, pfloat_t value, bool p2m)
{
  switch (index)
    {
    case KCeiling:
      if(p2m) return LINCV(0,1,-30,0,value);
      else return LINCV(-30,0,0,1,value);
    case KThreshold:
      if(p2m) return LINCV(0,1,-10,0,value);
      else return LINCV(-10,0,0,1,value);
    case KPreGain:
      if(p2m) return LINCV(0,1,0,30,value);
      else return LINCV(0,30,0,1,value);
    case KRelease:
    case KAttack:
      if(p2m) return 1000.*value*value*value*value;
      else return std::sqrt(std::sqrt(value/1000.));
    case KLookahead:
      if(p2m) return 8.*value*value*value*value;
      else return std::sqrt(std::sqrt(value/8.));
    case KLookaheadRatio:
      if(p2m) return LINCV(0,1,0.5,2,value);
      else return LINCV(0.5,2,0,1,value);
    case KRMS:
      if(p2m) return 100.*value*value*value*value;
      else return std::sqrt(std::sqrt(value/100.));
    case KFactor:
      if(p2m) return LINCV(0,1,0,5,value);
      else return LINCV(0,5,0,1,value);
    case KStLink:
    case KType:
    default:
      return value;
    }
}

pfloat_t X1Limiter::param2model(int index, float value)
{
  return pconv(index, (pfloat_t)value, true);
}

float X1Limiter::model2param(int index, pfloat_t value)
{
  return (float)pconv(index, value, false);
}

void X1Limiter::getParameterLabel(VstInt32 index, char *label)
{
  if(index < KNumParams) strcpy(label, parameterLabel[index]);
}

float X1Limiter::getSampleRate()
{
  return currentFs;
}

void X1Limiter::setSampleRate(float sampleRate)
{
  if(currentFs != sampleRate||model->getSampleRate() != sampleRate)
    {
      currentFs = (double)sampleRate;
      lock();
      model->setSampleRate(sampleRate);
      rmsL.setsize(currentFs*0.3), rmsR.setsize(currentFs*0.3);
      unlock();
    }
}

void X1Limiter::suspend()
{
  ;
}

void X1Limiter::resume()
{
  lock();
  if(updateBlockSize() > 0) allocProcessBlock(updateBlockSize());
  model->mute();
  ProcessBlock::mute();
  rmsL.mute(), rmsR.mute();
  unlock();
  setInitialDelay(model->getLatency()+ProcessBlock::getLatency());
}

void X1Limiter::process(float **inputs, float **outputs, VstInt32 sampleFrames)
{
  p_process(inputs, outputs, sampleFrames);
}

void X1Limiter::processReplacing(float **inputs, float **outputs, VstInt32 sampleFrames)
{
  p_processReplacing(inputs, outputs, sampleFrames);
}

#ifdef PLUGDOUBLE
void X1Limiter::processDoubleReplacing(double **inputs, double **outputs, VstInt32 sampleFrames)
{
  p_processDoubleReplacing(inputs, outputs, sampleFrames);
}
#endif

void X1Limiter::processLRModel(pfloat_t *inL, pfloat_t *inR, pfloat_t *outL, pfloat_t *outR, VstInt32 sampleFrames)
{
  float rms1 = 0, rms2 = 0;
  if(tryLock() == true)
    {
      for(VstInt32 i = 0;i < sampleFrames;i ++)
	{
	  rms1 = rmsL(inL[i]), rms2 = rmsR(inR[i]);
	  outL[i] = limiterGain * inL[i];
	  outR[i] = limiterGain * inR[i];
	}
      model->processreplace(outL,outR,outL,outR,(long int)sampleFrames);
      unlock();
    }
  else
    {
      memcpy(outL, inL, sizeof(pfloat_t)*sampleFrames);
      memcpy(outR, inR, sizeof(pfloat_t)*sampleFrames);
    }
  if(editor != NULL)
    {
      ((X1LimiterEditor*)editor)->setVuMeter(model->getCGain(), model->getDGain(), rms1, rms2);
    }
}

void X1Limiter::setConverterType(int type)
{
  converter_type = type;
}

void X1Limiter::setLatency(int size)
{
  ProcessBlock::setLatency(size);
}
