/**
 *  Freeverb3 user interface implementation
 *
 *  Copyright (C) 2006-2014 Teru Kamogashira
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef __StereoEnhancer_H
#define __StereoEnhancer_H

#define EFFECT_NAME "Freeverb3_StereoEnhancer"

#include <cstdlib>

#include <new>
#include <vector>
#include <string>
#include <audioeffectx.h>

#include <freeverb/stenh.hpp>

#include "ProcessBlock.hpp"
#include "FileLog.hpp"
#include "Locker.hpp"

#ifdef PLUGDOUBLE
typedef fv3::stenh_ STENH;
typedef double pfloat_t;
#else
typedef fv3::stenh_f STENH;
typedef float pfloat_t;
#endif

#if WIN32
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <process.h>
#endif

enum {
  KChValL = 0, KChValR, KBPF_LPF, KBPF_HPF,
  KBRF_LPF, KBRF_HPF, KWidth, KDry,
  KDiffusion, KThreshold, KRMS,
  KAttack, KRelease, KSoftKnee, KRatio,
  KBPF_Depth, KBRF_Depth, KAllDepth, KNumParams,
};

class StereoEnhancerProgram
{
public:
  StereoEnhancerProgram()
  {
    p[KChValL] = 1, p[KChValR] = 1, p[KBPF_LPF] = .5, p[KBPF_HPF] = .5,
      p[KBRF_LPF] = .5, p[KBRF_HPF] = .5, p[KWidth] = .7, p[KDry] = 1,
      p[KDiffusion] = .9, p[KThreshold] = -1, p[KRMS] = 0,
      p[KAttack] = 0, p[KRelease] = 0, p[KSoftKnee] = 10, p[KRatio] = 3,
      p[KBPF_Depth] = 4, p[KBRF_Depth] = 7, p[KAllDepth] = 7;
    strcpy(name, "(Default Preset)");
  }
  virtual ~StereoEnhancerProgram(){}
  void setProgram(char * n, float v[KNumParams])
  {
    for(int i = 0;i < KNumParams;i ++) p[i] = v[i];
  }
private:
  friend class StereoEnhancer;
  float p[KNumParams];
  char name[kVstMaxProgNameLen];
};

class StereoEnhancer :
  public AudioEffectX,
  public ProcessBlock,
  public Locker
{
public:
  StereoEnhancer(audioMasterCallback audioMaster) ALIGN_ARG_POINTER;
  ~StereoEnhancer();
  virtual VstPlugCategory getPlugCategory();
  virtual VstInt32 getProgram();
  virtual void     setProgram(VstInt32 program);
  virtual void setProgramName(char *name);
  virtual void getProgramName(char *name);
  virtual bool getProgramNameIndexed(VstInt32 category, VstInt32 index, char* text);
  virtual void getParameterLabel(VstInt32 index, char *label);
  virtual void getParameterDisplay(VstInt32 index, char *text);
  virtual void getParameterName(VstInt32 index, char *text);
  virtual bool getEffectName(char* name);
  virtual bool getVendorString(char* text);
  virtual bool getProductString(char* text);
  virtual VstInt32 canDo(char* text);
  virtual bool setBypass(bool onOff);
  virtual bool getInputProperties(VstInt32 index, VstPinProperties* properties);
  virtual bool getOutputProperties(VstInt32 index, VstPinProperties* properties);
  
  virtual void  suspend() ALIGN_ARG_POINTER;
  virtual void  resume() ALIGN_ARG_POINTER;
  virtual float getSampleRate() ALIGN_ARG_POINTER;
  virtual void  setSampleRate (float sampleRate) ALIGN_ARG_POINTER;
  virtual void  setParameter(VstInt32 index, float value) ALIGN_ARG_POINTER;
  virtual float getParameter(VstInt32 index) ALIGN_ARG_POINTER;
  virtual void process               (float **inputs, float **outputs, VstInt32 sampleFrames) ALIGN_ARG_POINTER;
  virtual void processReplacing      (float **inputs, float **outputs, VstInt32 sampleFrames) ALIGN_ARG_POINTER;
#ifdef PLUGDOUBLE
  virtual void processDoubleReplacing(double **inputs, double **outputs, VstInt32 sampleFrames) ALIGN_ARG_POINTER;
#endif
  virtual void processLRModel(pfloat_t *inL, pfloat_t *inR, pfloat_t *outL, pfloat_t *outR, VstInt32 sampleFrames) ALIGN_ARG_POINTER;

  void setLatency(int size);

  static pfloat_t param2model(int index, float value);
  static float model2param(int index, pfloat_t value);
  static pfloat_t pconv(int index, pfloat_t value, bool p2m);
  
private:
  StereoEnhancer(const StereoEnhancer& x);
  StereoEnhancer& operator=(const StereoEnhancer& x);

  void setParameterModel(VstInt32 index, float value);
  void setParameter(VstInt32 index, float value, bool vstp);

  StereoEnhancerProgram * programs;
  STENH * model;
  double currentFs;
  bool byPass;
};
#endif
