/**
 *  Freeverb3 user interface implementation
 *
 *  Copyright (C) 2006-2014 Teru Kamogashira
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "StereoEnhancer.hpp"
#include "GeneralEditor.hpp"

static char parameterName[KNumParams][kVstMaxParamStrLen+1] = {
  "ChValL", "ChValR", "BPF_LPF", "BPF_HPF", "BRF_LPF", "BRF_HPF",
  "Width", "Dry", "Diffus", "Thres", "RMS",
  "Attack", "Release", "SoftKnee", "Ratio",
  "BPFDepth", "BRFDepth", "AllDepth",
};

static char parameterLabel[KNumParams][kVstMaxParamStrLen+1] = {
  "%", "%", "%", "%", "%", "%",
  "%", "", "", "dB", "ms",
  "ms", "ms", "dB", "",
  "ms", "ms", "ms",
};

#define KNumPrograms 24

static const char * grpName[6] = { "LEV", "LIM", "FLT", "DEP", "", "", };
static const VstInt32 grpMtx[6][6] = {
  {KDry, KWidth, KDiffusion, -1, KChValL, KChValR,},
  {KThreshold, KAttack, KRelease, KRatio, KSoftKnee, KRMS,},
  {KBPF_LPF, KBPF_HPF, -1, KBRF_LPF, KBRF_HPF, -1,},
  {KBPF_Depth, KBRF_Depth, -1, KAllDepth, -1, -1,},
  {-1, -1, -1, -1, -1, -1,},
  {-1, -1, -1, -1, -1, -1,},
};
static const char * nameMtx[6][6] = {
  {"DRY", "WID", "DIF", "", "L", "R",},
  {"THR", "ATT", "REL", "RAT", "SKN", "RMS",},
  {"BLPF", "BHPF", "", "RLPF", "RHPF", "",},
  {"BPFD", "RPFD", "", "ALLD", "", "",},
  {"", "", "", "", "", "",},
  {"", "", "", "", "", "",},
};

static float ParamConverter(int index, float value){ return StereoEnhancer::model2param(index, value); }

StereoEnhancer::StereoEnhancer(audioMasterCallback audioMaster)
  : AudioEffectX(audioMaster, KNumPrograms, KNumParams), ProcessBlock(), Locker()
{
  setNumInputs(2);
  setNumOutputs(2);
  setUniqueID(CCONST('W', 'n', 'S', '1'));
  canProcessReplacing();
#ifdef PLUGDOUBLE
  canDoubleReplacing();
#endif
  model = new STENH();
  programs = new StereoEnhancerProgram[numPrograms];
  setSampleRate(48000);
  setProgram(0);
  GeneralEditor * _editor = new GeneralEditor(this);
  _editor->registerConst(EFFECT_NAME, grpName, grpMtx, nameMtx);
  _editor->registerParamConverter(ParamConverter);
  editor = _editor;
}

StereoEnhancer::~StereoEnhancer()
{
  delete[] programs;
  delete model;
}

bool StereoEnhancer::getEffectName (char* name)
{
  strcpy(name, EFFECT_NAME);
  return true;
}

bool StereoEnhancer::getVendorString (char* text)
{
  strcpy(text, VENDOR_STRING);
  return true;
}

bool StereoEnhancer::getProductString (char* text)
{
  strcpy(text, "Freeverb3");
  return true;
}

VstPlugCategory StereoEnhancer::getPlugCategory()
{
  return (kPlugCategRoomFx);
}

VstInt32 StereoEnhancer::canDo (char* text)
{
  if(!strcmp(text, "bypass"))  return 1;
  if(!strcmp(text, "1in1out")) return 1;
  if(!strcmp(text, "2in2out")) return 1;
  if(!strcmp(text, "1in2out")) return 1;
  return -1;
}

bool StereoEnhancer::getInputProperties(VstInt32 index, VstPinProperties* properties)
{
  bool returnCode = false;
  if(index == 0)
    {
      sprintf(properties->label, "%s Left Input", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  else if(index == 1)
    {
      sprintf(properties->label, "%s Right Input", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  return returnCode;
}

bool StereoEnhancer::getOutputProperties(VstInt32 index, VstPinProperties* properties)
{
  bool returnCode = false;
  if(index == 0)
    {
      sprintf(properties->label, "%s Left Output", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  else if(index == 1)
    {
      sprintf(properties->label, "%s Right Output", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  return returnCode;
}

bool StereoEnhancer::setBypass(bool onOff)
{
  byPass = onOff;
  return onOff;
}

VstInt32 StereoEnhancer::getProgram()
{
  return curProgram;
}

void StereoEnhancer::setProgram(VstInt32 program)
{
  if(program < numPrograms)
    {
      StereoEnhancerProgram * p = &programs[program];
      curProgram = program;
      for(int i = 0;i < KNumParams;i ++) setParameterModel(i, p->p[i]);
      lock();
      model->mute();
      ProcessBlock::mute();
      unlock();
    }
}

void StereoEnhancer::setProgramName(char *name)
{
  strcpy(programs[curProgram].name, name);
}

bool StereoEnhancer::getProgramNameIndexed(VstInt32 category, VstInt32 index, char* text)
{
  if(index >= numPrograms) return false;
  strcpy(text, programs[index].name);
  return true;
}

void StereoEnhancer::getProgramName(char *name)
{
  strcpy(name, programs[curProgram].name);
}

void StereoEnhancer::setParameter(VstInt32 index, float value)
{
  setParameter(index, value, true);
  if(editor != NULL) ((AEffGUIEditor*)editor)->setParameter(index, value);
}

void StereoEnhancer::setParameterModel(VstInt32 index, float value)
{
  setParameter(index, value, false);
  if(editor != NULL) ((AEffGUIEditor*)editor)->setParameter(index, model2param(index, value));
}

void StereoEnhancer::setParameter(VstInt32 index, float value, bool vstp)
{
  // The VST host stores parameters in float(0.0-1.0).
  if(vstp) value = param2model(index, value);
  // stores parameters to current Program.
  StereoEnhancerProgram * p = &programs[curProgram];
  switch(index)
    {
    case KChValL: model->setChValL(p->p[KChValL] = value); break;
    case KChValR: model->setChValR(p->p[KChValR] = value); break;
    case KWidth: model->setWidth(p->p[KWidth] = value); break;
    case KDry: model->setDry(p->p[KDry] = value); break;
    case KDiffusion: model->setDiffusion(p->p[KDiffusion] = value); break;
    case KThreshold: model->setThreshold(p->p[KThreshold] = value); break;
    case KAttack: model->setAttack(p->p[KAttack] = value); break;
    case KRelease: model->setRelease(p->p[KRelease] = value); break;
    case KSoftKnee: model->setSoftKnee(p->p[KSoftKnee] = value); break;
    case KRatio: model->setRatio(p->p[KRatio] = value); break;
    case KBPF_LPF: model->setBPF_LPF(p->p[KBPF_LPF] = value); break;
    case KBPF_HPF: model->setBPF_HPF(p->p[KBPF_HPF] = value); break;
    case KBRF_LPF: model->setBRF_LPF(p->p[KBRF_LPF] = value); break;
    case KBRF_HPF: model->setBRF_HPF(p->p[KBRF_HPF] = value); break;

    case KRMS:
      lock(); model->setRMS(p->p[KRMS] = value); unlock(); break;
    case KBPF_Depth:
      lock(); model->setBPFDepth(p->p[KBPF_Depth] = value); unlock(); break;
    case KBRF_Depth:
      lock(); model->setBRFDepth(p->p[KBRF_Depth] = value); unlock(); break;
    case KAllDepth:
      lock(); model->setOverallDepth(p->p[KAllDepth] = value); unlock(); break;
    default: break;
    }
}

float StereoEnhancer::getParameter(VstInt32 index)
{
  StereoEnhancerProgram * p = &programs[curProgram];
  return model2param(index, p->p[index]);
}

void StereoEnhancer::getParameterName(VstInt32 index, char *label)
{
  if(index < KNumParams) strcpy(label, parameterName[index]);
}

void StereoEnhancer::getParameterDisplay(VstInt32 index, char *text)
{
  StereoEnhancerProgram * p = &programs[curProgram];
  switch (index)
    {
    case KChValL: case KChValR:
    case KBPF_LPF: case KBPF_HPF: case KBRF_LPF: case KBRF_HPF:
    case KWidth:
      float2string(p->p[index]*100, text, kVstMaxParamStrLen);
      break;
    default:
      float2string(p->p[index], text, kVstMaxParamStrLen);
      break;
    }
}

#define LINCV(imin,imax,omin,omax,val) ((omin)+((val)-(imin))*((omax)-(omin))/((imax)-(imin)))

pfloat_t StereoEnhancer::pconv(int index, pfloat_t value, bool p2m)
{
  switch (index)
    {
    case KWidth:
      if(p2m) return LINCV(0,1,0,4,value);
      else return LINCV(0,4,0,1,value);
    case KDry:
      if(p2m) return LINCV(0,1,0,4,value);
      else return LINCV(0,4,0,1,value);
    case KDiffusion:
      if(p2m) return LINCV(0,1,0,3,value);
      else return LINCV(0,3,0,1,value);
    case KThreshold:
      if(p2m) return LINCV(0,1,-40,10,value);
      else return LINCV(-40,10,0,1,value);
    case KRMS:
      if(p2m) return LINCV(0,1,0,100,value);
      else return LINCV(0,100,0,1,value);
    case KAttack:
    case KRelease:
      if(p2m) return LINCV(0,1,0,1000,value);
      else return LINCV(0,1000,0,1,value);
    case KSoftKnee:
      if(p2m) return LINCV(0,1,0,20,value);
      else return LINCV(0,20,0,1,value);
    case KRatio:
      if(p2m) return LINCV(0,1,2,22,value);
      else return LINCV(2,22,0,1,value);
    case KBPF_Depth:
    case KBRF_Depth:
    case KAllDepth:
      if(p2m) return LINCV(0,1,0,80,value);
      else return LINCV(0,80,0,1,value);
    default:
      return value;
    }
}

pfloat_t StereoEnhancer::param2model(int index, float value)
{
  return pconv(index, (pfloat_t)value, true);
}

float StereoEnhancer::model2param(int index, pfloat_t value)
{
  return (float)pconv(index, value, false);
}

void StereoEnhancer::getParameterLabel(VstInt32 index, char *label)
{
  if(index < KNumParams) strcpy(label, parameterLabel[index]);
}

float StereoEnhancer::getSampleRate()
{
  return currentFs;
}

void StereoEnhancer::setSampleRate(float sampleRate)
{
  if(currentFs != sampleRate||model->getSampleRate() != sampleRate)
    {
      currentFs = sampleRate;
      lock();
      model->setSampleRate(sampleRate);
      unlock();
      setProgram(curProgram);
    }
}

void StereoEnhancer::suspend()
{
  ;
}

void StereoEnhancer::resume()
{
  lock();
  if(updateBlockSize() > 0) allocProcessBlock(updateBlockSize());
  model->mute();
  ProcessBlock::mute();
  unlock();
  setInitialDelay(model->getLatency()+ProcessBlock::getLatency());
}

void StereoEnhancer::process(float **inputs, float **outputs, VstInt32 sampleFrames)
{
  p_process(inputs, outputs, sampleFrames);
}

void StereoEnhancer::processReplacing(float **inputs, float **outputs, VstInt32 sampleFrames)
{
  p_processReplacing(inputs, outputs, sampleFrames);
}

#ifdef PLUGDOUBLE
void StereoEnhancer::processDoubleReplacing(double **inputs, double **outputs, VstInt32 sampleFrames)
{
  p_processDoubleReplacing(inputs, outputs, sampleFrames);
}
#endif

void StereoEnhancer::processLRModel(pfloat_t *inL, pfloat_t *inR, pfloat_t *outL, pfloat_t *outR, VstInt32 sampleFrames)
{
  try
    {
      if(tryLock() == true)
	{
	  model->processreplace(inL,inR,outL,outR,(long int)sampleFrames);
	  unlock();
	}
      else
	{
	  memcpy(outL, inL, sizeof(pfloat_t)*sampleFrames);
	  memcpy(outR, inR, sizeof(pfloat_t)*sampleFrames);
	}
    }
  catch(...)
    {
      ;
    }
}

void StereoEnhancer::setLatency(int size)
{
  ProcessBlock::setLatency(size);
}
