/**
 *  WindCompressor
 *
 *  Copyright (C) 2006-2014 Teru Kamogashira
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef __MBCompressor_HPP
#define __MBCompressor_HPP

#define EFFECT_NAME "Freeverb3_WindCompressor"

#include <new>
#include <vector>
#include <string>
#include <audioeffectx.h>

#include <fv3_config.h>

#include <freeverb/compmodel.hpp>
#include <freeverb/limitmodel.hpp>
#include <freeverb/fir3bandsplit.hpp>
#include <freeverb/firfilter.hpp>
#include <freeverb/firwindow.hpp>
#include <freeverb/utils.hpp>

#include "ProcessBlock.hpp"
#include "Locker.hpp"

#if WIN32
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <process.h>
#endif

#ifdef PLUGDOUBLE
typedef fv3::fir3bandsplit_ SPLITTER;
typedef fv3::compmodel_ COMPM;
typedef fv3::scomp_ SCOMP;
typedef fv3::limitmodel_ LIMITER;
typedef fv3::slimit_ SLIMIT;
typedef fv3::utils_ UTILS;
typedef fv3::delay_ DELAY;
typedef double pfloat_t;
#else
typedef fv3::fir3bandsplit_f SPLITTER;
typedef fv3::compmodel_f COMPM;
typedef fv3::scomp_f SCOMP;
typedef fv3::limitmodel_f LIMITER;
typedef fv3::slimit_f SLIMIT;
typedef fv3::utils_f UTILS;
typedef fv3::delay_f DELAY;
typedef float pfloat_t;
#endif

// Attack, Release, Threshold, Softknee, Ratio, Gain

enum {
  KFreqLowDivider, KFreqHighDivider, KCompRMS, KCompLookahead,
  KCompLowA, KCompLowR, KCompLowT, KCompLowSK, KCompLowRa, KCompLowG,
  KCompMidA, KCompMidR, KCompMidT, KCompMidSK, KCompMidRa, KCompMidG,
  KCompHighA, KCompHighR, KCompHighT, KCompHighSK, KCompHighRa, KCompHighG,
  KLimitRMS, KLimitLookahead, KLimitA, KLimitR, KLimitT, KLimitC, KAutoGain,
  KNumParams,
};

class MBCProgram
{
public:
  MBCProgram()
  {
    fFreqLowDivider = 140, fFreqHighDivider = 6300,
      fCompRMS = 30, fCompLookahead = 10,
      fCompLowA = 0, fCompLowR = 50, fCompLowT = -27, fCompLowSK = 10,
      fCompLowRa = 5, fCompLowG = -4,
      fCompMidA = 0, fCompMidR = 80, fCompMidT = -17, fCompMidSK = 10,
      fCompMidRa = 3, fCompMidG = -4,
      fCompHighA = 0, fCompHighR = 80, fCompHighT = -20, fCompHighSK = 10,
      fCompHighRa = 7, fCompHighG = -4,
      fLimitRMS = 0, fLimitLookahead = 0, fLimitA = 0, fLimitR = 100,
      fLimitT = -2, fLimitC = 0,
      fAutoGain = 1;
    strcpy(name, "(Default Preset)");
  }
  ~MBCProgram(){}
  void setProgram(char * n, float v[KNumParams])
  {
    fFreqLowDivider = v[0], fFreqHighDivider = v[1],
      fCompRMS = v[2], fCompLookahead = v[3],
      fCompLowA = v[4], fCompLowR = v[5], fCompLowT = v[6], fCompLowSK = v[7], fCompLowRa = v[8], fCompLowG = v[9],
      fCompMidA = v[10], fCompMidR = v[11], fCompMidT = v[12],fCompMidSK = v[13], fCompMidRa = v[14], fCompMidG = v[15],
      fCompHighA = v[16], fCompHighR = v[17], fCompHighT = v[18], fCompHighSK = v[19], fCompHighRa = v[20], fCompHighG = v[21],
      fLimitRMS = v[22], fLimitLookahead = v[23], fLimitA = v[24], fLimitR = v[25], fLimitT = v[26], fLimitC = v[27],
      fAutoGain = v[28];
    strcpy(name, n);
  }
  private:
  friend class MBCompressor;
  pfloat_t fFreqLowDivider, fFreqHighDivider, fCompRMS, fCompLookahead,
    fCompLowA, fCompLowR, fCompLowT, fCompLowSK, fCompLowRa, fCompLowG,
    fCompMidA, fCompMidR, fCompMidT, fCompMidSK, fCompMidRa, fCompMidG,
    fCompHighA, fCompHighR, fCompHighT, fCompHighSK, fCompHighRa, fCompHighG,
    fLimitRMS, fLimitLookahead, fLimitA, fLimitR, fLimitT, fLimitC, fAutoGain;
  char name[kVstMaxProgNameLen];
};

class MBCompressor :
  public AudioEffectX, ProcessBlock, Locker
{
public:
  MBCompressor(audioMasterCallback audioMaster);
  virtual ~MBCompressor();
  virtual VstPlugCategory getPlugCategory();
  virtual VstInt32 getProgram();
  virtual void     setProgram(VstInt32 program);
  virtual void setProgramName(char *name);
  virtual void getProgramName(char *name);
  virtual bool getProgramNameIndexed(VstInt32 category, VstInt32 index, char* text);
  virtual void getParameterLabel(VstInt32 index, char *label);
  virtual void getParameterDisplay(VstInt32 index, char *text);
  virtual void getParameterName(VstInt32 index, char *text);
  virtual bool getEffectName (char* name);
  virtual bool getVendorString (char* text);
  virtual bool getProductString (char* text);
  virtual VstInt32 canDo(char* text);
  virtual bool setBypass(bool onOff);
  virtual bool getInputProperties (VstInt32 index, VstPinProperties* properties);
  virtual bool getOutputProperties(VstInt32 index, VstPinProperties* properties);

  virtual void  suspend() ALIGN_ARG_POINTER;
  virtual void  resume() ALIGN_ARG_POINTER;
  virtual void  setSampleRate (float sampleRate) ALIGN_ARG_POINTER;
  virtual float getSampleRate() ALIGN_ARG_POINTER;
  virtual void  setParameter(VstInt32 index, float value) ALIGN_ARG_POINTER;
  virtual float getParameter(VstInt32 index) ALIGN_ARG_POINTER;
  virtual void process               (float **inputs, float **outputs, VstInt32 sampleFrames);
  virtual void processReplacing      (float **inputs, float **outputs, VstInt32 sampleFrames);
#ifdef PLUGDOUBLE
  virtual void processDoubleReplacing(double **inputs, double **outputs, VstInt32 sampleFrames);
#endif
  virtual void processLRModel(pfloat_t *inL, pfloat_t *inR, pfloat_t *ouL, pfloat_t *outR, VstInt32 sampleFrames) ALIGN_ARG_POINTER;
  
  void muteAll();
  void setFilterType(int type);
  void setFilterParameter(int value);
  void setTransitionBandwidth(int value);
  void setFragmentSize(int size);
  float getModelParameter(VstInt32 index);
  float model2param(int index, pfloat_t value);
  pfloat_t param2model(int index, float value);
  
private:
  pfloat_t pconv(int index, pfloat_t value, bool p2m);

  void setParameterM(VstInt32 index, float value);
  void setParameter(VstInt32 index, float value, bool vstp);
  float getParameter(VstInt32 index, bool vstp);

  int allocBlock(int size);
  void freeBlock();
  
  pfloat_t AGC(pfloat_t Threshold, pfloat_t Ratio);
  
  int sampleSize;
  SLOT lowBandInput, midBandInput, highBandInput, lowBandOutput, midBandOutput, highBandOutput, allOutput;
  MBCProgram * programs;
  SPLITTER split;
  COMPM compL, compM, compH;
  LIMITER limit;
  DELAY lLookaheadDelayL, lLookaheadDelayR;
  int totalDelay, limiterDelay;
  pfloat_t gainL, gainM, gainH, againL, againM, againH;
  float autoGain;
  int currentFs, filterType, filterParameter, fragmentSize, sfSize;
  bool byPass;
};

#endif
