/**
 *  Hibiki Reverb based on Zita-Rev
 *
 *  Copyright (C) 2006-2014 Teru Kamogashira
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "Hibiki_Reverb.hpp"
#include "GeneralEditor.hpp"

static char parameterName[KNumParams][kVstMaxParamStrLen+1] = {
  "SRCFact", "Dry", "ERefWet", "ERefWid", "ERefFac", "ERefSend", "LateWet", "Width", "PreDelay",
  "RoomSize", "RT60", "XOLow", "XOHigh", "RT60LoG", "RT60HiG", "Diffuse", "FDNApFb",
  "EarlyLPF", "EarlyHPF", "LateLPF", "LateHPF", "LFO1", "LFO2", "LFOFact", "Spin", "Wander", "SpinFact",
  "RevType", " ", " ", " ", " ",
};

static char parameterLabel[KNumParams][kVstMaxParamStrLen+1] = {
  "x", "dB", "dB", "x", "m", "x", "dB", "x", "ms",
  "m", "s", "Hz", "Hz", "x", "x", "%", "%",
  "Hz", "Hz", "Hz", "Hz", "Hz", "Hz", "x",
  "Hz", "ms", "x",
  "TL", "TE", " ", " ", " ",
};

#define KNumPrograms 64
#define KPresetPrograms 40

// LexHall, LexPlate, LexRoom, LexVintagePlate, LexChamber Late?FDN/COMB
// LexRandomHall Late?AP/Tank
static char presetProgramName[KPresetPrograms][kVstMaxProgNameLen] = {
  "Hibiki Hall",
  "Hibiki Hall (Dark)",
  "Hibiki Hall (Light)",

  "Hibiki Room",
  "Hibiki Room (Dark)",
  "Hibiki Room (Light)",

  "Small Narrow Hall",
  "Small Wide Hall",
  "Small Impact Hall",
  "Small Percussion Hall",
  "Small Clear Hall",
  "Small Vocal Hall 1",
  "Small Vocal Hall 2",
  "Small Vocal Hall 3",
  "Small Abrupt Hall",

  "Medium Hall",
  "Medium Deep Hall",
  "Medium Vocal Hall 1",
  "Medium Vocal Hall 2",
  "Medium Vocal Hall 3",
  "Medium Narrow Hall",
  "Medium Wide Hall",

  "Large Hall",
  "Large Early Hall",
  "Large Deep Hall",
  "Large Wide Hall",
  "Large Narrow Hall",
  "Large Vocal Hall 1",
  "Large Vocal Hall 2",
  "Large Vocal Hall 3",
  "Large Vocal Hall 4",
  "Large Neutral Hall",
  "Large Pop Hall",

  "Ginormous Hall",
  "Cathedral 1",
  "Cathedral 2",
  "Cathedral 3",

  "Studio E",
  "Studio F",
  "Studio G",
};

static float presetProgram[KPresetPrograms][KNumParams] = {
  //F  D  EW EWI ESiz  ES LW Wid PD  Siz  R60  XLow XHigh  RLo   RHi Dif ApF  ELPF HPF  LLPF LHPF  LFO1   2 LFOF Spn Wan  SpinFact
  {1, -10,-10, .7, 6.0, .4, -10, 1,18,  75, 2.5,  490, 6000, 1.2,  .35, 90, 90, 20000, 4, 13000, 4,  0.9, 1.3, .3, 2.4, 22, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .7, 6.0, .4, -10, 1,18,  75, 2.0,  490, 6000, 1.5,  .35, 90, 90,  7000, 4,  6000, 4,  0.9, 1.3, .3, 2.4, 22, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .7, 6.0, .4, -10, 1,18,  75, 2.7,  490, 7000, 1.4,  .75, 90, 90, 16000, 4, 10000, 4,  0.9, 1.3, .3, 2.4, 22, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },

  {1, -10,-10, .8, 3.0, .3, -10, 1, 6,  15, 0.6,  490, 6000, 0.9,  .80, 75, 75, 18000, 4, 14000, 4,  2.0, 3.0, .1, 0.3, 27, .2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .8, 3.0, .3, -10, 1, 7,  17, 0.7,  490, 6000, 0.9,  .90, 75, 75,  7000, 4,  6000, 4,  2.0, 3.0, .1, 0.3, 27, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .8, 3.0, .3, -10, 1, 7,  15, 0.6,  490, 6000, 0.9,  .90, 95, 95, 20000, 4, 20000, 4,  2.0, 3.0, .1, 0.3, 27, .2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },

  {1, -10,-10, .4, 4.0, .2, -10, 1,12,  26, 1.17, 500, 3000, 1.75, .35, 80, 80,  6000, 4,  5500, 4,  0.9, 1.3, .3, 2.5, 13, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .4, 4.0, .2, -10, 1,12,  24, 1.13, 500, 3750, 1.75, .35, 90, 90,  6000, 4,  8000, 4,  0.9, 1.3, .3, 2.5, 13, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .4, 4.0, .2, -10, 1,12,  23, 1.26, 400, 5500, 1.50, .35, 90, 90,  8000, 4,  6500, 4,  0.9, 1.3, .3, 2.5, 13, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .4, 4.0, .2, -10, 1,12,  24, 1.13, 250, 5500, 2.00, .35, 50, 50,  6000, 4,  4500, 4,  0.9, 1.3, .3, 2.0, 13, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .4, 4.0, .2, -10, 1, 0,18.5, 1.31, 500, 5500, 1.25, .35, 90, 90,  3250, 4,  7500, 4,  0.9, 1.3, .3, 3.3, 15, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .4, 4.0, .2, -10, 1, 0,19.5, 1.31, 500, 5000, 1.25, .35, 50, 50,  3250, 4,  6500, 4,  0.9, 1.3, .3, 3.1, 15, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .4, 4.0, .2, -10, 1, 0,19.5, 1.15, 500, 5000, 1.50, .35, 70, 70,  5500, 4,  7500, 4,  0.9, 1.3, .3, 2.9, 22, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .4, 4.0, .2, -10, 1, 0,  38, 1.08, 375, 5000, 1.25, .35, 70, 70,  5500, 4,  7500, 4,  0.9, 1.3, .3, 2.9, 22, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .4, 4.0, .2, -10, 1, 0,  42, 0.58, 800,20000, 1.25, .35, 90, 80,  4750, 4,  5500, 4,  0.9, 1.3, .3, 3.4,  9, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },

  {1, -10,-10, .4, 5.0, .2, -10, 1,14,  35, 1.69, 600, 4500, 1.50, .37, 85, 85,  7500, 4,  5500, 4,  0.9, 1.3, .3, 2.9, 15, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .4, 4.5, .2, -10, 1,14,  35, 1.33, 600, 4000, 1.75, .37, 85, 85,  4500, 4,  4000, 4,  0.9, 1.3, .3, 2.9, 15, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .4, 4.5, .2, -10, 1, 0,  24, 1.39, 600, 4000, 1.25, .37, 85, 85,  3250, 4,  6500, 4,  0.9, 1.3, .3, 2.8, 16, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .4, 4.0, .2, -10, 1, 0,  26, 1.53, 600, 5500, 1.50, .37, 85, 85,  5500, 4,  7500, 4,  0.9, 1.3, .3, 2.8, 22, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .4, 4.0, .2, -10, 1, 6,  52, 1.53, 600, 5000, 1.25, .37, 85, 85,  5500, 4,  7500, 4,  0.9, 1.3, .3, 2.9, 22, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .3, 4.0, .2, -10, .8,14, 35, 1.69, 600, 4500, 1.50, .37, 75, 75,  7500, 4,  4750, 4,  0.9, 1.3, .3, 2.9, 15, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .4, 4.0, .2, -10, 1,14,  35, 1.69, 600, 4500, 1.50, .37, 95, 95,  7500, 4,  4750, 4,  0.9, 1.3, .3, 2.9, 15, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  
  {1, -10,-10, .7, 5.5, .2, -10, 1,20,  32, 2.79, 600, 4500, 2.25, .37, 95, 95,  5000, 4,  6500, 4,  0.9, 1.3, .3, 2.1, 20, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .8, 5.5, .2, -10, 1,20,  27, 3.38, 600, 4500, 2.50, .37, 95, 95,  5000, 4,  6500, 4,  0.9, 1.3, .3, 2.1, 13, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .6, 5.5, .2, -10, 1,20,  40, 3.10, 600, 3250, 2.75, .37, 95, 95,  5000, 4,  6500, 4,  0.9, 1.3, .3, 2.1, 20, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .4, 6.5, .2, -10, 1,20,  32, 3.10, 600, 4500, 2.25, .37, 95, 95,  5000, 4,  6500, 4,  0.9, 1.3, .3, 2.1, 20, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .2, 4.5, .2, -10, .9,20, 32, 2.79, 600, 4500, 2.25, .37, 85, 85,  5000, 4,  6500, 4,  0.9, 1.3, .3, 2.1, 20, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .7, 5.0, .2, -10, 1,20,34.5, 3.00, 700, 4500, 2.25, .37, 95, 95,  5000, 4,  6500, 4,  0.9, 1.3, .3, 2.1, 17, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .4, 5.0, .2, -10, 1, 0,  29, 2.36, 600, 5000, 1.25, .37, 90, 90,  3250, 4,  6500, 4,  0.9, 1.3, .3, 3.1, 22, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .7, 5.0, .2, -10, 1, 0,32.5, 2.45, 600, 5000, 1.50, .30, 90, 90,  5000, 4,  7000, 4,  0.9, 1.3, .3, 2.9, 22, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .4, 5.0, .2, -10, 1, 0,32.5, 2.45, 600, 5000, 1.50, .30, 90, 90,  5000, 4,  7000, 4,  0.9, 1.3, .3, 2.9, 22, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .7, 5.0, .2, -10, 1,14,  35, 2.83, 550, 5000, 2.00, .30, 80, 80,  8000, 4,  6500, 4,  0.9, 1.3, .3, 2.2, 20, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .4, 5.0, .2, -10, 1,14,  35, 2.38, 450, 4750, 2.00, .30, 80, 80,  8000, 4,  5000, 4,  0.9, 1.3, .3, 3.4, 20, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },

  {1, -10,-10, .9, 6.0, .2, -10, .9,14, 60, 3.74, 750, 4000, 3.00, .30, 95, 95,  5500, 4,  5500, 4,  0.9, 1.3, .3, 2.6, 22, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .8, 6.0, .2, -10, 1,21,  57, 3.70, 850, 4000, 2.25, .30, 85, 85,  5000, 4,  6500, 4,  0.9, 1.3, .3, 2.1, 17, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .8, 7.0, .2, -10, 1,21,  66, 3.54, 850, 2500, 2.25, .40, 95, 95, 17000, 4,  5500, 4,  0.9, 1.3, .3, 1.8, 22, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .9, 7.0, .2, -10, 1,27,  66, 3.54,1000, 1800, 2.50, .40, 95, 95, 17000, 4,  5500, 4,  0.9, 1.3, .3, 1.8, 22, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  
  {1, -10,-10, .9, 3.5, .2, -10, 1, 5, 9.5, 0.74, 450, 5000, 1.25, .80, 55, 55,  7500, 4,  6000, 4,  2.0, 3.0, .1, 2.5,  7, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .9, 3.5, .2, -10, 1, 5, 7.0, 0.76, 150, 6000, 1.50, .70, 45, 45,  6000, 4,  4750, 4,  2.0, 3.0, .1, 2.5,  7, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
  {1, -10,-10, .9, 3.5, .2, -10, 1, 5, 14., 0.85, 225, 6000, 1.25, .70, 45, 45,  5000, 4,  4000, 4,  2.0, 3.0, .1, 2.5,  7, .3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
};

static const char * grpName[6] = { "LEV", "ERL", "RVB", "FLT", "RT60", "MOD", };
static const VstInt32 grpMtx[6][6] = {
  { KODry, -1, KOERefWet, KOERefSend, -1, KOLateWet, },
  { KOERefWet, KOERefFac, KOERefWid, KOERefSend, KOERType, -1, },
  { KOLateWet, KORoomSize, KOReverbType, KOSRCFact, KOWidth, KOPreDelay, },
  { KOEarlyLPF, KOEarlyHPF, KOLateLPF, KOLateHPF, KODiffuse, KOFDNApFb, },
  { KORT60, KOXOLow, KOXOHigh, KORT60LoG, KORT60HiG, -1, },
  { KOLFO1, KOLFO2, KOLFOFact, KOSpin, KOWander, KOSpinFact, },
};
static const char * nameMtx[6][6] = {
  { "DRY",  "",     "EWET", "ESEN", "",     "LWET", },
  { "EWET", "EFAC", "EWID", "ESEN", "ETYP", "",     },
  { "LWET", "SIZE", "LTYP", "LFAC", "LWID", "IDEL", },
  { "ELPF", "EHPF", "LLPF", "LHPF", "IDIF", "ADIF", },
  { "RT60", "XOL",  "XOH",  "RTLo", "RTHi", "",     },
  { "LFO1", "LFO2", "LFOF", "SPN",  "WAN",  "SPNF", },
};

static float ParamConverter(int index, float value){ return HibikiRev::model2param(index, value); }

HibikiRev::HibikiRev(audioMasterCallback audioMaster)
  : AudioEffectX(audioMaster, KNumPrograms, KNumParams), ProcessBlock(), Locker(),
    currentFs(FV3_REVBASE_DEFAULT_FS)
{
  setNumInputs(2);
  setNumOutputs(2);
  setUniqueID(CCONST('W', 'n', 'H', '1'));
  canProcessReplacing();
#ifdef PLUGDOUBLE
  canDoubleReplacing();
#endif
  converter_type = FV3_SRC_LPF_IIR_2;
  vERtoLate = dryDB = erDB = 0;

  dsp_eref = new EARLYREF();
  dsp_eref->setMuteOnChange(true);
  dsp_eref->setdryr(0);
  dsp_eref->setwet(0); // 0dB
  dsp_eref->setLRDelay(0.3);
  dsp_eref->setLRCrossApFreq(750, 4);
  dsp_eref->setDiffusionApFreq(150, 4);
  dsp_eref->setSampleRate(FV3_REVBASE_DEFAULT_FS);

  dsp_prev = new ZREV2();
  dsp_prev->setMuteOnChange(true);
  dsp_prev->setdryr(0); // mute dry signal
  dsp_prev->setSampleRate(FV3_REVBASE_DEFAULT_FS);

  programs = new HibikiRevProgram[numPrograms];
  for(int pc = 0;pc < KPresetPrograms;pc ++) programs[pc].setProgram(presetProgramName[pc], presetProgram[pc]);
  setProgram(0);
  GeneralEditor * _editor = new GeneralEditor(this);
  _editor->registerConst(EFFECT_NAME, grpName, grpMtx, nameMtx);
  _editor->registerParamConverter(ParamConverter);
  editor = _editor;
}

HibikiRev::~HibikiRev()
{
  freeProcessBlock();
  delete[] programs;
  delete dsp_eref;
  delete dsp_prev;
}

bool HibikiRev::getEffectName (char* name)
{
  strcpy (name, EFFECT_NAME);
  return true;
}

bool HibikiRev::getVendorString (char* text)
{
  strcpy (text, VENDOR_STRING);
  return true;
}

bool HibikiRev::getProductString (char* text)
{
  strcpy (text, "Freeverb3");
  return true;
}

VstPlugCategory HibikiRev::getPlugCategory()
{
  return (kPlugCategRoomFx);
}

VstInt32 HibikiRev::canDo (char* text)
{
  if (!strcmp (text, "1in1out")) return 1;
  if (!strcmp (text, "2in2out")) return 1;
  if (!strcmp (text, "1in2out")) return 1;
  return -1;
}

bool HibikiRev::getInputProperties(VstInt32 index, VstPinProperties* properties)
{
  bool returnCode = false;
  if(index == 0)
    {
      sprintf(properties->label, "%s Left Input", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  else if(index == 1)
    {
      sprintf(properties->label, "%s Right Input", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  return returnCode;
}

bool HibikiRev::getOutputProperties(VstInt32 index, VstPinProperties* properties)
{
  bool returnCode = false;
  if(index == 0)
    {
      sprintf(properties->label, "%s Left Output", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  else if(index == 1)
    {
      sprintf(properties->label, "%s Right Output", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  return returnCode;
}

VstInt32 HibikiRev::getProgram()
{
  return curProgram;
}

bool HibikiRev::setBypass(bool onOff)
{
  byPass = onOff;
  return onOff;
}

void HibikiRev::setProgram(VstInt32 program)
{
  if(programs == NULL||program >= numPrograms||program < 0) return;
  HibikiRevProgram * p = &programs[program];
  curProgram = program;
  setParameterM(KOSRCFact, p->fOSRCFact);
  setParameterM(KODry, p->fODry);
  setParameterM(KOERefWet, p->fOERefWet);
  setParameterM(KOERefWid, p->fOERefWid);
  setParameterM(KOERefFac, p->fOERefFac);
  setParameterM(KOERefSend, p->fOERefSend);
  setParameterM(KOLateWet, p->fOLateWet);
  setParameterM(KOWidth, p->fOWidth);
  setParameterM(KOPreDelay, p->fOPreDelay);
  setParameterM(KORoomSize, p->fORoomSize);
  setParameterM(KORT60, p->fORT60);
  setParameterM(KOXOLow, p->fOXOLow);
  setParameterM(KOXOHigh, p->fOXOHigh);
  setParameterM(KORT60LoG, p->fORT60LoG);
  setParameterM(KORT60HiG, p->fORT60HiG);
  setParameterM(KODiffuse, p->fODiffuse);
  setParameterM(KOFDNApFb, p->fOFDNApFb);
  setParameterM(KOEarlyLPF, p->fOEarlyLPF);
  setParameterM(KOEarlyHPF, p->fOEarlyHPF);
  setParameterM(KOLateLPF, p->fOLateLPF);
  setParameterM(KOLateHPF, p->fOLateHPF);
  setParameterM(KOLFO1, p->fOLFO1);
  setParameterM(KOLFO2, p->fOLFO2);
  setParameterM(KOLFOFact, p->fOLFOFact);
  setParameterM(KOSpin, p->fOSpin);
  setParameterM(KOWander, p->fOWander);
  setParameterM(KOSpinFact, p->fOSpinFact);
  setParameterM(KOReverbType, p->fOReverbType);
  setParameterM(KOERType, p->fOERType);
  lock();
  dsp_eref->mute();
  dsp_prev->mute();
  ProcessBlock::mute();
  unlock();
}

void HibikiRev::setProgramName(char *name)
{
  strcpy(programs[curProgram].name, name);
}

bool HibikiRev::getProgramNameIndexed(VstInt32 category, VstInt32 index, char* text)
{
  if(index >= numPrograms) return false;
  strcpy(text, programs[index].name);
  return true;
}

void HibikiRev::getProgramName(char *name)
{
  strcpy(name, programs[curProgram].name);
}

void HibikiRev::setParameter(VstInt32 index, float value)
{
  setParameter(index, value, true);
  if(editor != NULL) ((AEffGUIEditor*)editor)->setParameter(index, value);
}

void HibikiRev::setParameterM(VstInt32 index, float value)
{
  setParameter(index, value, false);
  if(editor != NULL) ((AEffGUIEditor*)editor)->setParameter(index, model2param(index, value));
}

void HibikiRev::setParameter(VstInt32 index, float value, bool vstp)
{
  // The VST host stores parameters in float(0.0-1.0).
  if(vstp) value = param2model(index, value);
  // stores parameters to current Program.
  HibikiRevProgram * p = &programs[curProgram];
  long factorp = 1, type = 0;
  switch (index)
    {
    case KOSRCFact:
      p->fOSRCFact = value;
      factorp = (long)std::ceil(value); if(factorp == 0) factorp = 1;
      if(factorp != dsp_prev->getOSFactor())
	{
	  lock();
	  dsp_prev->setOSFactor((int)factorp, converter_type);
	  unlock();
	}
      break;
    case KODry:
      dryDB = (p->fODry = value);
      break;
    case KOERefWet:
      erDB = (p->fOERefWet = value);
      break;
    case KOERefWid:
      dsp_eref->setwidth(p->fOERefWid = value);
      break;
    case KOERefFac:
      lock();
      dsp_eref->setRSFactor((p->fOERefFac = value)/7.);
      unlock();
      break;
    case KOERefSend:
      vERtoLate = (p->fOERefSend = value);
      break;

    case KOLateWet:
      dsp_prev->setwet(p->fOLateWet = value);
      break;
    case KOWidth:
      dsp_prev->setwidth(p->fOWidth = value);
      break;
    case KOPreDelay:
      lock();
      dsp_prev->setPreDelay(p->fOPreDelay = value);
      unlock();
      break;
    case KORoomSize:
      lock();
      dsp_prev->setRSFactor((p->fORoomSize = value)/80.);
      unlock();
      break;
    case KORT60:
      p->fORT60 = value;
      // for exception
      if(value >= 0.01) dsp_prev->setrt60(value);
      else dsp_prev->setrt60(0.01);
      break;
      
    case KOXOLow:
      dsp_prev->setxover_low(p->fOXOLow = value);
      break;
    case KOXOHigh:
      dsp_prev->setxover_high(p->fOXOHigh = value);
      break;
    case KORT60LoG:
      dsp_prev->setrt60_factor_low(p->fORT60LoG = value);
      break;
    case KORT60HiG:
      dsp_prev->setrt60_factor_high(p->fORT60HiG = value);
      break;
    case KODiffuse:
      dsp_prev->setidiffusion1((p->fODiffuse = value)/100.*.75);
      break;
    case KOFDNApFb:
      dsp_prev->setapfeedback((p->fOFDNApFb = value)/100.*.75);
      break;

    case KOEarlyLPF:
      dsp_eref->setoutputlpf(p->fOEarlyLPF = value);
      break;
    case KOEarlyHPF:
      dsp_eref->setoutputhpf(p->fOEarlyHPF = value);
      break;
    case KOLateLPF:
      dsp_prev->setoutputlpf(p->fOLateLPF = value);
      break;
    case KOLateHPF:
      dsp_prev->setoutputhpf(p->fOLateHPF = value);
      break;

    case KOLFO1:
      dsp_prev->setlfo1freq(p->fOLFO1 = value);
      break;
    case KOLFO2:
      dsp_prev->setlfo2freq(p->fOLFO2 = value);
      break;
    case KOLFOFact:
      dsp_prev->setlfofactor(p->fOLFOFact = value);
      break;
      
    case KOSpin:
      dsp_prev->setspin(p->fOSpin = value);
      break;
    case KOWander:
      lock();
      dsp_prev->setwander(p->fOWander = value);
      unlock();
      break;
    case KOSpinFact:
      dsp_prev->setspinfactor(p->fOSpinFact = value);
      break;
      
    case KOReverbType:
      type = (long)std::ceil(value);
      switch(type)
	{
	case 2:
	  dsp_prev->setReverbType(FV3_REVTYPE_ZREV);
	  p->fOReverbType = 2;
	  break;
	case 1:
	default:
	  dsp_prev->setReverbType(FV3_REVTYPE_ZREV2);
	  p->fOReverbType = 1;
	  break;
	}
      break;

    case KOERType:
      type = (long)std::ceil(value);
      lock();
      switch(type)
	{
	  // case 3:
	  // dsp_eref->loadPresetReflection(FV3_EARLYREF_PRESET_2);
	  // p->fOERType = 3;
	  // break;
	case 2:
	  dsp_eref->loadPresetReflection(FV3_EARLYREF_PRESET_1);
	  p->fOERType = 2;
	  break;
	case 1:
	default:
	  dsp_eref->loadPresetReflection(FV3_EARLYREF_PRESET_0);
	  p->fOERType = 1;
	  break;
	}
      unlock();
      break;
      
    case KORsvd3:
    case KORsvd4:
    case KORsvd5:
    case KORsvd6:
    case KORsvd7:
    case KORsvd8:
    case KORsvd9:
    case KORsvd10:
    default:
      break;
    }
}

float HibikiRev::getParameter(VstInt32 index)
{
  float ret = 0.0f;
  HibikiRevProgram * p = &programs[curProgram];
  ret = p->getParameterValue(index);
  return model2param(index, ret);
}

#define LINCV(imin,imax,omin,omax,val) ((omin)+((val)-(imin))*((omax)-(omin))/((imax)-(imin)))

pfloat_t HibikiRev::pconv(int index, pfloat_t value, bool p2m)
{
  switch (index)
    {
    case KOSRCFact:
      if(p2m) return LINCV(0,1,0,4,value);
      else return LINCV(0,4,0,1,value);

    case KODry:
    case KOERefWet:
    case KOLateWet:
      if(p2m) return LINCV(0,1,-70,10,value);
      else return LINCV(-70,10,0,1,value);
      
    case KOERefWid:
    case KOWidth:
      if(p2m) return LINCV(0,1,-1,1,value);
      else return LINCV(-1,1,0,1,value);

    case KOERefFac:
      if(p2m) return std::pow(value, (pfloat_t)3.)*25.+1.;
      else return std::pow((value-1.)/25., (pfloat_t)1./3.);

    case KOPreDelay:
      if(p2m) return (value*value)*1000.;
      else return std::sqrt(value/1000.);
    case KORoomSize:
      if(p2m) return std::pow(value, (pfloat_t)3.)*180.+2.;
      else return std::pow((value-2.)/180., (pfloat_t)1./3.);

    case KORT60:
      if(p2m) return std::pow(value, (pfloat_t)6.)*30.+.2;
      else return std::pow((value-.2)/30., (pfloat_t)1./6.);

    case KORT60LoG:
    case KORT60HiG:
      if(p2m) return LINCV(0,1,0.2,4.2,value);
      else return LINCV(0.2,4.2,0,1,value);

    case KOXOLow:
      if(p2m) return (value*value)*18000.+40.;
      else return std::sqrt((value-40.)/18000.);
    case KOXOHigh:
      if(p2m) return (value*value)*20000.+40.;
      else return std::sqrt((value-40.)/20000.);

    case KOLateLPF:
    case KOEarlyLPF:
      if(p2m) return (value*value)*20000.;
      else return std::sqrt(value/20000.);

    case KOLateHPF:
    case KOEarlyHPF:
      if(p2m) return (value*value)*20000.;
      else return std::sqrt(value/20000.);

    case KOSpin:
    case KOLFO1:
    case KOLFO2:
      if(p2m) return (value*value)*10.;
      else return std::sqrt(value/10.);
      
    case KOWander:
      if(p2m) return LINCV(0,1,0,30,value);
      else return LINCV(0,30,0,1,value);

    case KODiffuse:
    case KOFDNApFb:
      if(p2m) return LINCV(0,1,0,100,value);
      else return LINCV(0,100,0,1,value);

    case KOReverbType:
    case KOERType:
      if(p2m) return LINCV(0,1,1,6,value);
      else return LINCV(1,6,0,1,value);

    case KOSpinFact:
    case KOLFOFact:
    case KOERefSend:
    default:
      return value;
    }
}

pfloat_t HibikiRev::param2model(int index, float value)
{
  return pconv(index, (pfloat_t)value, true);
}

float HibikiRev::model2param(int index, pfloat_t value)
{
  return (float)pconv(index, value, false);
}

void HibikiRev::getParameterName(VstInt32 index, char *label)
{
  if(index < KNumParams) strcpy(label, parameterName[index]);
}

void HibikiRev::getParameterDisplay(VstInt32 index, char *text)
{
  HibikiRevProgram * p = &programs[curProgram];
  switch (index)
    {
    case KOSRCFact:
      snprintf(text, kVstMaxParamStrLen, "%ld", dsp_prev->getOSFactor());
      break;
    case KOReverbType:
    case KOERType:
      int2string((VstInt32)p->getParameterValue(index), text, kVstMaxParamStrLen);
      break;
    default:
      float2string(p->getParameterValue(index), text, kVstMaxParamStrLen);
      break;
    }
}

void HibikiRev::getParameterLabel(VstInt32 index, char *label)
{
  if(index < KNumParams) strcpy(label, parameterLabel[index]);
}

float HibikiRev::getSampleRate()
{
  return currentFs;
}

void HibikiRev::setSampleRate(float sampleRate)
{
  if(currentFs != sampleRate||dsp_prev->getSampleRate() != sampleRate)
    {
      currentFs = (double)sampleRate;
      lock();
      dsp_eref->setSampleRate(sampleRate);
      dsp_prev->setSampleRate(sampleRate);
      unlock();
    }
}

void HibikiRev::suspend()
{
  ;
}

void HibikiRev::resume()
{
  lock();
  if(updateBlockSize() > 0)
    {
      allocProcessBlock(updateBlockSize());
      tmp1Block.alloc(updateBlockSize(), 2);
      tmp2Block.alloc(updateBlockSize(), 2);
    }
  dsp_eref->mute();
  dsp_prev->mute();
  ProcessBlock::mute();
  unlock();
  setInitialDelay(dsp_eref->getLatency()+dsp_prev->getLatency()+ProcessBlock::getLatency());
}

void HibikiRev::process(float **inputs, float **outputs, VstInt32 sampleFrames)
{
  p_process(inputs, outputs, sampleFrames);
}

void HibikiRev::processReplacing(float **inputs, float **outputs, VstInt32 sampleFrames)
{
  p_processReplacing(inputs, outputs, sampleFrames);
}

#ifdef PLUGDOUBLE
void HibikiRev::processDoubleReplacing(double **inputs, double **outputs, VstInt32 sampleFrames)
{
  p_processDoubleReplacing(inputs, outputs, sampleFrames);
}
#endif

void HibikiRev::processLRModel(pfloat_t *inL, pfloat_t *inR, pfloat_t *outL, pfloat_t *outR, VstInt32 sampleFrames)
{
  UTILS::mute(outL, sampleFrames);
  UTILS::mute(outR, sampleFrames);
  
  if(tmp1Block.getsize() < sampleFrames)
    {
      lock();
      try
	{
	  tmp1Block.alloc(sampleFrames, 2);
	  tmp2Block.alloc(sampleFrames, 2);
	}
      catch(std::bad_alloc){ unlock(); return; }
      unlock();
    }

  if(tryLock() == true)
    {
      dsp_eref->processreplace(inL,inR,tmp1Block.L,tmp1Block.R,(long int)sampleFrames);
      for(long i = 0;i < sampleFrames;i ++)
	{
	  tmp2Block.L[i] = tmp1Block.L[i]*vERtoLate + inL[i];
	  tmp2Block.R[i] = tmp1Block.R[i]*vERtoLate + inR[i];
	}
      dsp_prev->processreplace(tmp2Block.L,tmp2Block.R,outL,outR,(long int)sampleFrames);
      pfloat_t dryR = UTILS::dB2R(dryDB), erR = UTILS::dB2R(erDB);
      for(long i = 0;i < sampleFrames;i ++)
	{
	  outL[i] += tmp1Block.L[i]*erR + inL[i]*dryR;
	  outR[i] += tmp1Block.R[i]*erR + inR[i]*dryR;
	}
      unlock();
    }
}

void HibikiRev::setConverterType(int type)
{
  converter_type = type;
}

void HibikiRev::setLatency(int size)
{
  ProcessBlock::setLatency(size);
}
