/**
 *  Impulser2
 *
 *  Copyright (C) 2006-2014 Teru Kamogashira
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "Freeverb.hpp"
#include "FreeverbEditor.hpp"

static char parameterName[KMaxParams][kVstMaxParamStrLen-2] =
  { "Width", "Wet", "MuteW", "LPF", "HPF", "M2S", "Swap", "Delay", "LRBal", };
// "Dry" Parameter was changed to "Mute-Wet".
// Values of "Dry" is < 1 in most cases, so Mute-Wet is on only if the value is 1. The Dry Parameter will be moved to oaParameter.

static char parameterLabel[KMaxParams][kVstMaxParamStrLen+1] =
  { "%", "dB", "dB", "%", "%",  "", "", "s", "L-C-R", };

static char oaParameterName[KOAMaxParams][kVstMaxParamStrLen-2] =
 { "Rele", "Ceili", "Thres", "Dry", "SkipL", };

static char oaParameterLabel[KOAMaxParams][kVstMaxParamStrLen-2] =
  { "ms", "dB", "dB", "dB", "", };

static void executeEvent(lfEventExecuterInfo * info)
{
  if(info == NULL||info->kmevent == KMEventNull) return;
  CFILELOADER fileLoader;
  const char *newFilename = NULL, *prevFilename = NULL;
  Freeverb * fv = static_cast<Freeverb*>(info->fv);
  FreeverbEditor * fve = *static_cast<FreeverbEditor**>(info->fve);
  int ret = 0;
  fv->writeLogA("executeEvent: EventId=0x%02x\n", info->kmevent);
  switch(info->kmevent)
    {
    case KMEventLoadBG:
    case KMEventLoadFG:
      if(strcmp(info->targetFilename.c_str(), "") == 0) break;
      fv->writeLogA("executeEvent: load \"%s\" to [%1x]\n", info->targetFilename.c_str(), info->channel);
      newFilename = info->targetFilename.c_str();
      prevFilename = fv->getSlotFileNameA(info->channel);
      if(newFilename != NULL&&prevFilename != NULL)
	{
	  if(strcmp(prevFilename, newFilename) == 0)
	    {
	      fv->writeLogA("executeEvent: The file is already loaded.\n");
	      break;
	    }
	}
      if(strcmp(newFilename, "") == 0||newFilename == NULL)
	{
	  fv->writeLogA("executeEvent: The target file is (null).\n");
	  break;
	}
      if(fve != NULL) fve->setSlotLabelA(info->channel, " Loading...");
      fileLoader.setAHDSR(info->ta, info->th, info->td, info->ts, info->tr);
      ret = fileLoader.load(newFilename, (int)fv->getSampleRate(),
			    PARAM2STRETCH(fv->getNRTParameter(KNRTParam(info->channel,KIRStretch))), 100, fv->getConverterType());
      if(ret == 0)
	{
	  fv->setSlotFileNameA(info->channel, newFilename);
	  fv->loadToSlot(info->channel, 0, fileLoader.out.getsize(), fileLoader.out.L, fileLoader.out.R);
	  if(fve != NULL)
	    {
	      fve->setSlotLabelA(info->channel, newFilename);
	      if(fv->getCurrentSlot() == info->channel) fve->setFileNameLabelA(fv->getSlotFileNameA(info->channel));
	    }
	}
      else
	{
	  if(info->errorMessage)
	    MacMessageBox("The file is not supported.", "Freeverb3 Impulser2: Load failed.");
	  if(fve != NULL) fve->setSlotLabelA(info->channel, fv->getSlotFileNameA(info->channel));
	}
      break;
      
    case KMEventUnloadBG:
    case KMEventUnloadFG:
      fv->writeLogA("executeEvent: unload [%1x]\n", info->channel);
      if(fve != NULL)
	{
	  fve->setSlotLabelA(info->channel, " ");
	  if(fv->getCurrentSlot() == info->channel) fve->setFileNameLabelA(" ");
	}
      fv->setSlotFileNameA(info->channel, "");
      fv->unloadSlot(info->channel, 0);
      break;

    case KMEventNull:
    default:
      break;
    }
}

#ifdef WIN32
static unsigned __stdcall lfEventThread(void * vdParam)
#else
static void * lfEventThread(void * vdParam)
#endif
{
  lfEventClassInfo *info = (lfEventClassInfo*)vdParam;
#ifdef WIN32
  unsigned int threadId = GetCurrentThreadId();
  wchar_t eventName[_MAX_PATH];
  wsprintfW(eventName, FV_EVENT_PREFIX L"%d", threadId);
  HANDLE threadEvent = OpenEventW(EVENT_ALL_ACCESS, FALSE, eventName);
  Freeverb * fv = (Freeverb*)info->fv;
#endif
  while(1)
    {
      lfEventExecuterInfo lfeeinfo;
#ifdef WIN32
      Sleep(0);
      WaitForSingleObject(threadEvent, INFINITE);
      ResetEvent(threadEvent);
#else
      sleep(0);
      info->event->wait();
      info->event->reset();
#endif
      while(1)
	{
	  lfeeinfo.kmevent = KMEventNull;
	  info->cueLocker->lock();
	  if(info->ecue->size() > 0)
	    {
	      lfeeinfo = *info->ecue->begin();
	      info->ecue->erase(info->ecue->begin());
	    }
	  info->cueLocker->unlock();
	  executeEvent(&lfeeinfo);
	  info->cueLocker->lock();
	  int cueSize = info->ecue->size();
	  info->cueLocker->unlock();
	  if(cueSize == 0) break;
	}
      if(*info->flags != 0) break;
    }
#ifdef WIN32
  fv->writeLogA("lfEventThread: lfEventThread End Id=%d\n", threadId);
  CloseHandle(threadEvent);  
  _endthreadex(0);
#else
  pthread_exit(NULL);
#endif
  return 0;
}

Freeverb::Freeverb(audioMasterCallback audioMaster)
  : AudioEffectX(audioMaster, 1, KNumParams*MODEL_SLOT_SIZE+KOANumParams),
    ProcessBlock(), FileLog(),
    currentFs(FV3_REVBASE_DEFAULT_FS)
{
  validModel = false;
  setNumInputs(2);
  setNumOutputs(2);
  setUniqueID(CCONST('W', 'n', 'I', '1'));
  canProcessReplacing();
#ifdef PLUGDOUBLE
  canDoubleReplacing();
#endif
  programsAreChunks(true);
  strcpy(programName, "Default Reverb");
  //currentBlockSize = updateBlockSize();
  currentBlockSize = 0;
  converter_type = SRC_SINC_BEST_QUALITY;
  setFragmentSize(1024);
  setFactor(16);
  currentSlot = 0;
  setInitialDelay(0);
  dryValue = DB2PARAM(-5.0);
  dryValueDB = -5.0;
  dryValueR = UTILS::dB2R(-5.0);
  editor = new FreeverbEditor(this);
  limiter.setSampleRate(FV3_REVBASE_DEFAULT_FS);
  for(int si = 0;si < MODEL_SLOT_SIZE;si ++) model[si][0] = new IR2;
  initModels();  
  callProcess = callDouble = callReplace = callEvents = 0;
  byPass = conf_mt = skipLimiter = false;
  zl = true;
  // Start Event Executer Thread
  hostThreadData.fv = this;
  hostThreadData.fve = (void**)&editor;
  hostThreadData.flags = &threadFlags;
  hostThreadData.ecue = &eventCue;
  hostThreadData.cueLocker = &eventCueLocker;
  threadFlags = 0;
#ifdef WIN32
  lEventThreadHandle = (HANDLE)_beginthreadex(NULL, 0, lfEventThread, &hostThreadData, CREATE_SUSPENDED, &threadId);
  wsprintfW(eventName, FV_EVENT_PREFIX L"%d", threadId);
  event = CreateEventW(NULL, TRUE, FALSE, eventName);
  ResumeThread(lEventThreadHandle);
#else
  hostThreadData.event = &event;
  event.reset();
  pthread_create(&lEventThreadHandle, NULL, lfEventThread, &hostThreadData);
#endif
}

Freeverb::~Freeverb()
{
  writeLogA("Impulser2: ~freeverb: Process %d Replace %d DoubleReplace %d Events %d\n", callProcess, callReplace, callDouble, callEvents);
  freeProcessBlock();
  for(int si = 0;si < MODEL_SLOT_SIZE;si ++) delete model[si][0];
  // End Event Executer Thread
  threadFlags = 1;
#ifdef WIN32
  HANDLE trigger = OpenEventW(EVENT_ALL_ACCESS, FALSE, eventName);
  SetEvent(trigger);
  WaitForSingleObject(lEventThreadHandle, INFINITE);
  CloseHandle(lEventThreadHandle);
  CloseHandle(trigger);
  CloseHandle(event);
#else
  event.trigger();
  pthread_join(lEventThreadHandle, NULL);
#endif
}

void Freeverb::initModels()
{
  for(int si = 0;si < MODEL_SLOT_SIZE;si ++)
    {
      model[si][0]->setInitialDelay(0);
      model[si][0]->setdry(-100);
      model[si][0]->setwet(-25);
      model[si][0]->setLPF(0);
      model[si][0]->setHPF(0);
      model[si][0]->setwidth(1);
      model[si][0]->setLRBalance(0);
      muteWet[si][0] = m2s[si] = swap[si] = false;
      conf_idelay[si][0] = 0;
      conf_stretch[si][0] = 0.5f;
      conf_attack[si][0] = 0;
      conf_hold[si][0] = 1;
      conf_decay[si][0] = 0;
      conf_sustain[si][0] = 0.8f;
      conf_release[si][0] = 0;
      slotFileNameA[si] = "";
    }
}

void Freeverb::setFragmentSize(int size)
{
  if(size <= 0) return;
  conf_fragmentSize = size;
}

void Freeverb::setFactor(int size)
{
  if(size <= 0) size = DEFAULT_FACTOR;
  conf_factor = size;
}

void Freeverb::setZeroLatency(int on)
{
  writeLogA("Impulser2: ZeroLatency: %d\n", on);
  if(on > 0)
    {
      zl = true;
      setInitialDelay(0);
    }
}

void Freeverb::setMT(int on)
{
  writeLogA("Impulser2: MultiThread: %d\n", on);
  for(int si = 0;si < MODEL_SLOT_SIZE;si ++) delete model[si][0];
  if(on > 0)
    {
      conf_mt = true;
      for(int si = 0;si < MODEL_SLOT_SIZE;si ++) model[si][0] = new IR3W;
      initModels();
    }
  else
    {
      conf_mt = false;
      for(int si = 0;si < MODEL_SLOT_SIZE;si ++) model[si][0] = new IR2;
      initModels();
    }
}

bool Freeverb::getEffectName (char* name)
{
  strcpy(name, EFFECT_NAME);
  return true;
}

bool Freeverb::getVendorString (char* text)
{
  strcpy(text, VENDOR_STRING);
  return true;
}

bool Freeverb::getProductString (char* text)
{
  strcpy(text, "Freeverb3");
  return true;
}

VstInt32 Freeverb::canDo(char* text)
{
  VstInt32 ret = -1;
  writeLogA("host requested canDo(%s):", text);
  if(!strcmp(text, "bypass"))  ret = 1;
  if(!strcmp(text, "1in1out")) ret = 1;
  if(!strcmp(text, "2in2out")) ret = 1;
  if(!strcmp(text, "1in2out")) ret = 1;
  if(!strcmp(text, "receiveVstEvents")) ret = 1;
  if(!strcmp(text, "receiveVstMidiEvent")) ret = 1;
  if(!strcmp(text, "receiveVstMidiEvents")) ret = 1; // for typo
  writeLogA("%d\n", ret);
  return ret;
}

bool Freeverb::getInputProperties(VstInt32 index, VstPinProperties* properties)
{
  bool returnCode = false;
  if(index == 0)
    {
      sprintf(properties->label, "%s Left Input", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  else if(index == 1)
    {
      sprintf(properties->label, "%s Right Input", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  return returnCode;
}

bool Freeverb::getOutputProperties(VstInt32 index, VstPinProperties* properties)
{
  bool returnCode = false;
  if(index == 0)
    {
      sprintf(properties->label, "%s Left Output", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  else if(index == 1)
    {
      sprintf(properties->label, "%s Right Output", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  return returnCode;
}

VstPlugCategory Freeverb::getPlugCategory()
{
  return (kPlugCategRoomFx);
}

VstInt32 Freeverb::processEvents(VstEvents* ev)
{
  callEvents ++;
  for (VstInt32 i = 0;i < ev->numEvents;i ++)
    {
      if((ev->events[i])->type != kVstMidiType)
	continue;
      VstMidiEvent* event = (VstMidiEvent*)ev->events[i];
      char* midiData = event->midiData;
      VstInt32 Channel = midiData[0] & 0x0f;
      MCHandle[Channel].Status = midiData[0] & 0xf0;
      if(MCHandle[Channel].Status == 0x90||MCHandle[Channel].Status == 0x80)
	{
	  MCHandle[Channel].Note = midiData[1] & 0x7f;
	  MCHandle[Channel].Velocity = midiData[2] & 0x7f;
	}
      // if(MCHandle[Channel].Status == 0xc0) MCHandle[Channel].Program = midiData[1] & 0x7f;
      if(MCHandle[Channel].Status == 0xb0) 
	{
	  if(midiData[1] == 0x00) MCHandle[Channel].MSB = midiData[2] & 0x7f;
	  if(midiData[1] == 0x20) MCHandle[Channel].LSB = midiData[2] & 0x7f;
	}
      
#ifdef DEBUG
      writeLogA("processEvents: MIDI %2x %2x %2x\n", midiData[0], midiData[1], midiData[2]);
#endif
      
      // [9n oo vv] NOTE ON
      //  vv=0x0 or 0x3      : none
      //  vv=0x1             : load (MSB/LSB/Note(oo)) to slot[channel(n)] (foreground)
      //  vv=0x2 or 0x4~0x7f : load (MSB/LSB/Note(oo)) to slot[channel(n)] (background)
      //  * If the specified (MSB/LSB/Note(oo)) is not in the registered XML file, the slot IR will be background unloaded.
      // [8n oo vv] NOTE OFF
      //  vv=0x0 or 0x4~0x7f : none
      //  vv=0x1             : unload (foreground)
      //  vv=0x2             : unload (background)
      // [bn 00 nn] set channel(n) MSB to nn
      // [bn 20 nn] set channel(n) LSB to nn
      
      int kmevent = KMEventNull;
      switch(MCHandle[Channel].Status)
	{
	case 0x90:
	  kmevent = KMEventLoadBG;
	  if(MCHandle[Channel].Velocity == 0x00) kmevent = KMEventNull;
	  if(MCHandle[Channel].Velocity == 0x01) kmevent = KMEventLoadFG;
	  if(MCHandle[Channel].Velocity == 0x03) kmevent = KMEventNull;
	  break;
	case 0x80:
	  kmevent = KMEventNull;
	  if(MCHandle[Channel].Velocity == 0x01) kmevent = KMEventUnloadFG;
	  if(MCHandle[Channel].Velocity == 0x02) kmevent = KMEventUnloadBG;
	  break;
	default:
	  kmevent = KMEventNull;
	  break;
	}

      std::string irFilename = irPrograms.getFilenameA(MCHandle[Channel].MSB, MCHandle[Channel].LSB, MCHandle[Channel].Note);
      if(irFilename == "") kmevent = KMEventUnloadBG;

      lfEventExecuterInfo lfeeinfo;
      lfeeinfo.kmevent = kmevent;
      lfeeinfo.channel = Channel;
      lfeeinfo.fv = this;
      lfeeinfo.fve = &editor;
      lfeeinfo.targetFilename = irFilename;
      lfeeinfo.ta = getNRTParameter(KNRTParam(Channel,KIRAttack));
      lfeeinfo.th = getNRTParameter(KNRTParam(Channel,KIRHold));
      lfeeinfo.td = getNRTParameter(KNRTParam(Channel,KIRDecay));
      lfeeinfo.ts = getNRTParameter(KNRTParam(Channel,KIRSustain));
      lfeeinfo.tr = getNRTParameter(KNRTParam(Channel,KIRRelease));
      lfeeinfo.errorMessage = false;
      
      if(kmevent == KMEventLoadBG||kmevent == KMEventUnloadBG)
	{
	  eventCueLocker.lock();
	  eventCue.push_back(lfeeinfo);
	  eventCueLocker.unlock();
	}
      else
	{
	  executeEvent(&lfeeinfo);
	}
      event ++;
    }
  return 1;
}

void Freeverb::setProgramName(char *name)
{
  strcpy(programName, name);
}

void Freeverb::getProgramName(char *name)
{
  strcpy(name, programName);
}


bool Freeverb::getProgramNameIndexed(VstInt32 category, VstInt32 index, char* text)
{
  if(index > 1) return false;
  strcpy(text, programName);
  return true;
}

void Freeverb::setParameter(VstInt32 index, float value)
{
  if(index >= KNumParams*MODEL_SLOT_SIZE)
    {
      int l = index-KNumParams*MODEL_SLOT_SIZE;
#ifdef DEBUG
	writeLogA("setParameter global KOA(%d) %.2f\n", index, value);
#endif
      switch(l)
	{
	case KOASkipLimiter:
	  if(value == 1.0f) skipLimiter = true;
	  else              skipLimiter = false;
	  break;
	case KOARelease:
	  limiter.setRelease(PARAM2RELEASE(value));
	  break;
	case KOACeiling:
	  limiter.setCeiling(PARAM2DBA(value));
	  if(limiter.getThreshold() > limiter.getCeiling())
	    limiter.setCeiling(limiter.getThreshold());
	  break;
	case KOAThreshold:
	  limiter.setThreshold(PARAM2DBA(value));
	  if(limiter.getThreshold() > limiter.getCeiling())
	    limiter.setThreshold(limiter.getCeiling());
	  break;
	case KOADry:
	  dryValue = value;
	  dryValueDB = PARAM2DB(value);
	  dryValueR = UTILS::dB2R(dryValueDB);
	  break;
	default:
	  break;
	}
      if(editor != NULL)
	((FreeverbEditor *)editor)->setParameter(index, value);
      return;
    }
  else
    {
      int slot = index/KNumParams;
      index = index%KNumParams;
#ifdef DEBUG      
      writeLogA("setParameter slot[%d] index[%d] %.2f\n", slot, index, value);
#endif
      switch (index)
	{
	case KMuteWet:
	  if(value == 1.0f)
	    { muteWet[slot][0] = true; }
	  else
	    { muteWet[slot][0] = false; value = 0; }
	  break;
	case KWet:
	  model[slot][0]->setwet(PARAM2DB(value));
	  break;
	case KWidth:
	  model[slot][0]->setwidth(value);
	  break;
	case KLPF:
	  model[slot][0]->setLPF(value);
	  break;
	case KHPF:
	  model[slot][0]->setHPF(value);
	  break;
	case KM2S:
	  if(value == 0.0f) m2s[slot] = false;
	  else              m2s[slot] = true;
	  break;
	case KSwap:
	  if(value == 0.0f) swap[slot] = false;
	  else              swap[slot] = true;
	  break;
	case KDelay:
	  conf_idelay[slot][0] = PARAM2DELAYTIME(value);
	  break;
	case KLRBalance:
	  model[slot][0]->setLRBalance(PARAM2LRB(value));
	  break;	  
	default:
	  break;
	}
      if(editor != NULL)
	((FreeverbEditor *)editor)->setParameter(KRTParam(slot,index), value);
      return;
    }
}


float Freeverb::getParameter(VstInt32 index)
{
  if(index >= KNumParams*MODEL_SLOT_SIZE)
    {
      float ret = 0.0f;
      int l = index-KNumParams*MODEL_SLOT_SIZE;
      switch(l)
	{
	case KOASkipLimiter:
	  if(skipLimiter == true) ret = 1.0f;
	  else                    ret = 0.0f;
	  break;
	case KOARelease:
	  ret = RELEASE2PARAM(limiter.getRelease());
	  break;
	case KOACeiling:
	  ret = DBA2PARAM(limiter.getCeiling());
	  break;
	case KOAThreshold:
	  ret = DBA2PARAM(limiter.getThreshold());
	  break;
	case KOADry:
	  ret = dryValue;
	  break;
	default:
	  break;
	}
      return ret;
    }

  int slot = index/KNumParams;
  index = index%KNumParams;
#ifdef DEBUG  
  writeLogA("getParameter slot[%d] index[%d]\n", slot, index);
#endif
  float ret = 0.0f;
  switch (index)
    {
    case KWidth:
      ret = model[slot][0]->getwidth();
      break;
    case KWet:
      ret = DB2PARAM(model[slot][0]->getwet());
      break;
    case KMuteWet:
      if(muteWet[slot][0] == true) ret = 1.0f;
      else ret = 0.0f;
      break;
    case KLPF:
      ret = model[slot][0]->getLPF();
      break;
    case KHPF:
      ret = model[slot][0]->getHPF();
      break;
    case KM2S:
      if(m2s[slot] == true) ret = 1.0f;
      else                  ret = 0.0f;
      break;
    case KSwap:
      if(swap[slot] == true) ret = 1.0f;
      else                   ret = 0.0f;
      break;
    case KDelay:
      ret = DELAYTIME2PARAM(conf_idelay[slot][0]);
      break;
    case KLRBalance:
      ret = LRB2PARAM(model[slot][0]->getLRBalance());
      break;
    default:
      break;
    }
  return ret;
}

void Freeverb::getParameterName(VstInt32 index, char *label)
{
  if(index >= KNumParams*MODEL_SLOT_SIZE)
    {
      int l = index-KNumParams*MODEL_SLOT_SIZE;
      if(l < KOAMaxParams)
	{ snprintf(label, kVstMaxParamStrLen, "%s", oaParameterName[l]); }
      else
	{ sprintf(label, "rsvd"); }
      return;
    }
  int slot = index/KNumParams;
  index = index%KNumParams;
  if(index < KMaxParams)
    { snprintf(label, kVstMaxParamStrLen, "%d-%s", slot, parameterName[index]); }
  else
    { snprintf(label, kVstMaxParamStrLen, "%d-rsvd", slot); }
}

void Freeverb::getParameterDisplay(VstInt32 index, char *text)
{
  if(index >= KNumParams*MODEL_SLOT_SIZE)
    {
      int l = index-KNumParams*MODEL_SLOT_SIZE;
      switch(l)
	{
	case KOASkipLimiter:
	  if(skipLimiter == true)
	    strcpy(text, "on");
	  else
	    strcpy(text, "off");
	  break;
	case KOARelease:
	  float2string(limiter.getRelease(), text, kVstMaxParamStrLen);
	  break;
	case KOACeiling:
	  float2string(limiter.getCeiling(), text, kVstMaxParamStrLen);
	  break;
	case KOAThreshold:
	  float2string(limiter.getThreshold(), text, kVstMaxParamStrLen);
	  break;
	case KOADry:
	  if(dryValue == 0)
	    strcpy(text, "-inf");
	  else
	    float2string(dryValueDB, text, kVstMaxParamStrLen);
	  break;
	default:
	  sprintf(text, "-");
	  break;
	}
      return;
    }
  int slot = index/KNumParams;
  index = index%KNumParams;
  switch (index)
    {
    case KWidth:
      float2string((float)(model[slot][0]->getwidth()*100), text, kVstMaxParamStrLen);
      break;
    case KWet:
      float2string(model[slot][0]->getwet(), text, kVstMaxParamStrLen);
      if(muteWet[slot][0] == true) strcpy(text, "-inf");
      break;
    case KMuteWet:
      if(muteWet[slot][0] == true) strcpy(text, "m on");
      else strcpy(text, "m off");
      break;
    case KLPF:
      float2string((float)(model[slot][0]->getLPF()*100), text, kVstMaxParamStrLen);
      break;
    case KHPF:
      float2string((float)(model[slot][0]->getHPF()*100), text, kVstMaxParamStrLen);
      break;
    case KM2S:
      if(m2s[slot] == true) strcpy(text, "on");
      else strcpy(text, "off");
      break;
    case KSwap:
      if(swap[slot] == true) strcpy(text, "on");
      else strcpy(text, "off");
      break;
    case KDelay:
      float2string(conf_idelay[slot][0], text, PARAMLENGTH);
      break;
    default:
      strcpy(text, "-");
      break;
    }
}

void Freeverb::getParameterLabel(VstInt32 index, char *label)
{
  if(index >= KNumParams*MODEL_SLOT_SIZE)
    {
      int l = index-KNumParams*MODEL_SLOT_SIZE;
      if(l < KOAMaxParams)
	snprintf(label, kVstMaxParamStrLen, "%s", oaParameterLabel[l]);
      else
	sprintf(label, "-");
      return;
    }
  
  //int slot = index/KNumParams;
  index = index%KNumParams;
  
  if(index < KMaxParams)
    snprintf(label, kVstMaxParamStrLen, "%s", parameterLabel[index]);
  else
    snprintf(label, kVstMaxParamStrLen, "-");
}

// Internal non-realtime parameters

float Freeverb::getNRTParameter(VstInt32 index)
{
  int slot = (index - KInternal)/KNumNRTParams;
  index = index - KNumNRTParams*slot;
  float ret = 0.0f;
  switch(index)
    {
    case KIRStretch:
      ret = conf_stretch[slot][0];
      break;
    case KIRAttack:
      ret = conf_attack[slot][0];
      break;
    case KIRHold:
      ret = conf_hold[slot][0];
      break;
    case KIRDecay:
      ret = conf_decay[slot][0];
      break;
    case KIRSustain:
      ret = conf_sustain[slot][0];
      break;
    case KIRRelease:
      ret = conf_release[slot][0];
      break;
    default:
      break;
    }
  return ret;
}

void Freeverb::setNRTParameter(VstInt32 index, float value)
{
  int slot = (index - KInternal)/KNumNRTParams;
  index = index - KNumNRTParams*slot;
  switch(index)
    {
    case KIRStretch:
      conf_stretch[slot][0] = value;
      break;
    case KIRAttack:
      conf_attack[slot][0] = value;
      break;
    case KIRHold:
      conf_hold[slot][0] = value;
      break;
    case KIRDecay:
      conf_decay[slot][0] = value;
      break;
    case KIRSustain:
      conf_sustain[slot][0] = value;
      break;
    case KIRRelease:
      conf_release[slot][0] = value;
      break;
    default:
      break;
    }
}

void Freeverb::suspend()
{
  writeLogA("Impulser2: suspend: ");
  globalLocker.lock();
  if(validModel != false)
    {
      validModel = false;
      for(int si = 0;si < MODEL_SLOT_SIZE;si ++)
	{
	  writeLogA("%d", si);
	  irSlotLocker[si][0].lock();
	  model[si][0]->suspend();
	  irSlotLocker[si][0].unlock();
	  writeLogA(".");
	}
    }
  globalLocker.unlock();
  writeLogA(" :-)\n");
}

void Freeverb::resume()
{
  writeLogA("Impulser2: inputLatency: %d outputLatency: %d\n", getInputLatency(), getOutputLatency());
  writeLogA("Impulser2: resume: ");
  globalLocker.lock();
  ProcessBlock::mute();
  limiter.mute();
  if(validModel != true)
    {
      validModel = true;
      for(int si = 0;si < MODEL_SLOT_SIZE;si ++)
	{
	  writeLogA("%d", si);
	  irSlotLocker[si][0].lock();
	  model[si][0]->resume();
	  model[si][0]->mute();
	  irSlotLocker[si][0].unlock();
	  writeLogA(".");
	}
    }
  
  /*
  writeLogA("Impulser2: resume: updateBlockSize=%d\n", updateBlockSize());
  if(updateBlockSize() > 0) allocProcessBlock(updateBlockSize());
  if(0 < updateBlockSize()&&currentBlockSize != updateBlockSize())
  {
  // reload with new blockSize was disabled.
  //setFragmentSize(UTILS::checkPow2(updateBlockSize()));
  //writeLogA("Impulser2: internal block: %dx%d\n", conf_fragmentSize, conf_factor);
  //for(int si = 0;si < MODEL_SLOT_SIZE;si ++)
  //{
  //loadToTmpSlotA(getNextFileNameA(si), (int)sampleRate,
  //PARAM2STRETCH(getNRTParameter(KNRTParam(si,KIRStretch))), 100);
  //writeLogA("Impulser2: continue:\n");
  //loadToModel(si);
  //}
  currentBlockSize = updateBlockSize();
  }
  //setInitialDelay(0);
  */
  DECLARE_VST_DEPRECATED(wantEvents)();
  globalLocker.unlock();
  writeLogA(" :-)\n");
}

float Freeverb::getSampleRate()
{
  return currentFs;
}

void Freeverb::setSampleRate(float sampleRate)
{
  writeLogA("Impulser2: setSampleRate: %.2f -> %.2f\n", currentFs, sampleRate);
  globalLocker.lock();
  if(currentFs != sampleRate)
    {
      limiter.setSampleRate(sampleRate);
      writeLogA("Impulser2: setSampleRate: Fs was changed, reloading models.\n");
      for(int si = 0;si < MODEL_SLOT_SIZE;si ++)
	{
	  CFILELOADER fileLoader;
	  fileLoader.setAHDSR(getNRTParameter(KNRTParam(si,KIRAttack)), getNRTParameter(KNRTParam(si,KIRHold)),
			      getNRTParameter(KNRTParam(si,KIRDecay)), getNRTParameter(KNRTParam(si,KIRSustain)),
			      getNRTParameter(KNRTParam(si,KIRRelease)));
	  int ret = fileLoader.load(getSlotFileNameA(si), (int)sampleRate,
				    PARAM2STRETCH(getNRTParameter(KNRTParam(si, KIRStretch))), 100, getConverterType());
	  if(ret == 0) loadToSlot(si, 0, fileLoader.out.getsize(), fileLoader.out.L, fileLoader.out.R);
	}
      currentFs = sampleRate;
    }
  globalLocker.unlock();
}

void Freeverb::process(float **inputs, float **outputs, VstInt32 sampleFrames)
{
  callProcess++;
  p_process(inputs, outputs, sampleFrames);
}

void Freeverb::processReplacing(float **inputs, float **outputs, VstInt32 sampleFrames)
{
  callReplace++;
  p_processReplacing(inputs, outputs, sampleFrames);
}

#ifdef PLUGDOUBLE
void Freeverb::processDoubleReplacing(double **inputs, double **outputs, VstInt32 sampleFrames)
{
  callDouble++;
  p_processDoubleReplacing(inputs, outputs, sampleFrames);
}
#endif

void Freeverb::processLRModel(pfloat_t *inL, pfloat_t *inR, pfloat_t *outL, pfloat_t *outR, VstInt32 sampleFrames)
{
  if(validModel == false) return;
#ifdef WIN32
  HANDLE trigger = OpenEventW(EVENT_ALL_ACCESS, FALSE, eventName);
  SetEvent(trigger);
  CloseHandle(trigger);
#endif

  // copy on byPass
  if(byPass == true)
    {
      ProcessBlock::processLRModel(inL,inR,outL,outR,sampleFrames);
      return;
    }

  // add dry
  if(dryValue > 0.0)
    {
      for(VstInt32 i = 0;i < sampleFrames;i ++)
	{
	  outL[i] = inL[i] * dryValueR;
	  outR[i] = inR[i] * dryValueR;
	}
    }
  else
    {
      UTILS::mute(outL, sampleFrames);
      UTILS::mute(outR, sampleFrames);
    }
  
  for(int si = 0;si < MODEL_SLOT_SIZE;si ++)
    {
      // add to outputL/R (skip init)
      if(model[si][0]->getSampleSize() == 0||muteWet[si][0] == true)
	continue;
      unsigned slotoption = FV3_IR_SKIP_INIT|FV3_IR_MUTE_DRY;
      if(getParameter(KRTParam(si,KWet)) == 0.0f)
	slotoption |= FV3_IR_MUTE_WET;
      if(model[si][0]->getLPF() == 0.0f&&model[si][0]->getHPF() == 0.0f)
	slotoption |= FV3_IR_SKIP_FILTER;
      if(m2s[si] == true)
	slotoption |= FV3_IR_MONO2STEREO;
      if(swap[si] == true)
	slotoption |= FV3_IR_SWAP_LR;
      if(irSlotLocker[si][0].tryLock() == true)
	{
	  int iDelay = (int)((float)currentFs*conf_idelay[si][0]/1000.0f);
	  if(model[si][0]->getInitialDelay() != iDelay) model[si][0]->setInitialDelay(iDelay);
	  model[si][0]->processreplace(inL, inR, outL, outR, (long int)sampleFrames, slotoption);
	  irSlotLocker[si][0].unlock();
	}
    } // slotloop
  if(skipLimiter == false)
    limiter.processreplace(outL, outR, outL, outR, (long int)sampleFrames);
  // VuMeter
  if(editor != NULL)
    {
      float sL = 0.0f, sR = 0.0f;
      for(int i = 0;i < sampleFrames;i ++)
	{
	  float al = fabs((float)outL[i]);
	  float ar = fabs((float)outR[i]);
	  if(sL < al) sL = al;
	  if(sR < ar) sR = ar;      
	}
      ((FreeverbEditor*)editor)->setVuMeter(sL, sR);
    }
}

void Freeverb::setSlotFileNameA(int slot, const char * path)
{
  if(slot < 0||slot >= MODEL_SLOT_SIZE) return;
  globalLocker.lock();
  slotFileNameA[slot] = path;
  globalLocker.unlock();
}

const char * Freeverb::getSlotFileNameA(int slot)

{
  if(slot < 0||slot >= MODEL_SLOT_SIZE) return NULL;
  return slotFileNameA[slot].c_str();
}

VstInt32 Freeverb::getChunk(void ** data, bool isPreset)
{
  if(isPreset != 0) return getChunk(data, 0);
  globalLocker.lock();
  int ret = 0, size = 0;
  writeLogA("Impulser2: getChunk: isPreset=%d\n", isPreset);
  // mode isPreset=0 ... store/save when the host exit/start
  chunk.clear();
  chunk.setVersion(FV3_IR2_CHUNK_VERSION_CURRENT);
  
  // Chunk structure
  // [KChunkNumMax] slot 0
  // [KChunkNumMax] slot 9
  // +KChunkIParamsH +[KNumParams] slot 0 automation parameters
  // +KChunkIParamsH +[KNumParams] slot 9 automation parameters
  // +KChunkIParamsH +[KOANumParams] overall automation parameters
  
  // GUI only Parameters
  for(int si = 0;si < MODEL_SLOT_SIZE;si ++)
    {
      writeLogA("Impulser2: getChunk: [%02d] \"%s\"\n", si, getSlotFileNameA(si));
      chunk.setSlotString(si*KChunkNumMax + KChunkFileName, getSlotFileNameA(si));
      chunk.setSlotFloat (si*KChunkNumMax + KChunkStretch,  getNRTParameter(KNRTParam(si,KIRStretch)));
      chunk.setSlotFloat (si*KChunkNumMax + KChunkAttack,   getNRTParameter(KNRTParam(si,KIRAttack)));
      chunk.setSlotFloat (si*KChunkNumMax + KChunkHold,     getNRTParameter(KNRTParam(si,KIRHold)));
      chunk.setSlotFloat (si*KChunkNumMax + KChunkDecay,    getNRTParameter(KNRTParam(si,KIRDecay)));
      chunk.setSlotFloat (si*KChunkNumMax + KChunkSustain,  getNRTParameter(KNRTParam(si,KIRSustain)));
      chunk.setSlotFloat (si*KChunkNumMax + KChunkRelease,  getNRTParameter(KNRTParam(si,KIRRelease)));
    }
  // automation parameters
  for(VstInt32 pi = 0;pi < KNumParams*MODEL_SLOT_SIZE+KOANumParams;pi ++)
    {
      char parameterName[kVstMaxParamStrLen];
      getParameterName(pi, parameterName);
#ifdef DEBUG
      writeLogA("Impulser2: getChunk: automation parameter %d(%s) = %.2f\n", pi, parameterName, getParameter(pi));
#endif
      chunk.setSlotFloat(KChunkIParamsH + pi, getParameter(pi));
    }
  ret = chunk.getChunk(&size, data);
  if(ret != 0)
    {
      writeLogA("Impulser2: getChunk: getChunk() failed! %d\n", ret);
      size = 0;
    }
  writeLogA("Impulser2: getChunk: size = %d\n", size);
#ifdef DEBUG
  writeDumpA("Impulser2: getChunk: DUMP\n", (const unsigned char*)*data, size);
#endif
  globalLocker.unlock();
  writeLogA("Impulser2: getChunk: done.\n");
  return size;
}

VstInt32 Freeverb::setChunk(void * data, VstInt32 byteSize, bool isPreset)
{
  if(isPreset != 0) return setChunk(data, byteSize, 0);
  globalLocker.lock();
  int ret = 0;
  writeLogA("Impulser2: setChunk: size=%d isPreset=%d\n", byteSize, isPreset);
#ifdef DEBUG
  writeDumpA("Impulser2: setChunk: DUMP\n", (const unsigned char*)data, byteSize);
#endif

  // 1. init/chunk load
  chunk.clear();
  ret = chunk.registerChunk(byteSize, data);
  if(ret != 0)
    {
      writeLogA("Impulser2: setChunk: registerChunk() failed! %d\n", ret);
      globalLocker.unlock();
      return 1;
    }
  
  // update chunk data
  if(chunk.getVersion() > FV3_IR2_CHUNK_VERSION_CURRENT)
    {
      writeLogA("Impulser2: setChunk: Chunk is newer than this plugin. Some data will be ignored or lost. (%d > %d)\n",
		chunk.getVersion(), FV3_IR2_CHUNK_VERSION_CURRENT);
    }
  if(chunk.getVersion() < FV3_IR2_CHUNK_VERSION_CURRENT)
    {
      writeLogA("Impulser2: setChunk: Chunk is old (%d < %d), updating...\n", chunk.getVersion(), FV3_IR2_CHUNK_VERSION_CURRENT);
      if(chunk.getVersion() <= FV3_IR2_CHUNK_VERSION_OLD_1)
	{
	  for(int si = 0;si < MODEL_SLOT_SIZE;si ++)
	    {
	      // Set new automation parameters (LRBalance)
	      chunk.setSlotFloat(KChunkIParamsH + KNumParams*si + KLRBalance,  0.5f);
	      chunk.setSlotFloat(KChunkIParamsH + KNumParams*si + KLRBalance2, 0.5f);

	      // Convert IR Stretch internal parameter (map old chunk num to new num)
	      int stretch = 50;
	      ret = chunk.getSlotInt(KChunkStretch+si*KChunkOldNum, &stretch);
	      if(ret != 0)
		{
		  float retf = PARAM2STRETCH_OLD_1((float)stretch);
		  retf = STRETCH2PARAM(retf);
		  chunk.setSlotFloat(KChunkNumMax*si + KChunkStretch, retf);
		}
	      else
		chunk.setSlotFloat(KChunkNumMax*si + KChunkStretch, 0.5f);
	      
	      // map old slotstring to new slotstring
	      char * cfilename = NULL;
	      ret = chunk.getSlotString(KChunkOldNum*si+KChunkFileName, &cfilename);
	      if(ret == 0)
		{
		  writeLogA("Impulser2: setChunk: OldSlot[%02d]\"%s\"\n", si, cfilename);
		}
	      if(ret == 0&&(KChunkOldNum*si+KChunkFileName) != (KChunkNumMax*si+KChunkFileName))
		{
		  writeLogA("Impulser2: setChunk: MapOldSlot[%02d](%03d->%03d)\n",
			    si, KChunkOldNum*si+KChunkFileName, KChunkNumMax*si+KChunkFileName);
		  ret = chunk.setSlotString(KChunkNumMax*si+KChunkFileName, cfilename);
		}
	      
	      // Set new IR AHDSR internal parameters
	      chunk.setSlotFloat(KChunkNumMax*si + KChunkAttack,  0.0f); // Attack/(Total-Release)
	      chunk.setSlotFloat(KChunkNumMax*si + KChunkHold,    1.0f); // Hold/(Total-Attack-Release-Decay)
	      chunk.setSlotFloat(KChunkNumMax*si + KChunkDecay,   0.0f); // Decay/(Total-Attack-Release)
	      chunk.setSlotFloat(KChunkNumMax*si + KChunkSustain, 0.8f); // Sustain level / Hold level
	      chunk.setSlotFloat(KChunkNumMax*si + KChunkRelease, 0.0f); // Release/Total
	    }
	}
      writeLogA("Impulser2: setChunk: Chunk was updated. (%d -> %d)\n", chunk.getVersion(), FV3_IR2_CHUNK_VERSION_CURRENT);
      chunk.setVersion(FV3_IR2_CHUNK_VERSION_CURRENT);
    }
  // 2 .GUI only Parameters including IR files
  for(int si = 0;si < MODEL_SLOT_SIZE;si ++)
    {
      char * cfilename = NULL;
      float stretch = 0, val = 0;
      CFILELOADER fileLoader;
      ret = chunk.getSlotString(KChunkNumMax*si + KChunkFileName, &cfilename);
      if(ret != 0)
	{
	  writeLogA("Impulser2: setChunk: getSlotString(%d=filename) failed! (%d)\n", KChunkFileName+KChunkNumMax*si, ret);
	  continue;
	}
      ret = chunk.getSlotFloat(KChunkNumMax*si + KChunkStretch, &stretch);
      if(ret != 0||stretch < 0.0||stretch > 1.0)
	{
	  writeLogA("Impulser2: setChunk: Stretch -> default to 0.5\n");
	  stretch = 0.5;
	}
      setNRTParameter(KNRTParam(si,KIRStretch), stretch);
      
      ret = chunk.getSlotFloat(KChunkNumMax*si + KChunkAttack, &val);
      setNRTParameter(KNRTParam(si,KIRAttack), val);
      ret = chunk.getSlotFloat(KChunkNumMax*si + KChunkHold, &val);
      setNRTParameter(KNRTParam(si,KIRHold), val);
      ret = chunk.getSlotFloat(KChunkNumMax*si + KChunkDecay, &val);
      setNRTParameter(KNRTParam(si,KIRDecay), val);
      ret = chunk.getSlotFloat(KChunkNumMax*si + KChunkSustain, &val);
      setNRTParameter(KNRTParam(si,KIRSustain), val);
      ret = chunk.getSlotFloat(KChunkNumMax*si + KChunkRelease, &val);
      setNRTParameter(KNRTParam(si,KIRRelease), val);

      writeLogA("Impulser2: setChunk: [%02d] \"%s\"\n", si, cfilename);
      
      fileLoader.setAHDSR(getNRTParameter(KNRTParam(si,KIRAttack)), getNRTParameter(KNRTParam(si,KIRHold)),
			  getNRTParameter(KNRTParam(si,KIRDecay)), getNRTParameter(KNRTParam(si,KIRSustain)),
			  getNRTParameter(KNRTParam(si,KIRRelease)));
      ret = fileLoader.load(cfilename, (int)getSampleRate(), PARAM2STRETCH(stretch), 100, getConverterType());
      if(ret == 0&&cfilename != NULL)
	{
	  setSlotFileNameA(si, cfilename);
	  loadToSlot(si, 0, fileLoader.out.getsize(), fileLoader.out.L, fileLoader.out.R);
	}
      else
	{
	  setSlotFileNameA(si, "");
	  unloadSlot(si, 0);
	}
    }
  // 3. automation parameters
  for(VstInt32 pi = 0;pi < KNumParams*MODEL_SLOT_SIZE+KOANumParams;pi ++)
    {
      float value;
      ret = chunk.getSlotFloat(KChunkIParamsH + pi, &value);
      if(ret == 0)
	{
	  char parameterName[kVstMaxParamStrLen];
	  getParameterName(pi, parameterName);
#ifdef DEBUG
	  writeLogA("Impulser2: setChunk: set automation parameter %d(%s) = %.2f\n",
		    pi, parameterName, value);
#endif
	  // CEI(ceiling)/THR(threshold) parameter load exception
	  if(pi == KATParam(KOACeiling))
	    {
	      if(value <= getParameter(KATParam(KOAThreshold)))
		setParameter(KATParam(KOAThreshold), value);
	      setParameter(KATParam(KOACeiling), value);
	    }
	  else if(pi == KATParam(KOAThreshold))
	    {
	      if(value >= getParameter(KATParam(KOACeiling)))
		setParameter(KATParam(KOACeiling), value);
	      setParameter(KATParam(KOAThreshold), value);
	    }
	  else
	    setParameter(pi, value);
	}
      else
	{
	  writeLogA("Impulser2: setChunk: chunk Error %d\n", KChunkIParamsH + pi);
	}
    }
  // update all GUI parameters
  for(int si = 0;si < MODEL_SLOT_SIZE;si ++)
    {
      if(editor != NULL) ((FreeverbEditor *)editor)->setViewSlot(si);
    }
  // init view to slot 0
  if(editor != NULL) ((FreeverbEditor *)editor)->setViewSlot(getCurrentSlot());
  globalLocker.unlock();
  writeLogA("Impulser2: setChunk: done.\n");
  return 0;
}


void Freeverb::pushLoadCueA(int slot, const char * filename)
{
  if(filename == NULL) return;
  lfEventExecuterInfo lfeeinfo;
  lfeeinfo.kmevent = KMEventLoadBG;
  lfeeinfo.channel = slot;
  lfeeinfo.fv = this;
  lfeeinfo.fve = &editor;
  lfeeinfo.targetFilename = filename;
  lfeeinfo.ta = getNRTParameter(KNRTParam(slot,KIRAttack));
  lfeeinfo.th = getNRTParameter(KNRTParam(slot,KIRHold));
  lfeeinfo.td = getNRTParameter(KNRTParam(slot,KIRDecay));
  lfeeinfo.ts = getNRTParameter(KNRTParam(slot,KIRSustain));
  lfeeinfo.tr = getNRTParameter(KNRTParam(slot,KIRRelease));
  lfeeinfo.errorMessage = true;
  eventCueLocker.lock();
  eventCue.push_back(lfeeinfo);
  eventCueLocker.unlock();
#ifdef WIN32
  HANDLE trigger = OpenEventW(EVENT_ALL_ACCESS, FALSE, eventName);
  SetEvent(trigger);
  CloseHandle(trigger);
#else
  event.trigger();
#endif
}

void Freeverb::pushReloadCue(int slot)
{
  std::string slotFileName = getSlotFileNameA(slot);
  setSlotFileNameA(slot, "");
  unloadSlot(slot, 0);
  pushLoadCueA(slot, slotFileName.c_str());
}

void Freeverb::unloadSlot(int slot, int rear)
{
  if(slot >= MODEL_SLOT_SIZE||rear >= 2||slot < 0||rear < 0) return;
  IR2 * targetSlot = model[slot][rear];
  irSlotLocker[slot][rear].lock();
  targetSlot->unloadImpulse();
  irSlotLocker[slot][rear].unlock();
}

void Freeverb::loadToSlot(int slot, int rear, long size, const pfloat_t * L, const pfloat_t * R)
{
  if(slot >= MODEL_SLOT_SIZE||rear >= 2||slot < 0||rear < 0||size < 0) return;
  IR2 * targetSlot = model[slot][rear];
  writeLogA("Impulser2: loadToSlot: [%d][%d](%ld)\n", slot, rear, size);
  irSlotLocker[slot][rear].lock();
  try
    {
      targetSlot->unloadImpulse();
      targetSlot->setFragmentSize(conf_fragmentSize, conf_factor);
      targetSlot->loadImpulse(L, R, size);
      writeLogA("Impulser2: loadToModel: %ld -> %ld / %ld x %ld / %ld x %ld\n",
		size, targetSlot->getSampleSize(),
		targetSlot->getSFragmentSize(), targetSlot->getSFragmentCount(),
		targetSlot->getLFragmentSize(), targetSlot->getLFragmentCount());
    }
  catch(std::bad_alloc)
    {
      writeLogA("Impulser2: loadToModel: std::bad_alloc!\n");
#ifdef WINDOWS
      MessageBoxA(NULL, "The file is too large to load.", "Freeverb3 Impulser2: Load failed.", MB_OK);
#endif
#ifdef MAC
      MacMessageBox("The file is too large to load.", "Freeverb3 Impulser2: Load failed.");
#endif
    }
  targetSlot->mute();
  irSlotLocker[slot][rear].unlock();
  writeLogA("Impulser2: loadToModel: success loading the impulse.\n");
}

void Freeverb::setIRProgsFileName(char * filename)
{
  if(filename != NULL)
    irProgramsFilename = filename;
  else
    irProgramsFilename = "";
}

int Freeverb::registerIRPrograms()
{
  return irPrograms.registerFromXML(irProgramsFilename.c_str());
}

void Freeverb::setConverterType(int type)
{
  converter_type = type;
}

int Freeverb::getConverterType()
{
  return converter_type;
}

void Freeverb::setGUIZoomFactor(float factor)
{
  if(factor < 0) return;
  if(editor != NULL)
    ((FreeverbEditor*)editor)->setGUIZoomFactor(factor);
}

bool Freeverb::setBypass(bool onOff)
{
  byPass = onOff;
  return onOff;
}

int Freeverb::getCurrentSlot()
{
  return currentSlot;
}

void Freeverb::setCurrentSlot(int slot)
{
  if(slot > MODEL_SLOT_SIZE)
    currentSlot = 0;
  else
    currentSlot = slot;
}
