/* 
 * FreeModbus Libary: A portable Modbus implementation for Modbus ASCII/RTU.
 * Copyright (c) 2006 Christian Walter <wolti@sil.at>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * File: $Id: mb.c,v 1.28 2010/06/06 13:54:40 wolti Exp $
 */

/* ----------------------- System includes ----------------------------------*/
#include "stdlib.h"
#include "string.h"
#include <stddef.h>

/* ----------------------- Platform includes --------------------------------*/
#include "port.h"

/* ----------------------- Modbus includes ----------------------------------*/
#include <mbstruct.h>
#include <mbconfig.h>
#include <../mbmulty_wrap.h>
#include "mbframe.h"
#include "mbproto.h"
#include "mbfunc.h"

#include "mbport.h"
#if MB_RTU_ENABLED == 1
#include "mbrtu.h"
#endif
#if MB_ASCII_ENABLED == 1
#include "mbascii.h"
#endif
#if MB_TCP_ENABLED == 1
#include "mbtcp.h"
#endif

#ifndef MB_PORT_HAS_CLOSE
#define MB_PORT_HAS_CLOSE 0
#endif

/* ----------------------- Static variables ---------------------------------*/

#ifndef MB_IS_MULTY
static UCHAR    ucMBAddress;
eMBMode  eMBCurrentMode;
MBState_t eMBState = STATE_NOT_INITIALIZED;
MBStatus eMBStatus={{0}};// = {STATE_NOT_INITIALIZED};

/* Functions pointer which are initialized in eMBInit( ). Depending on the
 * mode (RTU or ASCII) the are set to the correct implementations.
 */

#ifdef MB_USE_FPTRCUR

static peMBFrameSend peMBFramePostCur;

static peMBFrameSend peMBFrameSendCur;
static pvMBFrameStart pvMBFrameStartCur;
static pvMBFrameStop pvMBFrameStopCur;
static peMBFrameReceive peMBFrameReceiveCur;
static pvMBFrameClose pvMBFrameCloseCur;

/* Callback functions required by the porting layer. They are called when
 * an external event has happend which includes a timeout or the reception
 * or transmission of a character.
 */
BOOL( *pxMBFrameCBByteReceived ) ( void );
BOOL( *pxMBFrameCBTransmitterEmpty ) ( void );
BOOL( *pxMBPortCBTimerExpired ) ( void );

BOOL( *pxMBFrameCBReceiveFSMCur ) ( void );
BOOL( *pxMBFrameCBTransmitFSMCur ) ( void );

#endif

#else   //!MB_IS_MULTY

void  MB_init_multy(MB_PORT);

#endif  //MB_IS_MULTY

/* An array of Modbus functions handlers which associates Modbus function
 * codes with implementing functions.
 */
#ifndef MB_IS_MULTY
static xMBFunctionHandler xFuncHandlers[MB_FUNC_HANDLERS_MAX]
#else
static const xMBFunctionHandler FuncHandlersDefaults[]
#endif
= {
#if MB_FUNC_OTHER_REP_SLAVEID_ENABLED > 0
    {MB_FUNC_OTHER_REPORT_SLAVEID, MB_METHOD_NAME(eMBFuncReportSlaveID)},
#endif
#if MB_FUNC_READ_INPUT_ENABLED > 0
    {MB_FUNC_READ_INPUT_REGISTER, MB_METHOD_NAME(eMBFuncReadInputRegister)},
#endif
#if MB_FUNC_READ_HOLDING_ENABLED > 0
    {MB_FUNC_READ_HOLDING_REGISTER, MB_METHOD_NAME(eMBFuncReadHoldingRegister)},
#endif
#if MB_FUNC_WRITE_MULTIPLE_HOLDING_ENABLED > 0
    {MB_FUNC_WRITE_MULTIPLE_REGISTERS, MB_METHOD_NAME(eMBFuncWriteMultipleHoldingRegister)},
#endif
#if MB_FUNC_WRITE_HOLDING_ENABLED > 0
    {MB_FUNC_WRITE_REGISTER, MB_METHOD_NAME(eMBFuncWriteHoldingRegister)},
#endif
#if MB_FUNC_READWRITE_HOLDING_ENABLED > 0
    {MB_FUNC_READWRITE_MULTIPLE_REGISTERS, MB_METHOD_NAME(eMBFuncReadWriteMultipleHoldingRegister)},
#endif
#if MB_FUNC_READ_COILS_ENABLED > 0
    {MB_FUNC_READ_COILS, MB_METHOD_NAME(eMBFuncReadCoils)},
#endif
#if MB_FUNC_WRITE_COIL_ENABLED > 0
    {MB_FUNC_WRITE_SINGLE_COIL, MB_METHOD_NAME(eMBFuncWriteCoil)},
#endif
#if MB_FUNC_WRITE_MULTIPLE_COILS_ENABLED > 0
    {MB_FUNC_WRITE_MULTIPLE_COILS, MB_METHOD_NAME(eMBFuncWriteMultipleCoils)},
#endif
#if MB_FUNC_READ_DISCRETE_INPUTS_ENABLED > 0
    {MB_FUNC_READ_DISCRETE_INPUTS, MB_METHOD_NAME(eMBFuncReadDiscreteInputs)},
#endif
#ifdef MB_IS_MULTY
    {0, NULL}
#endif
};


#ifndef MB_PORT_HAS_HANDLERS
#define MBSlaveHandler(...) MB_METHOD_ACT(MBSlaveHandler, __VA_ARGS__)
pxMBFunctionHandler MB_METHOD_DECL(MBSlaveHandler, UCHAR ucFunctionCode){
            UCHAR i;
            for( i = 0; i < MB_FUNC_HANDLERS_MAX; i++ )
            {
                /* No more function handlers registered. Abort. */
                if( xFuncHandlers[i].ucFunctionCode == 0 )
                {
                    return(NULL);
                }
                else if( xFuncHandlers[i].ucFunctionCode == ucFunctionCode )
                {
                    return(xFuncHandlers[i].pxHandler);
                }
            }
            return(NULL);
}
#endif

/* ----------------------- Start implementation -----------------------------*/

#ifdef MB_IS_MULTY

void  MB_reset(MB_PORT){
    memset(MB_SELF, 0, sizeof(*MB_SELF));

    //memset(xFuncHandlers, 0, sizeof(xFuncHandlers));
    int i;
    xMBFunctionHandler* dst         = xFuncHandlers;
    const xMBFunctionHandler* src   = FuncHandlersDefaults;
    for( i = 0; i < MB_FUNC_HANDLERS_MAX; i++, dst++, src++)
    {
        /* No more function handlers registered. Abort. */
        if( src->ucFunctionCode == 0 )
            break;
        dst->pxHandler      = src->pxHandler;
        dst->ucFunctionCode = src->ucFunctionCode;
    }

    eMBState = STATE_NOT_INITIALIZED;
}

#else
#define MB_init()
#endif

eMBErrorCode
MB_METHOD_DECL(eMBInit, eMBMode eMode, UCHAR ucSlaveAddress, UCHAR ucPort, ULONG ulBaudRate, eMBParity eParity )
{
    eMBErrorCode    eStatus = MB_ENOERR;

    MB_reset(MB_SELF);

    /* check preconditions */
    if( ( ucSlaveAddress == MB_ADDRESS_BROADCAST ) ||
        ( ucSlaveAddress < MB_ADDRESS_MIN ) || ( ucSlaveAddress > MB_ADDRESS_MAX ) )
    {
        eStatus = MB_EINVAL;
    }
    else
    {
        ucMBAddress = ucSlaveAddress;

        switch ( eMode )
        {
#if MB_RTU_ENABLED > 0
        case MB_RTU:
#ifdef MB_USE_FPTRCUR
            ref_MBFrameStartCur = &(MB_METHOD_NAME(eMBRTUStart));
            ref_MBFrameStopCur  = &(MB_METHOD_NAME(eMBRTUStop));
            ref_MBFrameSendCur  = &(MB_METHOD_NAME(eMBRTUSend));
            ref_MBFramePostCur  = &(MB_METHOD_NAME(eMBRTUPost));
            ref_MBFrameReceiveCur= &(MB_METHOD_NAME(eMBRTUReceive));
            ref_MBFrameCloseCur = (MB_PORT_HAS_CLOSE)? MB_METHOD_NAME(vMBPortClose) : NULL;
            ref_MBFrameCBByteReceived       = MB_METHOD_NAME(xMBRTUReceiveFSM);
            ref_MBFrameCBTransmitterEmpty   = MB_METHOD_NAME(xMBRTUTransmitFSM);
            ref_MBPortCBTimerExpired        = MB_METHOD_NAME(xMBRTUTimerT35Expired);
#endif
            eStatus = eMBRTUInit(ucMBAddress, ucPort, ulBaudRate, eParity );
            break;
#endif
#if MB_ASCII_ENABLED > 0
        case MB_ASCII:
#ifdef MB_USE_FPTRCUR
            pvMBFrameStartCur = eMBASCIIStart;
            pvMBFrameStopCur = eMBASCIIStop;
            peMBFrameSendCur = eMBASCIISend;
            peMBFramePostCur = eMBASCIISend;
            peMBFrameReceiveCur = eMBASCIIReceive;
            pvMBFrameCloseCur = MB_PORT_HAS_CLOSE ? vMBPortClose : NULL;
            pxMBFrameCBByteReceived = xMBASCIIReceiveFSM;
            pxMBFrameCBTransmitterEmpty = xMBASCIITransmitFSM;
            pxMBPortCBTimerExpired = xMBASCIITimerT1SExpired;
#endif
            eStatus = eMBASCIIInit(ucMBAddress, ucPort, ulBaudRate, eParity );
            break;
#endif
        default:
            eStatus = MB_EINVAL;
        }

        if( eStatus == MB_ENOERR )
        {
            if( !xMBPortEventInit(  ) )
            {
                /* port dependent event module initalization failed. */
                eStatus = MB_EPORTERR;
            }
            else
            {
                eMBCurrentMode = eMode;
                eMBState = STATE_DISABLED;
            }
        }
    }
    return eStatus;
}

#if MB_TCP_ENABLED > 0
eMBErrorCode
MB_METHOD_DECL(eMBTCPInit, USHORT ucTCPPort )
{
    eMBErrorCode    eStatus = MB_ENOERR;

    MB_reset(MB_SELF);

    if( ( eStatus = eMBTCPDoInit( ucTCPPort ) ) != MB_ENOERR )
    {
        eMBState = STATE_DISABLED;
    }
    else if( !xMBPortEventInit(  ) )
    {
        /* Port dependent event module initalization failed. */
        eStatus = MB_EPORTERR;
    }
    else
    {
#ifdef MB_USE_FPTRCUR
        ref_MBFrameStartCur     = MB_METHOD_NAME(eMBTCPStart);
        ref_MBFrameStopCur      = MB_METHOD_NAME(eMBTCPStop);
        ref_MBFrameReceiveCur   = MB_METHOD_NAME(eMBTCPReceive);
        ref_MBFrameSendCur      = MB_METHOD_NAME(eMBTCPSend);
        ref_MBFramePostCur      = MB_METHOD_NAME(eMBTCPSend);
        ref_MBFrameCloseCur         = MB_PORT_HAS_CLOSE ? MB_METHOD_NAME(vMBTCPPortClose) : NULL;
#endif
        ucMBAddress = MB_TCP_PSEUDO_ADDRESS;
        eMBCurrentMode = MB_TCP;
        eMBState = STATE_DISABLED;
    }
    return eStatus;
}
#endif

eMBErrorCode
MB_METHOD_DECL(eMBRegisterCB, UCHAR ucFunctionCode, pxMBFunctionHandler pxHandler )
{
    int             i;
    eMBErrorCode    eStatus;

    if( ( 0 < ucFunctionCode ) && ( ucFunctionCode <= 127 ) )
    {
        xMBFunctionHandler* handle = xFuncHandlers;
        eStatus = MB_ENORES;
        for( i = 0; i < MB_FUNC_HANDLERS_MAX; i++, handle++ )
        {
            /* No more function handlers registered. Place New one. */
            if( handle->ucFunctionCode == 0 )
            {
                if( pxHandler != NULL )
                {
                    volatile xMBFunctionHandler* tmp = handle; 
                    tmp->pxHandler = pxHandler;
                    tmp->ucFunctionCode = ucFunctionCode;
                }
                eStatus = MB_ENOERR;
                break;
            }
            else if( handle->ucFunctionCode == ucFunctionCode )
            {
#           ifndef MB_CAN_UNREGISTER_CB
                handle->pxHandler = pxHandler;
#           else
                if( pxHandler != NULL ){
                    handle->pxHandler = pxHandler;
                }
                else if ( i < (MB_FUNC_HANDLERS_MAX-1)){
                    ENTER_CRITICAL_SECTION(  );
                    //memcpy(handle, (handle+1), sizeof(xMBFunctionHandler)*((MB_FUNC_HANDLERS_MAX-1) - i));
                    for (int j = i; j < (MB_FUNC_HANDLERS_MAX-1); j++, handle++){
                        handle[0].ucFunctionCode= handle[1].ucFunctionCode;
                        handle[0].pxHandler     = handle[1].pxHandler;
                        if (handle[1].ucFunctionCode == 0) break;
                    }
                    handle[1].ucFunctionCode = 0;
                    EXIT_CRITICAL_SECTION(  );
                }
#           endif //MB_CAN_UNREGISTER_CB
                eStatus = MB_ENOERR;
                break;
            }
        }
    }
    else
    {
        eStatus = MB_EINVAL;
    }
    return eStatus;
}


eMBErrorCode
MB_METHOD_DECL1(eMBClose)
{
    eMBErrorCode    eStatus = MB_ENOERR;

    if( eMBState == STATE_DISABLED )
    {
#if MB_PORT_HAS_CLOSE != 0
        if( ref_MBFrameCloseCur != NULL )
        {
            pvMBFrameCloseCur(  );
        }
#endif
    }
    else
    {
        eStatus = MB_EILLSTATE;
    }
    return eStatus;
}

eMBErrorCode
MB_METHOD_DECL1(eMBEnable)
{
    eMBErrorCode    eStatus = MB_ENOERR;

    if( eMBState == STATE_DISABLED )
    {
        /* Activate the protocol stack. */
        pvMBFrameStartCur(  );
        eMBState = STATE_ENABLED;
    }
    else
    {
        eStatus = MB_EILLSTATE;
    }
    return eStatus;
}

eMBErrorCode
MB_METHOD_DECL1(eMBDisable)
{
    eMBErrorCode    eStatus;

    if( eMBState == STATE_ENABLED )
    {
        pvMBFrameStopCur(  );
        eMBState = STATE_DISABLED;
        eStatus = MB_ENOERR;
    }
    else if( eMBState == STATE_DISABLED )
    {
        eStatus = MB_ENOERR;
    }
    else
    {
        eStatus = MB_EILLSTATE;
    }
    return eStatus;
}

eMBErrorCode
MB_METHOD_DECL1(eMBPoll)
{
    MB_LOCAL_SPEC UCHAR   *ucMBFrame;
    MB_LOCAL_SPEC UCHAR    ucRcvAddress;
    MB_LOCAL_SPEC UCHAR    ucFunctionCode;
    MB_LOCAL_SPEC UFRAMESIZE   usLength;
    MB_LOCAL_SPEC eMBException eException;

    eMBErrorCode    eStatus = MB_ENOERR;
    eMBEventType    eEvent;

    /* Check if the protocol stack is ready. */
    if( eMBState <= STATE_DISABLED )
    {
        return MB_EILLSTATE;
    }

    /* Check if there is a event available. If not return control to caller.
     * Otherwise we will handle the event. */
    if( xMBPortEventGet( &eEvent ) == TRUE )
    {
        switch ( eEvent )
        {
        case EV_READY:
            if (eMBState == STATE_TERMINATE){
                eMBState = STATE_CLOSED;
                return MB_CLOSED;
            }
            break;

        case EV_FRAME_RECEIVED:
            eStatus = peMBFrameReceiveCur( &ucRcvAddress, &ucMBFrame, &usLength );
            if( eStatus == MB_ENOERR )
            {
                /* Check if the frame is for us. If not ignore the frame. */
                if( ( ucRcvAddress == ucMBAddress ) || ( ucRcvAddress == MB_ADDRESS_BROADCAST ) )
#if (MB_POLL_EXECUTES <= 0)
                {
                    ( void )xMBPortEventPost( EV_EXECUTE );
                }
                break;
#else
                ;//just direct to EV_EXECUTE
                else break;
#endif
            }
            else
            break;

        case EV_EXECUTE:
          {
            ucFunctionCode = ucMBFrame[MB_PDU_FUNC_OFF];
            eException = MB_EX_ILLEGAL_FUNCTION;
            pxMBFunctionHandler Handle = MBSlaveHandler(ucFunctionCode);
            if (Handle != NULL) {
                    eException = Handle(MB_WITH_SELF ucMBFrame, &usLength );
            }

            /* If the request was not sent to the broadcast address we
             * return a reply. */
            if( ucRcvAddress != MB_ADDRESS_BROADCAST )
            {
                if( eException != MB_EX_NONE )
                {
                    /* An exception occured. Build an error frame. */
                    usLength = 0;
                    ucMBFrame[usLength++] = ( UCHAR )( ucFunctionCode | MB_FUNC_ERROR );
                    ucMBFrame[usLength++] = eException;
                }
#             if MB_ASCII_ENABLED
                if( ( eMBCurrentMode == MB_ASCII ) && MB_ASCII_TIMEOUT_WAIT_BEFORE_SEND_MS )
                {
                    vMBPortTimersDelay( MB_ASCII_TIMEOUT_WAIT_BEFORE_SEND_MS );
                }
#             endif
                eStatus = peMBFrameSendCur( ucMBAddress, ucMBFrame, usLength );
            }
          }
            break;

        case EV_FRAME_SENT:
            break;

        case EV_CLOSE :
            eMBState = STATE_TERMINATE;
            break;

        default : 
			break;
        }
    }
    return MB_ENOERR;
}



#include <mbmaster.h>
#ifndef MB_IS_MULTY
UCHAR    ucSentAddress;
const  UCHAR * SentFrameBuf;
UFRAMESIZE SentFrameLength;

//ACK frame valid after post completes with ok
MB_frame_PDU  * MBACKFrame;
UCHAR    MBACKAddress;
UFRAMESIZE   MBACKLength;
USHORT  MBRetry_count;

eMBErrorCode    MBPostState = MB_BUSY;
MB_protocol_state MBServe_state = MB_ST_IDLE;
#endif

eMBErrorCode    MBS_METHOD_DECL1(MBRepostFrame){
    MB_SERVE_PORT_DECL;
    if (MBPostState != MB_BUSY) {
        MBRetry_count++;
        eMBErrorCode result = peMBFramePostCur(ucSentAddress, SentFrameBuf, SentFrameLength);
        MBPostState = (result == MB_ENOERR) ? MB_BUSY : result;
        return( result );
    }
    else {
        return(MB_EIO);
    }
}

eMBErrorCode    MBS_METHOD_DECL(MBPostFrame, UCHAR slaveAddress,
                            const UCHAR * pucFrame,
                            UFRAMESIZE usLength )
{
    //MB_SERVE_PORT_DECL;
    if (MBPostState != MB_BUSY) {
        ucSentAddress = slaveAddress;
        SentFrameBuf = pucFrame;
        SentFrameLength = usLength;
        MBRetry_count = 0;
        return MBS_METHOD_ACT1(MBRepostFrame);
    }
    else {
        return(MB_EIO);
    }
}



#ifndef RETRY_LIMIT
#define RETRY_LIMIT 1
#endif

#ifndef MB_SENTOK_CB
#define MB_SENTOK_CB(x)
#endif

#ifndef MB_ACKERROR_CB
#define MB_ACKERROR_CB(x)
#endif

#ifndef MB_ERROR_CB
#define MB_ERROR_CB(x)
#endif


eMBErrorCode    MBS_METHOD_DECL1(eMBPollMaster)
{
#define ucMBFrame MBACKFrame
#define ucRcvAddress MBACKAddress
#define usLength MBACKLength
    MB_SERVE_PORT_DECL;

    eMBErrorCode    eStatus = MB_ENOERR;
    eMBEventType    eEvent;

    /* Check if the protocol stack is ready. */
    if( eMBState <= STATE_DISABLED )
    {
        return MB_EILLSTATE;
        MBPostState = MB_BUSY;
    }

    /* Check if there is a event available. If not return control to caller.
     * Otherwise we will handle the event. */
    if( xMBPortEventGet( &eEvent ) == TRUE )
    {
        switch ( eEvent )
        {
        case EV_READY:
            MBServe_state = MB_ST_IDLE;
            MBPostState = eStatus;
            break;

        case EV_FRAME_RECEIVED:
            switch (MBServe_state) {
                case MBM_ST_WAIT_ACK:   //master wait ack
                    MBProtocolTimeOutOff();
                    eStatus = peMBFrameReceiveCur( &ucRcvAddress, (UCHAR**)&ucMBFrame, &usLength );
                    if( ( eStatus == MB_ENOERR ) 
                        ) 
                    {
                        /* Check if the frame is for us. If not ignore the frame. */
                        if ( ucRcvAddress == ucSentAddress ) 
                      {
                        if ((* ((UCHAR*)ucMBFrame) & MB_FUNC_ERROR) != 0)
                            { eStatus = MB_EREPLY_FAIL; }

                            //Acknowledge fine, so report it
                            MBPostState = eStatus;
                            MB_SENTOK_CB(eStatus);
                      }
                      else {
                        /*If not ignore the frame. leave wait ack state*/
                        break;
                      }
                    }
                    else {
                        eStatus = MB_EACK_FAIL;
                        MBPostState = eStatus;
                        if (MBRetry_count > RETRY_LIMIT) {
                            MB_ACKERROR_CB(eStatus);
                        }
                        else { 
                            MBRepostFrame();
                        }
                    }
                    MBServe_state = MB_ST_IDLE;
                    break;
#ifdef MB_MIXED_MASTERSLAVE
                case MB_ST_ACK :        //ack send after receive slave
                case MBM_ST_SEND:       //master send
                     // receiving during send is it possible? ommit received frame
                    break;
                case MB_ST_IDLE :       //generic idle
                        MBProtocolTimeOutOff();
                        //slave request received
                        eStatus = peMBFrameReceiveCur( &ucRcvAddress, &ucMBFrame, &usLength );
                        if( eStatus == MB_ENOERR )
                        {
                            /* Check if the frame is for us. If not ignore the frame. */
                            if( ( ucRcvAddress == ucMBAddress ) || ( ucRcvAddress == MB_ADDRESS_BROADCAST ) )
                            {
                                ( void )xMBPortEventPost( EV_EXECUTE );
                            }
                        }
                        break;
#else
                default :
                        break;
#endif
            }
            break;

#ifdef MB_MIXED_MASTERSLAVE
        case EV_EXECUTE: {
            /*static*/ UCHAR        ucFunctionCode;
            /*static*/ eMBException eException;
            ucFunctionCode = ucMBFrame[MB_PDU_FUNC_OFF];
            eException = MB_EX_ILLEGAL_FUNCTION;
            pxMBFunctionHandler Handle = MBSlaveHandler(ucFunctionCode);
            if (Handle != NULL) {
                    eException = Handle( ucMBFrame, &usLength );
            }

            /* If the request was not sent to the broadcast address we
             * return a reply. */
            MBServe_state = MB_ST_IDLE;
            if( ucRcvAddress != MB_ADDRESS_BROADCAST )
            {
                if( eException != MB_EX_NONE )
                {
                    /* An exception occured. Build an error frame. */
                    ucMBFrame[0] = ( UCHAR )( ucFunctionCode | MB_FUNC_ERROR );
                    ucMBFrame[1] = eException;
                    usLength = 2;
                }
                MBServe_state = MB_ST_ACK;
                eStatus = peMBFrameSendCur( ucMBAddress, ucMBFrame, usLength );
            }
            }
            break;
#endif

        case EV_FRAME_SENT:
            MBProtocolTimeOutOff();
//#ifdef MB_MIXED_MASTERSLAVE
            if (MBServe_state == MB_ST_ACK) {
                MBServe_state = MB_ST_IDLE;
                break;
            }
//#endif
            //starts waiting Acknowledge answer after master sent frame
            if ( ucSentAddress != MB_ADDRESS_BROADCAST ) 
                { MBProtocolTimeOut( MB_ACK_TIMEOUT );}
            else
                { MBProtocolTimeOut( MB_TURNAROUND_TIMEOUT );}
            MBServe_state = MBM_ST_WAIT_ACK;
            break;

            //if ACK
        //case EV_RECEIVER_TIMEOUT :
        case EV_PROTOCOL_TIMEOUT : {
            MB_protocol_state wasstate = MBServe_state;
            MBServe_state = MB_ST_IDLE;
            MBPostState = eStatus;
            switch (wasstate) {
                case MBM_ST_WAIT_ACK:   //master wait ack
                        if (eMBStatus.IsRecvFrameLoading != 0)  {
                            MBServe_state = wasstate;
                            MBProtocolTimeOut( MB_ACK_TIMEOUT );
                            break;
                        }
                        if ( ucSentAddress != MB_ADDRESS_BROADCAST ) {
                            if (MBRetry_count > RETRY_LIMIT) {
                                eStatus = MB_EACK_TIMEOUT;
                                MBPostState = eStatus;
                                MB_ACKERROR_CB(eStatus);
                            }
                            else { 
                                MBPostState = eStatus;
                                MBRepostFrame();
                            }
                        }
                        else{
                            //broadcast frame success after turnaround time
                            MBPostState = eStatus;
                            MB_SENTOK_CB(eStatus);
                        }
                        break;
                default : 
                        eStatus = MB_SEND_TIMEOUT;
                        MBPostState = eStatus;
                        MB_ERROR_CB(eStatus);
            } // EV_PROTOCOL_TIMEOUT : switch (state)
        }//case EV_PROTOCOL_TIMEOUT
            break;
        default : 
            break;    
        }//switch ( eEvent )
    } //if( xMBPortEventGet
    else{
        if (MBProtocolTimeOutExpired()) {
            MBProtocolTimeOutOff();
            ( void )xMBPortEventPost( EV_PROTOCOL_TIMEOUT );
        }
    }
    return MB_ENOERR;
}


BOOL MB_METHOD_DECL1(MBTerminate){
    return xMBPortEventPost(EV_CLOSE);
}
