/*
 * FreeModbus Libary: uOS-elvees Port
 * ru UTF8
    Copyright (C) 2015  Alexandr Litjagin (aka AlexRayne) AlexRaynePE196@gmail.com
                                                          AlexRaynePE196@hotbox.ru
   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
 */

#include "runtime/lib.h"
#include "kernel/uos.h"
#include <kernel/internal.h>
#include <stddef.h>

#include <timer/timer.h>
#include <timer/timeout.h>

#if defined (ELVEES)
//#   include <uos/elvees-io.h>
#else
#   error "modbus cant handle uncknown CPU"
#endif

#include <port.h>
#include <mbtask.h>

#include <usart-defs.h>

/* ----------------------- Modbus includes ----------------------------------*/
#include <mb.h>
#include <mbport.h>
#include <mbstruct.h>
#include <mbrtu.h>

#include "portevent.h"

#ifndef MB_USE_RXFIFO
#define MB_USE_RXFIFO       MC_FCR_TRIGGER_8
#define MB_RXFIFO_LEVEL     8
#define MB_RXFIFO_CTIDELAY  4
#endif

typedef timeout_t       mb_timer_t;
typedef timeout_time_t  mb_time_t;

struct mb_context{
#if defined(UART_MODBUS)
    //const UART_REGS_TYPE* io = UART_MODBUS;
#else
// это пока бесполезный хук, все равно модбас умеет работать только с одним портом пока 
    UART_REGS_TYPE* io;
#endif
    char            irq_state;
    char            fifo_rx_tglevel;
    mutex_t         io_signal;
    mb_timer_t      timer;
    mutex_t         timer_lock;
    mutex_irq_t     timer_irq;
    mb_time_t       timeout;
#ifdef MB_USE_RXFIFO
    mb_time_t       to_175;
    mb_time_t       to_cti;
#endif
    ARRAY (signals_group, (sizeof(mutex_group_t) + 2 * sizeof(mutex_slot_t)) ) ;
};
struct mb_context mb_ctx;


inline UART_REGS_TYPE* mb_uio(struct mb_context* self){
#ifdef UART_MODBUS
    return &UART_MODBUS;
#else
    return self->io;
#endif
}

volatile BOOL InException = 0;
bool_t MBPortSerial_ISR (void* data);
bool_t MBPortTimeout_ISR (void* data);


#define UART_BAUD_CALC(UART_BAUD_RATE,F_OSC) \
    ( ( F_OSC ) / ( ( UART_BAUD_RATE ) * 16UL ) - 1 )

CODE_ISR 
void 
vMBPortSerialEnable( BOOL xRxEnable, BOOL xTxEnable )
{
    UART_REGS_TYPE* io = mb_uio(&mb_ctx);
    assert(io != NULL);

    if( xRxEnable ) {
        io->IER.data |= MC_IER_ERXRDY|MC_IER_ERLS;
        RE_ON();
    }
    else {
        io->IER.data &= ~(MC_IER_ERXRDY|MC_IER_ERLS);
        RE_OFF();
    }

    if( xTxEnable )
    {
        io->IER.bits.ETBEI = 1;
        DE_ON();
    }
    else
    {
        io->IER.bits.ETBEI = 0;
        DE_OFF();
    }
}




BOOL
xMBPortSerialInit( UCHAR ucPORT, ULONG ulBaudRate, UCHAR ucDataBits, eMBParity eParity )
{
    unsigned divisor = UART_BAUD_CALC (ulBaudRate, F_CPU);
#ifdef UART_MODBUS
    UART_REGS_TYPE* io = &UART_MODBUS;
#else
    UART_REGS_TYPE* io = &UART_IO(ucPORT);
    mb_ctx.io = io;
#endif
    io->SCLR.data = 0;
    io->SPR.data = 0;
    io->IER.data = 0;
    io->LCR.data = MC_LCR_8BITS | MC_LCR_DLAB;
    io->DLM.data = divisor >> 8;
    io->DLL.data = divisor;
    unsigned par = (eParity != MB_PAR_NONE)? MC_LCR_PENAB : 0;
    par |= (eParity == MB_PAR_EVEN)? MC_LCR_PEVEN : 0;
    io->LCR.data = MC_LCR_8BITS | MC_LCR_STOPB | par ;
    //enable FIFO with 1char receive awake level
    unsigned fcr = MC_FCR_TRIGGER_1 ;
    mb_ctx.fifo_rx_tglevel = 1;
#if MB_USE_RXFIFO > 0
    const unsigned char_bits = 12;
    mb_ctx.to_cti = (MB_RXFIFO_CTIDELAY*char_bits*divisor*16)/(F_CPU/1000000UL);
    if (ulBaudRate > 27500ul){
        fcr = MB_USE_RXFIFO ;
        mb_ctx.fifo_rx_tglevel = MB_RXFIFO_LEVEL;
    }
#endif
    io->FCR.data = MC_FCR_RCV_RST | MC_FCR_XMT_RST | MC_FCR_ENABLE | fcr; 
    /* Clear pending status, data and irq. */
    (void) io->LSR;
    (void) io->MSR;
    (void) io->RBR;
    (void) io->IIR;

    DE_INIT();
    RE_INIT();
    vMBPortSerialEnable( FALSE, FALSE );

    mutex_init(&(mb_ctx.timer_lock));
    mutex_init(&(mb_ctx.io_signal));

    mutex_attach_irq(&(mb_ctx.io_signal), UART_IO_EVT(io), &(MBPortSerial_ISR), (void*)&mb_ctx);

    mutex_group_t *mb_signals;
    mb_signals = mutex_group_init (mb_ctx.signals_group, sizeof(mb_ctx.signals_group));
    mutex_group_add (mb_signals, &(mb_ctx.io_signal) );
    mutex_group_add (mb_signals, &(mb_ctx.timer_lock));
    mutex_group_listen (mb_signals);
    
    return TRUE;
}

void  xMBPortSerialClose( void ){
    UART_REGS_TYPE* io = mb_uio(&mb_ctx);
    if (io == NULL)return;
    //reset and disable FIFO with 1char receive awake level
    io->FCR.data = MC_FCR_RCV_RST | MC_FCR_XMT_RST;
    //disable all isr
    io->IER.data = 0;
    DE_OFF();
    RE_OFF();
    mutex_group_unlisten((mutex_group_t *)(mb_ctx.signals_group));
}

CODE_ISR 
BOOL
xMBPortSerialPutByte( UCHAR ucByte )
{
    UART_REGS_TYPE* io = mb_uio(&mb_ctx);
    if (io != NULL){
        io->THR.data = ucByte;
        DEBUG_MBtx(ucByte);
    }
    return TRUE;
}

#ifndef MODBUS_FIFOTX_SIZE
#define MODBUS_FIFOTX_SIZE 1
#endif

CODE_ISR 
UFRAMESIZE
xMBPortSerialPutBytes( UCHAR* ucBytes, UFRAMESIZE len )
{
    UART_REGS_TYPE* io = mb_uio(&mb_ctx);
    if (io != NULL){
        if (io->LSR.bits.THRE != 0){
            if (len > MODBUS_FIFOTX_SIZE)
                len = MODBUS_FIFOTX_SIZE;
        }
        else
            return 0;
        UFRAMESIZE res = len;
        for ( ;(len > 0); len--)
        {
            DEBUG_MBtx(*ucBytes);
            io->THR.data = *ucBytes++;
        }
        return res;
    }
    return 0;
}

CODE_ISR 
BOOL
xMBPortSerialGetByte( UCHAR * pucByte )
{
    UART_REGS_TYPE* io = mb_uio(&mb_ctx);
    if (io != NULL){
        unsigned stat = io->LSR.data;
        if ((stat & MC_LSR_RXRDY) != 0)
            *pucByte = (io->RBR.data) & 0xff;
        else
            return FALSE;
        if ((stat & MC_LSR_BI) != 0) {
            // have line-Break, just skip it
            stat = io->LSR.data;
            if ((stat & MC_LSR_RXRDY) != 0)
                *pucByte = (io->RBR.data) & 0xff;
            else
                return FALSE;
        }
        DEBUG_MBrx((*pucByte));
        return TRUE;
    }
    return FALSE;
}

//* same as xMBPortSerialGetByte, also returns uart error status
//* \param[out] puByte - low byte - received char, 
//*                       high byte - error info, 0 - if no errors
CODE_ISR 
BOOL            xMBPortSerialGetByteX( unsigned * puByte ){
    UART_REGS_TYPE* io = mb_uio(&mb_ctx);
    if (io != NULL){
        unsigned stat = io->LSR.data;
        unsigned tmp = 0;
        if ((stat & MC_LSR_RXRDY) != 0)
            tmp = (io->RBR.data) & 0xff;
        else
            return FALSE;
        if ((stat & MC_LSR_BI) != 0) {
            // have line-Break, just skip it
            stat = io->LSR.data;
            if ((stat & MC_LSR_RXRDY) != 0)
                tmp = (io->RBR.data) & 0xff;
            else
                return FALSE;
        }
        const unsigned errors = MC_LSR_OE | MC_LSR_PE | MC_LSR_FE;
        tmp |= ((stat & errors) << 8);
        *puByte = tmp;
        DEBUG_MBrx((*puByte));
        return TRUE;
    }
    return FALSE;
}

/*
 * interrupt handler.
 */
CODE_ISR 
bool_t 
MBPortSerial_ISR (void* data)
{

    struct mb_context* self = (struct mb_context*)data;
    UART_REGS_TYPE* io = mb_uio(self);
    //mcIIR_IID_Code iir_code = io->IIR.bits.IP
    MB_PortSetWithinException(TRUE);

    int need_poll = 0;
    unsigned stat = io->LSR.data;

    if (io->IER.bits.ERBI) {

        if ((stat & MC_LSR_BI) != 0){
            (io->RBR.data);
            stat = io->LSR.data;
        }

#if MB_USE_RXFIFO > 0
        trace_probe_mb_irx_on();
        unsigned iircode = 0;
        unsigned limit = ~0;
    if (self->to_cti > 0)
    {
        iircode = io->IIR.data & MC_IIR_IMASK;
        if (iircode == MC_IIR_RXTOUT){
            //received char timedout after 4char delay, so wait least timeout after it 
            self->timeout = self->to_175 - self->to_cti;
            vMBPortTimersEnable(  );
        }
        //else - при полном буфере приема, мы его вычитаем весь, и на пустом буфере 
        //  4char таймаут не возникает!!!
        else{
            // чтобы не потерять прерывание от 4char не считываю последний символ!
            limit = self->fifo_rx_tglevel-1;
            vMBPortTimersDisable(  );
        }
        // отключу ненужный старт таймаута - отложу его после обработки всего фифо   
        self->timeout = 0;
    }
    trace_probe_mb_irx_off();

        for ( ; ((stat & MC_LSR_RXRDY) != 0) && (limit > 0) ;
                limit --
            )
        {
            need_poll += (pxMBFrameCBByteReceived());
            stat = io->LSR.data;
        }
        self->timeout = self->to_175;
#else
    while ((stat & MC_LSR_RXRDY) != 0){
        need_poll += (pxMBFrameCBByteReceived());
        stat = io->LSR.data;
    }
#endif


    } //if (io->IER.bits.ERBI)

    if (io->IER.bits.ETBEI) {
    if ((stat & MC_LSR_TXRDY) != 0){
        trace_probe_mb_itx_on();
        need_poll += (pxMBFrameCBTransmitterEmpty());
        trace_probe_mb_itx_off();
    }
    if ((stat & MC_LSR_TEMT) != 0){
        DE_OFF();
    }
    } //if (io->IER.bits.ETBEI)

    MB_PortSetWithinException(FALSE);
    // !!!HARDCODE - UART_ISR может зависеть от выбраного порта xMBPortSerialInit 
    arch_intr_allow (UART_IO_EVT(io));
    return (need_poll == TRUE)?0:1;
}



//***************************************************************************
//                          other
//***************************************************************************
CODE_FAST 
void
MB_PortEnterCritical( void )
{
    arch_state_t x;
    arch_intr_disable(&x);
    mb_ctx.irq_state = IRQ_disable(UART_IO_EVT(mb_uio(&mb_ctx)));
    arch_intr_restore (x);
}

CODE_FAST 
void
MB_PortExitCritical( void )
{
    if (mb_ctx.irq_state == 0)
        return;
    arch_state_t x;
    arch_intr_disable(&x);
    IRQ_enable(UART_IO_EVT(mb_uio(&mb_ctx)));
    arch_intr_restore (x);
}



void
vMBPortClose( void )
{
    xMBPortEventPost(EV_CLOSE);
    /*  
    // этот код должен быть выполнен после перехода модуля(поллера) в состояние
    //  MB_CLOSED  
    ENTER_CRITICAL_SECTION();
    xMBPortSerialClose(  );
    xMBPortTimerClose(  );
    EXIT_CRITICAL_SECTION();
    xMBPortEventClose(  );
    */
}


BOOL MBWaitEvent(){
    mutex_group_t *g = (mutex_group_t *)mb_ctx.signals_group;
    mutex_t *m = NULL;
    void *msg  = NULL;
    trace_probe_mb_ev_on();
    mutex_group_wait (g, &m, &msg);
    trace_probe_mb_ev_off();
    return m != NULL;
}

//***************************************************************************
//                          timer
//***************************************************************************
/* ----------------------- Defines ------------------------------------------*/
#define MB_TIMER_PRESCALER      ( 1 )
#define MB_TIMER_TICKS          ( F_CPU / MB_TIMER_PRESCALER )
#define MB_50US_TICKS           ( MB_TIMER_TICKS/20000UL )

/* ----------------------- Static variables ---------------------------------*/

extern timer_t          sys_clock;

/* ----------------------- Start implementation -----------------------------*/
#ifndef USEC_TIMER
#error <ModBUS need timer submilisecond resolution>
#endif

BOOL
xMBPortTimersInit( USHORT usTimerout50us )
{
    struct mb_context* self = &mb_ctx;
    mb_timer_t* mb_timer = &(self->timer);
    timeout_clear(mb_timer);
    timeout_set_autoreload(mb_timer, tsLoadOnce);
    self->timeout = usTimerout50us*50UL;
#if MB_USE_RXFIFO > 0
    self->to_175 = self->timeout;
#endif
    mutex_lock_swi(&(self->timer_lock), &(self->timer_irq), &(MBPortTimeout_ISR), (void*)self);
    mutex_unlock(&(self->timer_lock));
    timeout_set_mutex(mb_timer, &(self->timer_lock), self);
    timeout_add(&sys_clock, mb_timer);
    return TRUE;
}

void
xMBPortTimersClose( void )
{
    mb_timer_t* mb_timer = &mb_ctx.timer;
    timeout_stop(mb_timer);
}

// timeout == 0  turned off checking
CODE_ISR 
void vMBPortTimersEnable(  )
{
    struct mb_context* self = &mb_ctx;
    if (self->timeout <= 0)
        return;
    trace_probe_mb_to_on();
    timeout_arm_us_nomt(&(self->timer), self->timeout);
}

CODE_ISR 
void vMBPortTimersDisable(  )
{
    mb_timer_t* mb_timer = &(mb_ctx.timer);
    timeout_stop(mb_timer);
    trace_probe_mb_to_off();
}

CODE_ISR 
bool_t 
MBPortTimeout_ISR (void* data)
{
    bool_t need_poll = FALSE;
    MB_PortSetWithinException(TRUE);
    need_poll = pxMBPortCBTimerExpired();
    MB_PortSetWithinException(FALSE);
    return (need_poll == TRUE)?0:1;
}

void MBProtocolTimeOut( USHORT usTimeOutX50 ){
    struct mb_context*self = &mb_ctx;
    mb_timer_t* mb_timer = &(self->timer);
    self->timeout = usTimeOutX50*50;
    timeout_arm_us(mb_timer, self->timeout);
}

void  MBProtocolTimeOutOff(void){
    mb_timer_t* mb_timer = &mb_ctx.timer;
    timeout_break(mb_timer);
    trace_probe_mb_to_off();
}

BOOL MBProtocolTimeOutExpired( void ){
    mb_timer_t* mb_timer = &mb_ctx.timer;
    return timeout_expired(mb_timer);
}
