package jp.sourceforge.freegantt.swing;

import java.awt.Color;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.text.SimpleDateFormat;
import java.util.Calendar;

import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollBar;
import javax.swing.undo.CompoundEdit;

import jp.sourceforge.freegantt.locale.Resource;
import jp.sourceforge.freegantt.util.CalendarUtil;
import jp.sourceforge.freegantt.util.GraphicsUtil;

public class TaskLineHeaderPane extends JPanel implements MouseListener, MouseMotionListener {
	private static final long serialVersionUID = 4416176242747877014L;
	
	public final static int OPERATION_NONE = 0x00;
	public final static int OPERATION_FREE_SCROLL = 0x01;
	
	
	Application app;
	TaskLinePane taskLinePane;
	CursorState cursorState;
	FreeScrollUI freeScrollUI;
	int dragOperation = OPERATION_NONE;
	
	Color lineColor = new Color(0xC0, 0xC0, 0xC0);
	Color bgColor = Color.white;
	Color fontColor = Color.black;
	
	public TaskLineHeaderPane(Application app, TaskLinePane taskLinePane) {
		super();
		this.app = app;
		this.taskLinePane = taskLinePane;
		this.cursorState = new CursorState(this);
		this.freeScrollUI = new FreeScrollUI();
		setBackground(bgColor);
		addMouseListener(this);
		addMouseMotionListener(this);
	}

	private int getCellHeight() {
		return taskLinePane.getCellHeight();
	}
	
	private int getCellWidth() {
		return taskLinePane.getCellWidth();
	}
	
	public void updateWidth() {
		Calendar fromDate = taskLinePane.getChartFromDate();
		Calendar toDate = taskLinePane.getChartToDate();
		setSize(CalendarUtil.subDate(toDate, fromDate) * getCellWidth(), getHeight());
	}

	@Override
	protected void paintComponent(Graphics g) {
		super.paintComponent(g);
		
		g.setColor(lineColor);
		g.drawLine(0, getCellHeight() - 1, getWidth(), getCellHeight() - 1);
		g.drawLine(0, getCellHeight() * 2 - 1, getWidth(), getCellHeight() * 2 - 1);
		
		Calendar now = (Calendar)taskLinePane.getChartFromDate().clone();
		Calendar to = taskLinePane.getChartToDate();

		int offset = 0;
		SimpleDateFormat format = new SimpleDateFormat(Resource.get("chartHeaderMonth"), Resource.getLocale());
		while (now.getTimeInMillis() < to.getTimeInMillis()) {
			if (app.getProject().isCalendarModeDate()) {
				g.setColor(fontColor);
				if (now.get(Calendar.DAY_OF_MONTH ) == 1) {
					String text = format.format(now.getTime());
					GraphicsUtil.drawStringLeft(g, text, offset, 0, getCellWidth(), getCellHeight());
				}
				GraphicsUtil.drawScaledDate(g, String.valueOf(now.get(Calendar.DAY_OF_MONTH)), offset, getCellHeight(), getCellWidth(), getCellHeight(), 0.7);
					
				now.add(Calendar.DATE, 1);
				offset += getCellWidth();
				
				// 縦線の描画（月末は上段まで線を引く）
				g.setColor(lineColor);
				if (now.get(Calendar.DAY_OF_MONTH ) == 1) {
					g.drawLine(offset - 1, 0, offset - 1, getHeight() - 1);
				} else {
					g.drawLine(offset - 1, getCellHeight() - 1, offset - 1, getCellHeight() * 2 - 1);
				}
			} else {
				g.setColor(fontColor);
				if (now.get(Calendar.DAY_OF_MONTH ) == 1) {
					String text = format.format(now.getTime());
					GraphicsUtil.drawStringLeft(g, text, offset, 0, getCellWidth(), getCellHeight());
				}
				if (now.get(Calendar.DAY_OF_WEEK) == Calendar.MONDAY) {
					GraphicsUtil.drawScaledDate(g, String.valueOf(now.get(Calendar.DAY_OF_MONTH)), offset, getCellHeight(), getCellWidth(), getCellHeight(), 0.7);
				}
					
				now.add(Calendar.DATE, 1);
				offset += getCellWidth();
				
				// 縦線の描画（月末は上段まで線を引く）
				g.setColor(lineColor);
				if (now.get(Calendar.DAY_OF_MONTH ) == 1) {
					g.drawLine(offset - 1, 0, offset - 1, getCellHeight() - 1);
				}
				if (now.get(Calendar.DAY_OF_WEEK) == Calendar.MONDAY) {
					g.drawLine(offset - 1, getCellHeight() - 1, offset - 1, getCellHeight() * 2 - 1);
				}
			}
		}
	}

	@Override
	public void mouseClicked(MouseEvent e) {
	}

	@Override
	public void mousePressed(MouseEvent e) {
		if (e.getButton() == MouseEvent.BUTTON1) {
			if (freeScrollUI.mousePressed(e)) return;
		}
		if (e.isPopupTrigger() && e.getY() >= getCellHeight()) {
			triggerPopup(e);
		}
	}

	@Override
	public void mouseReleased(MouseEvent e) {
		if (e.getButton() == MouseEvent.BUTTON1) {
			if (freeScrollUI.mouseReleased(e)) return;
		}
		if (e.isPopupTrigger() && e.getY() >= getCellHeight()) {
			triggerPopup(e);
		}
	}
	

	@Override
	public void mouseEntered(MouseEvent e) {
	}
	

	@Override
	public void mouseExited(MouseEvent e) {
	}

	@Override
	public void mouseDragged(MouseEvent e) {
		if (freeScrollUI.mouseDragged(e)) return;
	}

	@Override
	public void mouseMoved(MouseEvent e) {
	}

	private Calendar getCalendarAtPoint(Point p) {
		long offsetDate = p.x / getCellWidth();
		Calendar calendar = Calendar.getInstance();
		calendar.setTimeInMillis(taskLinePane.getChartFromDate().getTimeInMillis() + offsetDate * 86400000);
		return calendar;
	}
	
	private void triggerPopup(MouseEvent e) {
		final Calendar calendar = getCalendarAtPoint(e.getPoint());
		
		JPopupMenu menu = new JPopupMenu();
		JMenuItem item = new JMenuItem(Resource.get("chartHeaderAddHoliday"));
		item.setEnabled(!app.getProject().isHoliday(calendar));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				CompoundEdit compound = new CompoundEdit();
				app.getProject().getUndoManager().addEdit(compound);
				try {
					app.getProject().getController().addAdditionalHoliday(calendar);
					app.getProject().update();
				} finally {
					compound.end();
				}
			}
		});
		menu.add(item);
		item = new JMenuItem(Resource.get("chartHeaderRemoveHoliday"));
		item.setEnabled(app.getProject().isAdditionalHoliday(calendar));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				CompoundEdit compound = new CompoundEdit();
				app.getProject().getUndoManager().addEdit(compound);
				try {
					app.getProject().getController().removeAdditionalHoliday(calendar);
					app.getProject().update();
				} finally {
					compound.end();
				}
			}
		});
		menu.add(item);
		menu.show(this, e.getX(), e.getY());
	}
	
	class FreeScrollUI {

		/** ドラッグ開始位置 */
		Point dragStartPoint;
		/** フリースクロール開始位置 */
		Point freeScrollStartPoint;

		public boolean mousePressed(MouseEvent e) {
			dragStartPoint = e.getLocationOnScreen();
			freeScrollStartPoint = new Point(
					app.getTaskLineRootPane().getHorizontalScrollBar().getValue(), 0);
			dragOperation = OPERATION_FREE_SCROLL;
			cursorState.setCursor(Cursor.HAND_CURSOR);
			return true;
		}

		public boolean mouseDragged(MouseEvent e) {
			if (dragOperation == OPERATION_FREE_SCROLL) {
				Point pressPoint = e.getLocationOnScreen();
				Dimension moveDistance = new Dimension(pressPoint.x - dragStartPoint.x, pressPoint.y - dragStartPoint.y);
				JScrollBar hbar = app.getTaskLineRootPane().getHorizontalScrollBar();
				
				int hvalue = freeScrollStartPoint.x - moveDistance.width;
				hvalue = Math.min(hbar.getMaximum() - hbar.getVisibleAmount() - 1, Math.max(1, hvalue));
				
				hbar.setValue(hvalue);
				return true;
			}
			return false;
		}

		public boolean mouseReleased(MouseEvent e) {
			if (dragOperation == OPERATION_FREE_SCROLL) {
				dragOperation = OPERATION_NONE;
				cursorState.setDefaultCursor();
				return true;
			}
			return false;
		}

	}

}
