package jp.sourceforge.freegantt.data.serializer;

import java.awt.print.PageFormat;
import java.awt.print.Paper;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import jp.sourceforge.freegantt.data.Column;
import jp.sourceforge.freegantt.data.Member;
import jp.sourceforge.freegantt.data.Print;
import jp.sourceforge.freegantt.data.Project;
import jp.sourceforge.freegantt.data.Task;
import jp.sourceforge.freegantt.data.View;
import jp.sourceforge.freegantt.util.XmlUtil;

import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

public class FtgSerializer {
	
	public static String VERSION = "1";
	
	public void read(Project project, InputStream in) throws SAXException, IOException, FileFormatException {
		try {
			DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
			DocumentBuilder builder = factory.newDocumentBuilder();
			Document doc = builder.parse(in);
			
			Element element = doc.getDocumentElement();
			parseProject(project, element);
			
		} catch (ParserConfigurationException e) {
		} catch (NumberFormatException e) {
			throw new FileFormatException(e);
		}
	}
	
	
	public void parseProject(Project project, Element ele) throws FileFormatException {
		project.setName( XmlUtil.getTextByTagName(ele, "name") );
		project.setSummary( XmlUtil.getTextByTagName(ele, "summary") );
		project.setMembers(parseMembers(ele));
		project.setTasks(parseTasks(project, ele));
		
		Element fixedHolidays = XmlUtil.getFirstElementByTag(ele, "fixedHolidays");
		if (fixedHolidays != null) {
			List<String> weeks = XmlUtil.getTextsByTagName(fixedHolidays, "week");
			for (String week: weeks) {
				project.getFixedHolidays().add(Integer.parseInt(week));
			}
		}
		
		Element additionaldHolidays = XmlUtil.getFirstElementByTag(ele, "additionalHolidays");
		if (additionaldHolidays != null) {
			List<String> dates = XmlUtil.getTextsByTagName(additionaldHolidays, "date");
			for (String date: dates) {
				Calendar calendar = Calendar.getInstance();
				calendar.setTimeInMillis(Long.parseLong(date));
				project.getAdditionalHolidays().add(calendar);
			}
		}
		
		project.setView(parseView(ele));
		project.setPrint(parsePrint(ele));
	}
	
	
	private List<Member> parseMembers(Element ele) throws FileFormatException {
		List<Member> members = new ArrayList<Member>();
		Element membersEle = XmlUtil.getFirstElementByTag(ele, "resources");
		if (membersEle == null) return members;
		
		NodeList memberNodes = membersEle.getElementsByTagName("resource");
		for (int i=0; i<memberNodes.getLength(); i++) {
			Member member = parseMember((Element)memberNodes.item(i));
			if (member != null) members.add(member);
		}
			
		return members;
	}
	
	
	private Member parseMember(Element ele) throws FileFormatException {
		Member member = new Member();
		member.setName(XmlUtil.getTextByTagName(ele, "name"));
		return member;
	}
	
	
	private List<Task> parseTasks(Project project, Element ele) throws FileFormatException {
		List<Task> tasks = new ArrayList<Task>();
		Element tasksEle = XmlUtil.getFirstElementByTag(ele, "tasks");
		if (tasksEle == null) return tasks;
		
		NodeList taskNodes = tasksEle.getElementsByTagName("task");
		for (int i=0; i<taskNodes.getLength(); i++) {
			parseTask(project, tasks, (Element)taskNodes.item(i));
		}
		project.setTasks(tasks);
		
		for (int i=0; i<taskNodes.getLength(); i++) {
			parseRestrictions(project, (Element)taskNodes.item(i));
		}

		return tasks;
	}
	
	
	private void parseTask(Project project, List<Task> tasks, Element ele) throws FileFormatException {
		int index = XmlUtil.getIntegerByTagName(ele, "index");
		
		Task task = new Task();
		task.setName(XmlUtil.getTextByTagName(ele, "name"));
		task.setStartDate(XmlUtil.getCalendarByTagName(ele, "startDate", null));
		task.setPeriod(XmlUtil.getIntegerByTagName(ele, "period", null));
		task.setCompletion(XmlUtil.getIntegerByTagName(ele, "completion"));
		task.setLevel(XmlUtil.getIntegerByTagName(ele, "level"));
		Integer memberIndex = XmlUtil.getIntegerByTagName(ele, "resource", null);
		if (memberIndex != null && memberIndex < project.getMembers().size()) {
			task.setMember(project.getMembers().get(memberIndex));
		}
		task.setOpened(XmlUtil.getBooleanByTagName(ele, "opened", true));
		task.setVisible(XmlUtil.getBooleanByTagName(ele, "visible", true));

		while (tasks.size() <= index) {
			tasks.add(new Task());
		}
		tasks.set(index, task);
	}
	
	
	private void parseRestrictions(Project project, Element ele) throws FileFormatException {
		int index = XmlUtil.getIntegerByTagName(ele, "index");
		Task task = project.getTaskAtIndex(index);
		
		Element restrictions = XmlUtil.getFirstElementByTag(ele, "restrictions");
		if (restrictions != null) {
			List<String> dests = XmlUtil.getTextsByTagName(restrictions, "dest");
			for (String dest: dests) {
				int destIndex = Integer.parseInt(dest);
				Task destTask = project.getTaskAtIndex(destIndex);
				task.getRestrictions().add(destTask);
			}
		}

	}
	
	
	private View parseView(Element ele) throws FileFormatException {
		View view = new View();
		Element viewEle = XmlUtil.getFirstElementByTag(ele, "view");
		if (viewEle == null) return view;
		
		view.setTaskTableWidth(XmlUtil.getIntegerByTagName(viewEle, "taskTableWidth", view.getTaskTableWidth()));
		
		Element columnsEle = XmlUtil.getFirstElementByTag(ele, "columns");
		if (columnsEle == null) return null;
		
		List<Column> columns = new ArrayList<Column>();
		List<Element> columnEles = XmlUtil.getElementsByTagName(columnsEle, "column");
		for (Element columnEle: columnEles) {
			Column column = new Column();
			column.setKey(XmlUtil.getTextByTagName(columnEle, "key"));
			column.setWidth(XmlUtil.getIntegerByTagName(columnEle, "width"));
			columns.add(column);
		}
		view.setColumns(columns);
		
		
		return view;
	}
	
	
	private Print parsePrint(Element ele) throws FileFormatException {
		Print print = new Print();
		Element printEle = XmlUtil.getFirstElementByTag(ele, "print");
		if (printEle == null) return print;
		
		print.setPrintCellSize(XmlUtil.getDimensionByTagName(printEle, "printCellSize", print.getPrintCellSize()));
		
		PageFormat pageFormat = print.getPageFormat();
		Paper paper = pageFormat.getPaper();
		pageFormat.setOrientation(XmlUtil.getIntegerByTagName(printEle, "orientation", pageFormat.getOrientation()));
		paper.setSize(
				XmlUtil.getDoubleByTagName(printEle, "paperWidth", paper.getWidth()), 
				XmlUtil.getDoubleByTagName(printEle, "paperHeight", paper.getHeight()));
		paper.setImageableArea(
				XmlUtil.getDoubleByTagName(printEle, "paperImageableX", paper.getImageableX()), 
				XmlUtil.getDoubleByTagName(printEle, "paperImageableY", paper.getImageableY()),
				XmlUtil.getDoubleByTagName(printEle, "paperImageableWidth", paper.getImageableWidth()), 
				XmlUtil.getDoubleByTagName(printEle, "paperImageableHeight", paper.getImageableHeight()));
		pageFormat.setPaper(paper);
		
		return print;
	}

	
	public void write(Project project, OutputStream out) {
		try {
			DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
			DocumentBuilder builder = factory.newDocumentBuilder();
			DOMImplementation dom = builder.getDOMImplementation();
			Document doc = dom.createDocument("", "freeganttProject", null);
			
			addProject(project, doc);
			
			DOMSource source = new DOMSource(doc);
			StreamResult result = new StreamResult(out);
			
			TransformerFactory transFactory = TransformerFactory.newInstance();
			Transformer transformer = transFactory.newTransformer();
			transformer.setOutputProperty(OutputKeys.INDENT, "yes");
			transformer.setOutputProperty(OutputKeys.METHOD, "xml");
			transformer.transform(source, result);	
			
		} catch (ParserConfigurationException e) {
			e.printStackTrace();
		} catch (TransformerConfigurationException e) {
			e.printStackTrace();
		} catch (TransformerException e) {
			e.printStackTrace();
		}
	}
	
	
	private void addProject(Project project, Document doc) {
		Element root = doc.getDocumentElement();
		root.setAttribute("version", VERSION);

		root.appendChild(XmlUtil.createTextElement(doc, "name", project.getName()));
		root.appendChild(XmlUtil.createTextElement(doc, "summary", project.getSummary()));
		
		root.appendChild(getMembersElement(project.getMembers(), doc));
		root.appendChild(getTasksElement(project, project.getTasks(), doc));
		
		Element fixedHolidays = doc.createElement("fixedHolidays");
		for (Integer holiday: project.getFixedHolidays()) {
			fixedHolidays.appendChild(XmlUtil.createTextElement(doc, "week", holiday.toString()));
		}
		root.appendChild(fixedHolidays);
		
		Element additionalHolidays = doc.createElement("additionalHolidays");
		for (Calendar holiday: project.getAdditionalHolidays()) {
			additionalHolidays.appendChild(XmlUtil.createTextElement(doc, "date", String.valueOf(holiday.getTimeInMillis())));
		}
		root.appendChild(additionalHolidays);

		root.appendChild(getViewElement(project.getView(), doc));
		root.appendChild(getPrintElement(project.getPrint(), doc));
	}
	
	
	private Element getMembersElement(List<Member> members, Document doc) {
		Element ele = doc.createElement("resources");
		for (Member member: members) {
			ele.appendChild(getMemberElement(member, doc));
		}
		return ele;
	}

	
	private Element getMemberElement(Member member, Document doc) {
		Element element = doc.createElement("resource");
		element.appendChild(XmlUtil.createTextElement(doc, "name", member.getName()));
		return element;
	}
	
	
	private Element getTasksElement(Project project, List<Task> tasks, Document doc) {
		Element ele = doc.createElement("tasks");
		int index = 0;
		for (Task task: project.getTasks()) {
			ele.appendChild(getTaskElement(project, task, index, doc));
			index ++;
		}
		return ele;
	}

	
	private Element getTaskElement(Project project, Task task, int index, Document doc) {
		Element element = doc.createElement("task");
		element.appendChild(XmlUtil.createTextElement(doc, "index", String.valueOf(index)));
		element.appendChild(XmlUtil.createTextElement(doc, "name", (task.getName() == null ? "" : task.getName())));
		if (task.getStartDate() != null) {
			element.appendChild(XmlUtil.createTextElement(doc, "startDate", String.valueOf(task.getStartDate().getTimeInMillis())));
		}
		if (task.getPeriod() != null) {
			element.appendChild(XmlUtil.createTextElement(doc, "period", String.valueOf(task.getPeriod().intValue())));			
		}
		element.appendChild(XmlUtil.createTextElement(doc, "completion", String.valueOf(task.getCompletion())));			
		if (task.getMember() != null) {
			int memberIndex = project.getMembers().indexOf(task.getMember());
			element.appendChild(XmlUtil.createTextElement(doc, "resource", String.valueOf(memberIndex)));			
		}
		element.appendChild(XmlUtil.createTextElement(doc, "level", String.valueOf(task.getLevel())));			
		if (!task.getRestrictions().isEmpty()) {
			Element restrictions = doc.createElement("restrictions");
			for (Task restriction: task.getRestrictions()) {
				restrictions.appendChild(XmlUtil.createTextElement(doc, "dest", String.valueOf(project.getIndexByTask(restriction))));
			}
			element.appendChild(restrictions);
		}
		element.appendChild(XmlUtil.createTextElement(doc, "opened", String.valueOf(task.isVisible())));
		element.appendChild(XmlUtil.createTextElement(doc, "visible", String.valueOf(task.isVisible())));
		
		return element;
	}
	
	
	private Element getViewElement(View view, Document doc) {
		Element ele = doc.createElement("view");
		ele.appendChild(XmlUtil.createTextElement(doc, "taskTableWidth", String.valueOf(view.getTaskTableWidth())));
		
		Element columnsEle = doc.createElement("columns");
		for (Column column: view.getColumns()) {
			Element columnEle = doc.createElement("column");
			columnEle.appendChild(XmlUtil.createTextElement(doc, "key", column.getKey()));
			columnEle.appendChild(XmlUtil.createTextElement(doc, "width", String.valueOf(column.getWidth())));
			columnsEle.appendChild(columnEle);
		}
		ele.appendChild(columnsEle);
		
		return ele;
	}
	
	
	private Element getPrintElement(Print print, Document doc) {
		PageFormat page = print.getPageFormat();
		Element ele = doc.createElement("print");
		ele.appendChild(XmlUtil.createDimensionElement(doc, "printCellSize", print.getPrintCellSize()));
		ele.appendChild(XmlUtil.createTextElement(doc, "orientation", String.valueOf(page.getOrientation())));
		ele.appendChild(XmlUtil.createTextElement(doc, "paperWidth", String.valueOf(page.getPaper().getWidth())));
		ele.appendChild(XmlUtil.createTextElement(doc, "paperHeight", String.valueOf(page.getPaper().getHeight())));
		ele.appendChild(XmlUtil.createTextElement(doc, "paperImageableX", String.valueOf(page.getPaper().getImageableX())));
		ele.appendChild(XmlUtil.createTextElement(doc, "paperImageableY", String.valueOf(page.getPaper().getImageableY())));
		ele.appendChild(XmlUtil.createTextElement(doc, "paperImageableWidth", String.valueOf(page.getPaper().getImageableWidth())));
		ele.appendChild(XmlUtil.createTextElement(doc, "paperImageableHeight", String.valueOf(page.getPaper().getImageableHeight())));
		return ele;
	}
}
