﻿#define DEBUG
using System;
using System.Xml.Serialization;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Threading.Tasks;
using System.IO;
using Windows.ApplicationModel.Store;
using Windows.System;
using Windows.Storage;
using Windows.UI.Core;
using Windows.UI.Xaml;
using Windows.UI.Xaml.Controls;
using System.Runtime.Serialization;
using FooEditEngine.Metro;
using Windows.ApplicationModel.DataTransfer;
using Windows.ApplicationModel.Resources;
using Windows.Graphics.Printing;
using Windows.Storage.Pickers;
using Windows.UI.Popups;

// 基本ページのアイテム テンプレートについては、http://go.microsoft.com/fwlink/?LinkId=234237 を参照してください

namespace FooEditor
{
    /// <summary>
    /// 多くのアプリケーションに共通の特性を指定する基本ページ。
    /// </summary>
    public sealed partial class MainPage : FooEditor.Common.LayoutAwarePage
    {
        MainPageViewModel vm;
        public MainPage()
        {
            this.InitializeComponent();
            this.vm = new MainPageViewModel(this.Frame);
            this.DataContext = this.vm;
            this.DocumentCollectionBar.DataContext = this.vm;
        }

        /// <summary>
        /// このページには、移動中に渡されるコンテンツを設定します。前のセッションからページを
        /// 再作成する場合は、保存状態も指定されます。
        /// </summary>
        /// <param name="navigationParameter">このページが最初に要求されたときに
        /// <see cref="Frame.Navigate(Type, Object)"/> に渡されたパラメーター値。
        /// </param>
        /// <param name="pageState">前のセッションでこのページによって保存された状態の
        /// ディクショナリ。ページに初めてアクセスするとき、状態は null になります。</param>
        protected override async void LoadState(Object navigationParameter, Dictionary<String, Object> pageState)
        {
            this.TopAppBar.Opened += this.AppBar_Opened;
            this.BottomAppBar.Opened += this.AppBar_Opened;
            await this.vm.NavigateHomePage(pageState);
        }

        void AppBar_Opened(object sender, object e)
        {
            AppBar appBar = (AppBar)sender;
            appBar.IsOpen = !this.vm.IsAppBarClosed();
        }

        /// <summary>
        /// アプリケーションが中断される場合、またはページがナビゲーション キャッシュから破棄される場合、
        /// このページに関連付けられた状態を保存します。値は、
        /// <see cref="SuspensionManager.SessionState"/> のシリアル化の要件に準拠する必要があります。
        /// </summary>
        /// <param name="pageState">シリアル化可能な状態で作成される空のディクショナリ。</param>
        protected override async void SaveState(Dictionary<String, Object> pageState)
        {
            await this.vm.LeaveMainPage(pageState);
        }

        private void DocumentCollectionList_ItemClick(object sender, ItemClickEventArgs e)
        {
            DocumentPageInfo info = (DocumentPageInfo)e.ClickedItem;
            this.vm.ActiveDocument(info);
        }

        private async void SaveButton_Click(object sender, RoutedEventArgs e)
        {
            await this.vm.Save();
        }

        private void SearchButton_Click(object sender, RoutedEventArgs e)
        {
            this.vm.ShowSearchFlyout();
        }

        private void GoToButton_Click(object sender, RoutedEventArgs e)
        {
            this.vm.ShowGoToFlyout();
        }

        private void PropertyButton_Click(object sender, RoutedEventArgs e)
        {
            this.vm.ShowPropertyFlyout();
        }

        private void UndoButton_Click(object sender, RoutedEventArgs e)
        {
            this.vm.Undo();
        }

        private void RedoButton_Click(object sender, RoutedEventArgs e)
        {
            this.vm.Redo();
        }
    }
    public struct DocumentPageInfo : IXmlSerializable
    {
        public DocumentControl doc;
        public string Title
        {
            get
            {
                return doc.Title;
            }
        }
        public DocumentPageInfo(DocumentControl doc)
        {
            this.doc = doc;
        }

        public System.Xml.Schema.XmlSchema GetSchema()
        {
            return null;
        }

        public void ReadXml(System.Xml.XmlReader reader)
        {
            reader.ReadStartElement("DocumentPageInfo");
            this.doc = new DocumentControl();
            this.doc.ReadXml(reader);
        }

        public void WriteXml(System.Xml.XmlWriter writer)
        {
            writer.WriteStartElement("DocumentControl");
            this.doc.WriteXml(writer);
            writer.WriteEndElement();
        }
    }
    class MainPageViewModel : ViewModelBase
    {
        Frame Frame;
        public MainPageViewModel(Frame frame)
        {
            this.Frame = frame;
        }

        public async Task NavigateHomePage(Dictionary<String, Object> pageState)
        {
            Window.Current.CoreWindow.KeyUp += CoreWindow_KeyUp;
            DataTransferManager.GetForCurrentView().DataRequested += DocumentPage_DataRequested;
            PrintManager.GetForCurrentView().PrintTaskRequested += DocumentPage_PrintTaskRequested;

            try
            {
                StorageFile file = await ApplicationData.Current.LocalFolder.GetFileAsync("DocumentCollection.xml");
                using (Stream fs = await file.OpenStreamForReadAsync())
                {
                    DataContractSerializer serializer = new DataContractSerializer(typeof(ObservableCollection<DocumentPageInfo>));
                    this.DocumentCollection = (ObservableCollection<DocumentPageInfo>)serializer.ReadObject(fs);
                }
                await file.DeleteAsync();
                if (pageState != null && pageState.Count > 0)
                {
                    int selIndex = (int)pageState["ActiveIndex"];
                    if (selIndex != -1)
                    {
                        this.ActiveDocument(selIndex);
                        return;
                    }
                }
            }
            catch (FileNotFoundException)
            {
                this.DocumentCollection = new ObservableCollection<DocumentPageInfo>();
            }
            this.Frame.Navigate(typeof(HomePage), this);
        }

        public async Task LeaveMainPage(Dictionary<String, Object> pageState)
        {
            DataTransferManager.GetForCurrentView().DataRequested -= DocumentPage_DataRequested;

            PrintManager.GetForCurrentView().PrintTaskRequested -= DocumentPage_PrintTaskRequested;

            pageState["ActiveIndex"] = this.CurrentDocumentIndex;
            DataContractSerializer serializer = new DataContractSerializer(typeof(ObservableCollection<DocumentPageInfo>));
            StorageFile file = await ApplicationData.Current.LocalFolder.CreateFileAsync("DocumentCollection.xml", CreationCollisionOption.ReplaceExisting);
            using (Stream fs = await file.OpenStreamForWriteAsync())
            {
                serializer.WriteObject(fs, this.DocumentCollection);
            }
        }

        void CoreWindow_KeyUp(CoreWindow sender, KeyEventArgs e)
        {
            bool isCtrlPressed = WindowUtils.IsModiferKeyPressed(VirtualKey.Control);
            bool isShiftPressed = WindowUtils.IsModiferKeyPressed(VirtualKey.Shift);
            if (isCtrlPressed)
            {
                switch (e.VirtualKey)
                {
                    case VirtualKey.N:
                        this.AddCommand.Execute(null);
                        break;
                    case VirtualKey.Tab:
                        int selIndex = isShiftPressed ? this.CurrentDocumentIndex - 1 : this.CurrentDocumentIndex + 1;
                        if (selIndex > this.DocumentCollection.Count - 1)
                            selIndex = 0;
                        else if (selIndex < 0)
                            selIndex = this.DocumentCollection.Count - 1;
                        this.ActiveDocument(selIndex);
                        break;
                    case VirtualKey.F:
                        this.ShowSearchFlyout();
                        break;
                    case VirtualKey.G:
                        this.ShowGoToFlyout();
                        break;
                    case VirtualKey.S:
                        this.ShowPropertyFlyout();
                        break;
                }
            }
        }

        int _CurrentDocumentIndex;
        public int CurrentDocumentIndex
        {
            get
            {
                return this._CurrentDocumentIndex;
            }
            set
            {
                this._CurrentDocumentIndex = value;
                this.OnPropertyChanged();
            }
        }

        bool _AppBarOpen;
        public bool AppBarOpen
        {
            get
            {
                return this._AppBarOpen;
            }
            set
            {
                this._AppBarOpen = value;
                this.OnPropertyChanged();
            }
        }

        ObservableCollection<DocumentPageInfo> _DocumentCollection;
        public ObservableCollection<DocumentPageInfo> DocumentCollection
        {
            get
            {
                return this._DocumentCollection;
            }
            private set
            {
                this._DocumentCollection = value;
                this.OnPropertyChanged();
            }
        }

        DocumentControl CurrentDocument;

        public bool IsAppBarClosed()
        {
            return this.CurrentDocument == null || this.CurrentDocument.IsHitTextArea(Window.Current.CoreWindow.PointerPosition);
        }

        void DocumentPage_PrintTaskRequested(PrintManager sender, PrintTaskRequestedEventArgs args)
        {
            this.CurrentDocument.Print(args);
        }

        void DocumentPage_DataRequested(DataTransferManager sender, DataRequestedEventArgs args)
        {
            this.CurrentDocument.GetData(args);
        }

        public DelegateCommand<object> AddCommand
        {
            get
            {
                return new DelegateCommand<object>((param) =>
                {
                    this.Frame.Navigate(typeof(HomePage), this);
                    this.AppBarOpen = false;
                    this.CurrentDocument = null;
                });
            }
        }

        public DelegateCommand<DocumentPageInfo> RemoveCommand
        {
            get
            {
                return new DelegateCommand<DocumentPageInfo>((param) =>
                {
                    param.doc.Dispose();
                    int index = this.DocumentCollection.IndexOf(param);
                    this.DocumentCollection.RemoveAt(index);
                    if (this.DocumentCollection.Count == 0)
                    {
                        this.Frame.Navigate(typeof(HomePage), this);
                        this.CurrentDocument = null;
                    }
                    else if (index != -1)
                    {
                        this.ActiveDocument(index == 0 ? 0 : index - 1);
                    }
                    this.AppBarOpen = false;
                });
            }
        }

        public async Task CreateDocument(FileType doctype)
        {
            DocumentControl doc = new DocumentControl();

            var loader = new Windows.ApplicationModel.Resources.ResourceLoader();
            doc.Title = string.Format(loader.GetString("NewDocumentTitle"), this.DocumentCollection.Count);
            await doc.SetDocumentType(doctype);

            DocumentPageInfo info = new DocumentPageInfo(doc);
            this.DocumentCollection.Add(info);
            this.ActiveDocument(info);
        }

        public async Task CreateDocument(StorageFile file)
        {
            if (file != null)
            {
                DocumentControl doc = new DocumentControl();

                await doc.LoadFile(file);
                doc.Title = file.Name;

                DocumentPageInfo info = new DocumentPageInfo(doc);
                this.DocumentCollection.Add(info);
                this.ActiveDocument(info);
            }
        }

        public async Task Save()
        {
            FileSavePicker savePicker = new FileSavePicker();
            savePicker.SuggestedStartLocation = PickerLocationId.DocumentsLibrary;
            ObservableCollection<FileType> collection = AppSettings.Current.FileTypeCollection;
            foreach (FileType type in collection)
                savePicker.FileTypeChoices.Add(type.DocumentTypeName, type.ExtensionCollection);
            savePicker.SuggestedStartLocation = PickerLocationId.DocumentsLibrary;
            StorageFile file = await savePicker.PickSaveFileAsync();
            if (file != null)
            {
                await this.CurrentDocument.SaveFile(file);
            }
        }

        public void ShowSearchFlyout()
        {
            var popup = this.CurrentDocument.CreatePopup(typeof(FindFlyout));
            popup.IsOpen = true;
            CloseAllAppBar();
        }

        public void ShowGoToFlyout()
        {
            var popup = this.CurrentDocument.CreatePopup(typeof(GoToFlyout));
            popup.IsOpen = true;
            CloseAllAppBar();
        }

        public void ShowPropertyFlyout()
        {
            var flyout = new PropertyFlyout(this.CurrentDocument);
            flyout.PropertyChanged += async (s, args) =>
            {
                PropertyFlyoutViewModel vm = (PropertyFlyoutViewModel)s;
                if (vm.DocumentType != this.CurrentDocument.DocumentType)
                    await this.CurrentDocument.SetDocumentType(vm.DocumentType);

                if (this.CurrentDocument.LineFeed != vm.LineFeed)
                    this.CurrentDocument.LineFeed = vm.LineFeed;

                if (this.CurrentDocument.Encode != vm.Encode)
                {
                    ResourceLoader loader = new ResourceLoader();
                    MessageDialog dialog = new MessageDialog(loader.GetString("ReloadMessage"));
                    dialog.Commands.Add(new UICommand(loader.GetString("YesButton"), async (p) =>
                    {
                        await this.CurrentDocument.ReloadFile(vm.Encode);
                    }));
                    dialog.Commands.Add(new UICommand(loader.GetString("NoButton"), (p) =>
                    {
                        this.CurrentDocument.Encode = vm.Encode;
                    }));
                    await dialog.ShowAsync();
                }
            };
            var popup = FlyoutUtils.CreateFlyoutUnderTopAppBar(flyout);
            popup.IsOpen = true;
            CloseAllAppBar();
        }

        private void CloseAllAppBar()
        {
            this.AppBarOpen = false;
        }

        private void UndoButton_Click(object sender, RoutedEventArgs e)
        {
            this.CurrentDocument.Undo();
        }

        private void RedoButton_Click(object sender, RoutedEventArgs e)
        {
            this.CurrentDocument.Redo();
        }

        private void ActiveDocument(int index)
        {
            DocumentPageInfo info = this.DocumentCollection[index];
            info.doc.ApplySetting(AppSettings.Current);
            if (this.CurrentDocumentIndex != index)
                this.CurrentDocumentIndex = index;
            this.Frame.Content = info.doc;
            this.CurrentDocument = info.doc;
        }

        public void ActiveDocument(DocumentPageInfo info)
        {
            this.ActiveDocument(this.DocumentCollection.IndexOf(info));
        }

        internal void Redo()
        {
            this.CurrentDocument.Redo();
        }

        internal void Undo()
        {
            this.CurrentDocument.Undo();
        }
    }
}
