﻿using System;
using System.Threading.Tasks;
using System.Printing;
using System.Windows;
using System.Windows.Xps;
using Shapes = System.Windows.Shapes;
using System.Collections.Generic;
using System.Windows.Controls;
using System.Windows.Documents;
using System.Windows.Documents.Serialization;
using System.Windows.Media;

namespace FooEditEngine.WPF
{
    /// <summary>
    /// イベントデータ
    /// </summary>
    public sealed class ParseCommandEventArgs
    {
        /// <summary>
        /// 印刷中のページ番号
        /// </summary>
        public int PageNumber;
        /// <summary>
        /// ページ範囲内で許容されている最大の番号
        /// </summary>
        public int MaxPageNumber;
        /// <summary>
        /// 処理前の文字列
        /// </summary>
        public string Original;
        /// <summary>
        /// コンストラクター
        /// </summary>
        /// <param name="nowPage">印刷中のページ番号</param>
        /// <param name="maxPage">印刷すべき最大のページ番号</param>
        /// <param name="org">処理前の文字列</param>
        public ParseCommandEventArgs(int nowPage,int maxPage,string org)
        {
            this.PageNumber = nowPage;
            this.MaxPageNumber = maxPage;
            this.Original = org;
        }
    }

    /// <summary>
    /// コマンド処理用デリゲート
    /// </summary>
    /// <param name="sender">送信元のクラス</param>
    /// <param name="e">イベントデータ</param>
    /// <returns>処理後の文字列</returns>
    public delegate string ParseCommandHandler(object sender,ParseCommandEventArgs e);

    /// <summary>
    /// 印刷用のクラス
    /// </summary>
    public class FooPrintText
    {
        /// <summary>
        /// コンストラクター
        /// </summary>
        public FooPrintText()
        {
            this.ParseHF = new ParseCommandHandler((s, e) => { return e.Original; });
        }

        /// <summary>
        /// 印刷する最小のページ番号
        /// </summary>
        public int StartPage
        {
            get;
            set;
        }

        /// <summary>
        /// 印刷する最大のページ番号
        /// </summary>
        public int EndPage
        {
            get;
            set;
        }

        /// <summary>
        /// 印刷する領域の大きさ
        /// </summary>
        public System.Windows.Rect PageRect
        {
            get;
            set;
        }

        /// <summary>
        /// 対象となるドキュメント
        /// </summary>
        public Document Document
        {
            get;
            set;
        }

        /// <summary>
        /// レタリング時のフロー方向を示す
        /// </summary>
        public FlowDirection FlowDirection
        {
            get;
            set;
        }

        /// <summary>
        /// 行番号を表示するかどうか
        /// </summary>
        public bool DrawLineNumber
        {
            get;
            set;
        }

        /// <summary>
        /// ハイパーリンクに下線を引くなら真
        /// </summary>
        public bool MarkURL
        {
            get;
            set;
        }

        /// <summary>
        /// デフォルトの文字ブラシ
        /// </summary>
        public System.Windows.Media.Color Foreground
        {
            get;
            set;
        }

        /// <summary>
        /// URLを表すブラシ
        /// </summary>
        public System.Windows.Media.Color URL
        {
            get;
            set;
        }

        /// <summary>
        /// キーワード１を表すブラシ
        /// </summary>
        public System.Windows.Media.Color Keyword1
        {
            get;
            set;
        }

        /// <summary>
        /// キーワード２を表すブラシ
        /// </summary>
        public System.Windows.Media.Color Keyword2
        {
            get;
            set;
        }

        /// <summary>
        /// コメントを表すブラシ
        /// </summary>
        public System.Windows.Media.Color Comment
        {
            get;
            set;
        }

        /// <summary>
        /// 文字リテラルを表すブラシ
        /// </summary>
        public System.Windows.Media.Color Litral
        {
            get;
            set;
        }

        /// <summary>
        /// 印刷に使用するフォント
        /// </summary>
        public FontFamily Font
        {
            get;
            set;
        }

        /// <summary>
        /// フォントサイズ
        /// </summary>
        public double FontSize
        {
            get;
            set;
        }

        /// <summary>
        /// 用紙の端で折り返すなら真
        /// </summary>
        public bool isWordRap
        {
            get;
            set;
        }

        /// <summary>
        /// ヘッダー
        /// </summary>
        public string Header
        {
            get;
            set;
        }

        /// <summary>
        /// フッター
        /// </summary>
        public string Footer
        {
            get;
            set;
        }

        /// <summary>
        /// シンタックスハイライター
        /// </summary>
        public IHilighter Hilighter
        {
            get;
            set;
        }

        /// <summary>
        /// ヘッダーやフッターを処理する
        /// </summary>
        public ParseCommandHandler ParseHF;

        /// <summary>
        /// 印刷する
        /// </summary>
        /// <param name="pd">プリントダイアログ</param>
        public void Print(PrintDialog pd)
        {
            if (this.Font == null || this.Document == null)
                throw new InvalidOperationException();

            WPFRender render = new WPFRender(this.Font, this.FontSize);
            render.Foreground = this.Foreground;
            render.Comment = this.Comment;
            render.Keyword1 = this.Keyword1;
            render.Keyword2 = this.Keyword2;
            render.Literal = this.Litral;
            render.Url = this.URL;
            render.RightToLeft = this.FlowDirection == System.Windows.FlowDirection.RightToLeft;
            render.Printing = true;
            PrintableView view = new PrintableView(this.Document, render);
            view.isLineBreak = this.isWordRap;
            view.DrawLineNumber = this.DrawLineNumber;
            view.Header = this.Header;
            view.Footer = this.Footer;
            view.UrlMark = this.MarkURL;
            view.PageBound = this.PageRect;
            view.Hilighter = this.Hilighter;
            view.PerfomLayouts();

            try
            {
                FixedDocument fd = new FixedDocument();
                fd.DocumentPaginator.PageSize = this.PageRect.Size;
                
                int currentPage = 0;

                bool result = false;

                while (!result)
                {
                    if (this.EndPage != -1 && currentPage > this.EndPage)
                        break;

                    if (this.StartPage == -1 || currentPage >= this.StartPage)
                    {
                        PageContent pc = new PageContent();
                        
                        FixedPage fp = new FixedPage();
                        fp.Width = this.PageRect.Width;
                        fp.Height = this.PageRect.Height;

                        pc.Child = fp;

                        view.Header = this.ParseHF(this, new ParseCommandEventArgs(currentPage, this.EndPage, this.Header));
                        view.Footer = this.ParseHF(this, new ParseCommandEventArgs(currentPage, this.EndPage, this.Footer));

                        DrawingVisual dv = new DrawingVisual();

                        using (DrawingContext dc = dv.RenderOpen())
                        {
                            render.SetDrawingContext(dc);
                            view.Draw(view.PageBound);
                        }

                        VisualHost host = new VisualHost();
                        host.AddVisual(dv);

                        fp.Children.Add(host);

                        fd.Pages.Add(pc);
                    }
                    result = view.TryPageDown();
                }

                pd.PrintDocument(fd.DocumentPaginator,"");
            }
            catch (PrintingCanceledException)
            {
            }
            finally
            {
                view.Dispose();
            }
        }
    }

    class VisualHost : FrameworkElement
    {
        private List<Visual> fVisuals;

        public VisualHost()
        {
            fVisuals = new List<Visual>();
        }

        protected override Visual GetVisualChild(int index)
        {
            return fVisuals[index];
        }

        protected override int VisualChildrenCount
        {
            get { return fVisuals.Count; }
        }

        public void AddVisual(Visual visual)
        {
            fVisuals.Add(visual);
            base.AddVisualChild(visual);
        }

        public void RemoveVisual(Visual visual)
        {
            fVisuals.Remove(visual);
            base.RemoveVisualChild(visual);
        }
    }
}
