﻿using System;
using System.Runtime.InteropServices;
using System.Windows;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Controls;
using System.Windows.Controls.Primitives;
using System.Windows.Documents;
using System.Windows.Interop;
using System.Windows.Threading;
using DotNetTextStore;
using DotNetTextStore.UnmanagedAPI.TSF;
using DotNetTextStore.UnmanagedAPI.WinDef;
using Microsoft.Win32;

namespace FooEditEngine.WPF
{
    /// <summary>
    /// オートインデントを行うためのデリゲートを表す
    /// </summary>
    /// <param name="sender">イベント発生元のオブジェクト</param>
    /// <param name="e">イベントデーター</param>
    public delegate void AutoIndentHookerHandler(object sender,EventArgs e);

    /// <summary>
    /// WPFでのFooTextBoxの実装
    /// </summary>
    public class FooTextBox : Control,IDisposable
    {        
        View View;
        Controller Controller;
        WPFRender Render;
        Image image;
        D3DImage imageSource; 
        ScrollBar verticalScrollBar, horizontalScrollBar;
        TextStore textStore;
        DispatcherTimer timer;
        bool disposed = false;
        
        static FooTextBox()
        {
            DefaultStyleKeyProperty.OverrideMetadata(typeof(FooTextBox), new FrameworkPropertyMetadata(typeof(FooTextBox)));
            KeyboardNavigation.IsTabStopProperty.OverrideMetadata(typeof(FooTextBox), new FrameworkPropertyMetadata(true));
            KeyboardNavigation.TabNavigationProperty.OverrideMetadata(typeof(FooTextBox), new FrameworkPropertyMetadata(KeyboardNavigationMode.None));
        }

        /// <summary>
        /// コンストラクター
        /// </summary>
        public FooTextBox()
        {
            this.image = new Image();
            this.image.SizeChanged += new SizeChangedEventHandler(host_SizeChanged);

            this.textStore = new TextStore();
            this.textStore.IsReadOnly += () => false;
            this.textStore.GetStringLength += () => this.Document.Length;
            this.textStore.GetString += _textStore_GetString;
            this.textStore.GetSelectionIndex += _textStore_GetSelectionIndex;
            this.textStore.SetSelectionIndex += _textStore_SetSelectionIndex;
            this.textStore.InsertAtSelection += _textStore_InsertAtSelection;
            this.textStore.GetHWnd += _textStore_GetHWnd;
            this.textStore.GetScreenExtent += _textStore_GetScreenExtent;
            this.textStore.GetStringExtent += _textStore_GetStringExtent;
            this.textStore.CompositionStarted += textStore_CompositionStarted;
            this.textStore.CompositionUpdated += textStore_CompositionUpdated;
            this.textStore.CompositionEnded += textStore_CompositionEnded;

            this.Document = new Document();
            this.Document.Update += new DocumentUpdateEventHandler(Document_Update);

            this.Render = new WPFRender(this, 200, 200);
            this.Render.ShowFullSpace = this.ShowFullSpace;
            this.Render.ShowHalfSpace = this.ShowHalfSpace;
            this.Render.ShowTab = this.ShowTab;

            this.View = new View(this.Document, this.Render);
            this.View.InsertMode = this.InsertMode;
            this.View.DrawLineNumber = this.DrawLineNumber;
            this.View.HideCaret = !this.DrawCaret;
            this.View.HideLineMarker = !this.DrawCaretLine;
            this.View.UrlMark = this.MarkURL;
            this.View.TabStops = this.TabChars;

            this.Controller = new Controller(this.Document, this.View);
            this.Controller.CaretMoved += new EventHandler(Controller_CaretMoved);

            this.imageSource = new D3DImage();
            this.imageSource.Lock();
            this.imageSource.SetBackBuffer(D3DResourceType.IDirect3DSurface9, this.Render.Surface.NativePointer);  //設定しないとロード時に例外が発生する
            this.imageSource.Unlock();

            this.image = new Image();
            this.image.Stretch = Stretch.Fill;
            this.image.Source = this.imageSource;
            this.image.HorizontalAlignment = HorizontalAlignment.Left;
            this.image.VerticalAlignment = VerticalAlignment.Top;

            this.CommandBindings.Add(new CommandBinding(ApplicationCommands.Copy, CopyCommand));
            this.CommandBindings.Add(new CommandBinding(ApplicationCommands.Cut, CutCommand));
            this.CommandBindings.Add(new CommandBinding(ApplicationCommands.Paste, PasteCommand));
            this.CommandBindings.Add(new CommandBinding(ApplicationCommands.Delete, DeleteCommand));
            this.CommandBindings.Add(new CommandBinding(ApplicationCommands.SelectAll, SelectAllCommand));
            this.CommandBindings.Add(new CommandBinding(ApplicationCommands.Undo, UndoCommand));
            this.CommandBindings.Add(new CommandBinding(ApplicationCommands.Redo, RedoCommand));
            this.CommandBindings.Add(new CommandBinding(EditingCommands.ToggleInsert, ToggleInsertCommand));
            this.CommandBindings.Add(new CommandBinding(FooTextBoxCommands.ToggleRectSelectMode, ToggleRectSelectCommand));

            this.InputBindings.Add(new InputBinding(ApplicationCommands.Copy, new KeyGesture(Key.C, ModifierKeys.Control)));
            this.InputBindings.Add(new InputBinding(ApplicationCommands.Cut, new KeyGesture(Key.X, ModifierKeys.Control)));
            this.InputBindings.Add(new InputBinding(ApplicationCommands.Paste, new KeyGesture(Key.V, ModifierKeys.Control)));
            this.InputBindings.Add(new InputBinding(ApplicationCommands.Delete, new KeyGesture(Key.Delete, ModifierKeys.None)));
            this.InputBindings.Add(new InputBinding(ApplicationCommands.SelectAll, new KeyGesture(Key.A, ModifierKeys.Control)));
            this.InputBindings.Add(new InputBinding(ApplicationCommands.Undo, new KeyGesture(Key.Z, ModifierKeys.Control)));
            this.InputBindings.Add(new InputBinding(ApplicationCommands.Redo, new KeyGesture(Key.Y, ModifierKeys.Control)));
            this.InputBindings.Add(new InputBinding(EditingCommands.ToggleInsert, new KeyGesture(Key.Insert, ModifierKeys.None)));

            this.timer = new DispatcherTimer();
            this.timer.Interval = new TimeSpan(0, 0, 0, 0, 200);
            this.timer.Tick += new EventHandler(timer_Tick);

            this.Loaded += new RoutedEventHandler(FooTextBox_Loaded);

            this.AutoIndentHooker = (s,e)=>{};

            SystemEvents.UserPreferenceChanged += new UserPreferenceChangedEventHandler(SystemEvents_UserPreferenceChanged);

            this.SystemEvents_UserPreferenceChanged(null, new UserPreferenceChangedEventArgs(UserPreferenceCategory.Keyboard));
        }

        /// <summary>
        /// ファイナライザー
        /// </summary>
        ~FooTextBox()
        {
            this.DestructResource();
        }

        /// <summary>
        /// オートインデントを行うためのイベント
        /// </summary>
        public AutoIndentHookerHandler AutoIndentHooker;

        /// <summary>
        /// テンプレートを適用します
        /// </summary>
        public override void OnApplyTemplate()
        {
            base.OnApplyTemplate();

            Grid grid = this.GetTemplateChild("PART_Grid") as Grid;
            if (grid != null)
            {
                Grid.SetRow(this.image, 0);
                Grid.SetColumn(this.image, 0);
                grid.Children.Add(this.image);
            }

            this.horizontalScrollBar = this.GetTemplateChild("PART_HorizontalScrollBar") as ScrollBar;
            if (this.horizontalScrollBar != null)
            {
                this.horizontalScrollBar.SmallChange = 10;
                this.horizontalScrollBar.LargeChange = 100;
                this.horizontalScrollBar.Maximum = this.horizontalScrollBar.LargeChange + 1;
                this.horizontalScrollBar.Scroll += new ScrollEventHandler(horizontalScrollBar_Scroll);
            }
            this.verticalScrollBar = this.GetTemplateChild("PART_VerticalScrollBar") as ScrollBar;
            if (this.verticalScrollBar != null)
            {
                this.verticalScrollBar.SmallChange = 1;
                this.verticalScrollBar.LargeChange = 10;
                this.verticalScrollBar.Maximum = this.View.LayoutLines.Count + this.verticalScrollBar.LargeChange + 1;
                this.verticalScrollBar.Scroll += new ScrollEventHandler(verticalScrollBar_Scroll);
            }
        }

        /// <summary>
        /// ドキュメントを選択する
        /// </summary>
        /// <param name="start">開始インデックス</param>
        /// <param name="length">長さ</param>
        public void Select(int start, int length)
        {
            this.Controller.Select(start, length);
            this.textStore.NotifySelectionChanged();
        }

        /// <summary>
        /// キャレットを指定した行に移動させます
        /// </summary>
        /// <param name="index">インデックス</param>
        /// <remarks>このメソッドを呼び出すと選択状態は解除されます</remarks>
        public void JumpCaret(int index)
        {
            this.Controller.JumpCaret(index);
        }
        /// <summary>
        /// キャレットを指定した行と桁に移動させます
        /// </summary>
        /// <param name="row">行番号</param>
        /// <param name="col">桁</param>
        /// <remarks>このメソッドを呼び出すと選択状態は解除されます</remarks>
        public void JumpCaret(int row, int col)
        {
            this.Controller.JumpCaret(row, col);
        }

        /// <summary>
        /// 選択中のテキストをクリップボードにコピーします
        /// </summary>
        public void Copy()
        {
            string text = this.Controller.SelectedText;
            if (text != null && text != string.Empty)
                Clipboard.SetText(text);
        }

        /// <summary>
        /// 選択中のテキストをクリップボードに切り取ります
        /// </summary>
        public void Cut()
        {
            string text = this.Controller.SelectedText;
            if (text != null && text != string.Empty)
            {
                Clipboard.SetText(text);
                this.Controller.SelectedText = "";
            }
        }

        /// <summary>
        /// 選択中のテキストを貼り付けます
        /// </summary>
        public void Paste()
        {
            if (Clipboard.ContainsText() == false)
                return;
            string text = Clipboard.GetText();
            this.Controller.SelectedText = text;
        }

        /// <summary>
        /// 選択を解除する
        /// </summary>
        public void DeSelectAll()
        {
            this.Controller.DeSelectAll();
            this.textStore.NotifySelectionChanged();
        }

        /// <summary>
        /// 対応する座標を返します
        /// </summary>
        /// <param name="tp">テキストポイント</param>
        /// <returns>座標</returns>
        /// <remarks>テキストポイントがクライアント領域の原点より外にある場合、返される値は原点に丸められます</remarks>
        public System.Windows.Point GetPostionFromTextPoint(TextPoint tp)
        {
            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");
            return this.View.GetPostionFromTextPoint(tp);
        }

        /// <summary>
        /// 対応するテキストポイントを返します
        /// </summary>
        /// <param name="p">クライアント領域の原点を左上とする座標</param>
        /// <returns>テキストポイント</returns>
        public TextPoint GetTextPointFromPostion(System.Windows.Point p)
        {
            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");
            return this.View.GetTextPointFromPostion(p);
        }

        /// <summary>
        /// 行の高さを取得します
        /// </summary>
        /// <param name="row">レイアウト行</param>
        /// <returns>行の高さ</returns>
        public double GetLineHeight(int row)
        {
            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");
            return this.Render.GetHeight(this.View.LayoutLines[row]);
        }

        /// <summary>
        /// インデックスに対応する座標を得ます
        /// </summary>
        /// <param name="index">インデックス</param>
        /// <returns>座標を返す</returns>
        public System.Windows.Point GetPostionFromIndex(int index)
        {
            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");
            TextPoint tp = this.View.GetLayoutLineFromIndex(index);
            return this.View.GetPostionFromTextPoint(tp);
        }

        /// <summary>
        /// 座標からインデックスに変換します
        /// </summary>
        /// <param name="p">座標</param>
        /// <returns>インデックスを返す</returns>
        public int GetIndexFromPostion(System.Windows.Point p)
        {
            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");
            TextPoint tp = this.View.GetTextPointFromPostion(p);
            return this.View.GetIndexFromLayoutLine(tp);
        }

        /// <summary>
        /// 再描写する
        /// </summary>
        public void Refresh()
        {
            this.Refresh(this.View.PageBound);
        }

        /// <summary>
        /// アンマネージドリソースを開放する
        /// </summary>
        public void Dispose()
        {
            this.disposed = true;
            if (this.disposed)
                return;
            this.Dispose(false);
            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// リソースを開放する
        /// </summary>
        /// <param name="disposing">真ならマネージドリソースも開放し、そうでないならアンマネージドリソースのみを開放する</param>
        protected virtual void Dispose(bool disposing)
        {
            if (disposing)
                this.DestructResource();
            this.textStore.Dispose();
        }
        
        void Refresh(Rectangle updateRect)
        {
            if (this.imageSource.IsFrontBufferAvailable == false || this.disposed)
                return;

            this.timer.Stop();

            this.imageSource.Lock();
            this.Render.BegineDraw();
            this.View.Draw(updateRect);
            this.Render.EndDraw();
            this.imageSource.SetBackBuffer(D3DResourceType.IDirect3DSurface9, this.Render.Surface.NativePointer);
            this.imageSource.AddDirtyRect(new Int32Rect(0, 0, (int)this.imageSource.Width, (int)this.imageSource.Height));
            this.imageSource.Unlock();

            this.timer.Start();
        }

        #region Commands
        void CopyCommand(object sender, RoutedEventArgs e)
        {
            this.Copy();
        }

        void CutCommand(object sender, RoutedEventArgs e)
        {
            this.Cut();
            this.Refresh();
        }

        void PasteCommand(object sender, RoutedEventArgs e)
        {
            this.Paste();
            this.Refresh();
        }

        void DeleteCommand(object sender, RoutedEventArgs e)
        {
            int oldLength = this.Document.Length;
            this.Controller.DoDeleteAction();
            this.Refresh();
        }

        void SelectAllCommand(object sender, RoutedEventArgs e)
        {
            this.Select(0, this.Document.Length);
            this.Refresh();
        }

        void UndoCommand(object sender, RoutedEventArgs e)
        {
            int oldLength = this.Document.Length;
            this.Document.UndoManager.undo();
            this.Refresh();
        }

        void RedoCommand(object sender, RoutedEventArgs e)
        {
            int oldLength = this.Document.Length;
            this.Document.UndoManager.redo();
            this.Refresh();
        }

        void ToggleInsertCommand(object sender, RoutedEventArgs e)
        {
            if (this.InsertMode)
                this.InsertMode = false;
            else
                this.InsertMode = true;
            this.Refresh();
        }

        void ToggleRectSelectCommand(object sender, RoutedEventArgs e)
        {
            if (this.RectSelectMode)
                this.RectSelectMode = false;
            else
                this.RectSelectMode = true;
        }
        #endregion
        #region TSF
        int SelStart, SelEnd;   //Documentとは別に持たせないと例外が発生する
        bool ComstionNow = false;

        internal TextStore TextStore
        {
            get { return this.textStore; }
        }

        void textStore_CompositionEnded()
        {
            this.Controller.JumpCaret(this.SelStart);
            this.ComstionNow = false;
            this.Document.UndoManager.EndUndoGroup();
            this.Refresh();
        }

        void textStore_CompositionUpdated(int start, int end)
        {
            if (this.ComstionNow == false)
                return;
            using (Unlocker locker = this.textStore.LockDocument(false))
            {
                foreach (TextDisplayAttribute attr in this.textStore.EnumAttributes(start, end))
                {
                    if (attr.attribute.bAttr == TF_DA_ATTR_INFO.TF_ATTR_TARGET_CONVERTED)
                    {
                        TextPoint startTextPoint = this.View.GetLayoutLineFromIndex(attr.startIndex);
                        double x = this.Render.GetXFromIndex(this.View.LayoutLines[startTextPoint.row], startTextPoint.col);
                        if (x < this.View.Src.X ||
                            x > this.View.Src.X + this.View.PageBound.Width - this.verticalScrollBar.Width)
                        {
                            this.View.TryScroll(x, this.View.Src.Row);
                            this.Refresh();
                            break;
                        }
                        if (startTextPoint.row < this.View.Src.Row ||
                            startTextPoint.row > this.View.Src.Row + this.View.LineCountOnScreen)
                        {
                            this.View.TryScroll(this.View.Src.X, startTextPoint.row);
                            this.Refresh();
                            break;
                        }
                    }
                }
            }
        }
        bool textStore_CompositionStarted()
        {
            this.ComstionNow = true;
            this.Document.UndoManager.BeginUndoGroup();
            return true;
        }

        string _textStore_GetString(int start, int length)
        {
            return this.Document.ToString(start, length);
        }

        IntPtr _textStore_GetHWnd()
        {
            var hwndSource = HwndSource.FromVisual(this) as HwndSource;
            if (hwndSource != null)
                return hwndSource.Handle;
            else
                return IntPtr.Zero;
        }

        void _textStore_GetStringExtent(
            int i_startIndex,
            int i_endIndex,
            out POINT o_topLeft,
            out POINT o_bottomRight
        )
        {
            var endIndex = i_endIndex < 0 ? this.Document.Length - 1 : i_endIndex;
            var startPos = new Point(0, 0);
            var endPos = new Point(0, 0);
            TextPoint endTextPoint;
            View view = this.View;

            startPos = this.TranslatePoint(this.GetPostionFromIndex(i_startIndex), this);

            endTextPoint = view.GetLayoutLineFromIndex(endIndex);
            endPos = view.GetPostionFromTextPoint(endTextPoint);
            endPos = this.TranslatePoint(endPos, this);
            endPos.Y += this.Render.GetHeight(view.LayoutLines[endTextPoint.row]);

            startPos = PointToScreen(startPos);
            endPos = PointToScreen(endPos);

            o_topLeft = new POINT((int)startPos.X, (int)startPos.Y);
            o_bottomRight = new POINT((int)endPos.X, (int)endPos.Y);
        }

        void _textStore_GetScreenExtent(out POINT o_topLeft, out POINT o_bottomRight)
        {
            var pointTopLeft = new Point(0, 0);
            var pointBottomRight = new Point(this.RenderSize.Width, this.RenderSize.Height);

            pointTopLeft = PointToScreen(pointTopLeft);
            pointBottomRight = PointToScreen(pointBottomRight);

            o_topLeft = new POINT((int)pointTopLeft.X, (int)pointTopLeft.Y);
            o_bottomRight = new POINT((int)pointBottomRight.X, (int)pointBottomRight.Y);
        }

        void _textStore_GetSelectionIndex(out int o_startIndex, out int o_endIndex)
        {
            o_startIndex = this.SelStart;
            o_endIndex = this.SelEnd;
        }

        void _textStore_SetSelectionIndex(int i_startIndex, int i_endIndex)
        {
            this.SelStart = i_startIndex;
            this.SelEnd = i_endIndex;
            if (i_startIndex == i_endIndex)
            {
                this.Controller.JumpCaret(i_startIndex);
            }
            this.Refresh();
        }

        void _textStore_InsertAtSelection(string i_value)
        {
            bool prevMode = this.Controller.RectSelection;
            this.Controller.RectSelection = false;
            if (this.SelStart != this.SelEnd)
                this.Controller.Select(this.SelStart, this.SelEnd - this.SelStart);
            this.Controller.DoInputString(i_value);
            this.Controller.RectSelection = prevMode;
            this.Refresh();
        }

        /// <summary>
        /// キーボードフォーカスが取得されたときに呼ばれます
        /// </summary>
        /// <param name="e">イベントデーター</param>
        protected override void OnGotKeyboardFocus(KeyboardFocusChangedEventArgs e)
        {
            base.OnGotKeyboardFocus(e);
            this.textStore.SetFocus();
            this.DrawCaret = true;
            this.Refresh();
        }

        /// <summary>
        /// キーボードフォーカスが失われたときに呼ばれます
        /// </summary>
        /// <param name="e">イベントデーター</param>
        protected override void OnLostKeyboardFocus(KeyboardFocusChangedEventArgs e)
        {
            base.OnLostKeyboardFocus(e);
            this.DrawCaret = false;
            this.Refresh();
        }

        /// <summary>
        /// 文字列が入力されたときに呼ばれます
        /// </summary>
        /// <param name="e">イベントデーター</param>
        protected override void OnTextInput(TextCompositionEventArgs e)
        {
            if (e.Text == "\r")
            {
                this.Controller.DoEnterAction();
                this.AutoIndentHooker(this, null);
            }
            else if (e.Text == "\b")
            {
                this.Controller.DoBackSpaceAction();
            }
            else
            {
                this.Controller.DoInputString(e.Text);
            }
            this.Refresh();
            base.OnTextInput(e);
            e.Handled = true;
        }
        #endregion
        #region Event
        /// <summary>
        /// キーが押されたときに呼ばれます。
        /// </summary>
        /// <param name="e">イベントパラメーター</param>
        protected override void OnKeyDown(KeyEventArgs e)
        {
            if (this.ComstionNow)
                return;

            int toRow;
            ModifierKeys modiferKeys = e.KeyboardDevice.Modifiers;
            switch (e.Key)
            {
                case Key.Up:
                    this.Controller.MoveCaretVertical(-1, this.IsPressedModifierKey(modiferKeys, ModifierKeys.Shift));
                    this.Refresh();
                    e.Handled = true;
                    break;
                case Key.Down:
                    this.Controller.MoveCaretVertical(+1, this.IsPressedModifierKey(modiferKeys, ModifierKeys.Shift));
                    this.Refresh();
                    e.Handled = true;
                    break;
                case Key.Left:
                    this.Controller.MoveCaretHorizontical(-1, this.IsPressedModifierKey(modiferKeys, ModifierKeys.Shift), this.IsPressedModifierKey(modiferKeys, ModifierKeys.Control));
                    this.Refresh();
                    e.Handled = true;
                    break;
                case Key.Right:
                    this.Controller.MoveCaretHorizontical(1, this.IsPressedModifierKey(modiferKeys, ModifierKeys.Shift), this.IsPressedModifierKey(modiferKeys, ModifierKeys.Control));
                    this.Refresh();
                    e.Handled = true;
                    break;
                case Key.PageUp:
                    toRow = Math.Max(0, this.View.Src.Row - this.View.LineCountOnScreen);
                    this.Controller.ScrollWithCaretMove(this.View.Src.X, toRow, this.IsPressedModifierKey(modiferKeys, ModifierKeys.Shift));
                    this.Refresh();
                    break;
                case Key.PageDown:
                    toRow = Math.Min(this.View.Src.Row + this.View.LineCountOnScreen, this.View.LayoutLines.Count - 1);
                    this.Controller.ScrollWithCaretMove(this.View.Src.X, toRow, this.IsPressedModifierKey(modiferKeys, ModifierKeys.Shift));
                    this.Refresh();
                    break;
                case Key.Home:
                    if (this.IsPressedModifierKey(modiferKeys, ModifierKeys.Control))
                        this.Controller.JumpToHead(this.IsPressedModifierKey(modiferKeys, ModifierKeys.Shift));
                    else
                        this.Controller.JumpCaret(this.View.CaretPostion.row, 0);
                    this.Refresh();
                    break;
                case Key.End:
                    if (this.IsPressedModifierKey(modiferKeys, ModifierKeys.Control))
                        this.Controller.JumpToEnd(this.IsPressedModifierKey(modiferKeys, ModifierKeys.Shift));
                    else
                        this.Controller.JumpCaret(this.View.CaretPostion.row, this.View.LayoutLines[this.View.CaretPostion.row].Length - 1);
                    this.Refresh();
                    break;
                case Key.Tab:
                    int oldLength = this.Document.Length;
                    this.Controller.DoInputChar('\t');
                    this.Refresh();
                    e.Handled = true;
                    break;
            }
            base.OnKeyDown(e);
        }

        bool IsPressedModifierKey(ModifierKeys keys, ModifierKeys pressed)
        {
            if (keys == pressed)
                return true;
            if ((keys & pressed) == pressed)
                return true;
            return false;
        }

        /// <summary>
        /// ダブルクリックされたときに呼ばれます
        /// </summary>
        /// <param name="e">イベントパラメーター</param>
        /// <remarks>
        /// イベントパラメーターはFooMouseEventArgsにキャスト可能です。
        /// e.Handledを真にした場合、単語単位の選択が行われなくなります
        /// </remarks>
        protected override void OnMouseDoubleClick(MouseButtonEventArgs e)
        {
            System.Windows.Point p = e.GetPosition(this);
            int index = this.GetIndexFromPostion(p);

            FooMouseButtonEventArgs newEventArgs = new FooMouseButtonEventArgs(e.MouseDevice,
                e.Timestamp,
                e.ChangedButton,
                e.StylusDevice,
                index);
            newEventArgs.RoutedEvent = e.RoutedEvent;
            base.OnMouseDoubleClick(newEventArgs);

            if (newEventArgs.Handled)
                return;

            if (e.LeftButton == MouseButtonState.Pressed)
            {

                this.Controller.SelectWord(index);
                this.textStore.NotifySelectionChanged();
                this.Refresh();
            }
        }

        /// <summary>
        /// マウスボタンが押されたときに呼ばれます
        /// </summary>
        /// <param name="e">イベントパラメーター</param>
        /// <remarks>
        /// イベントパラメーターはFooMouseEventArgsにキャスト可能です。
        /// e.Handledを真にした場合、キャレットの移動処理が行われなくなります
        /// </remarks>
        protected override void OnMouseDown(MouseButtonEventArgs e)
        {
            System.Windows.Point p = e.GetPosition(this);
            TextPoint tp = this.View.GetTextPointFromPostion(p);
            int index = this.View.LayoutLines.GetIndexFromTextPoint(tp);

            FooMouseButtonEventArgs newEventArgs = new FooMouseButtonEventArgs(e.MouseDevice,
                e.Timestamp,
                e.ChangedButton,
                e.StylusDevice,
                index);
            newEventArgs.RoutedEvent = e.RoutedEvent;
            base.OnMouseDown(newEventArgs);

            if (newEventArgs.Handled)
                return;

            if (e.LeftButton == MouseButtonState.Pressed)
            {
                this.Controller.JumpCaret(tp.row, tp.col);
                this.Refresh();
                this.Focus();
            }
        }

        /// <summary>
        /// マウスが移動したときに呼ばれます
        /// </summary>
        /// <param name="e">イベントパラメーター</param>
        /// <remarks>
        /// イベントパラメーターはFooMouseEventArgsにキャスト可能です。
        /// e.Handledを真にした場合、選択処理と状況に応じたカーソルの変化が行われなくなります
        /// </remarks>
        protected override void  OnMouseMove(MouseEventArgs e)
        {
            System.Windows.Point p = e.GetPosition(this);
            TextPoint tp = this.View.GetTextPointFromPostion(p);
            int index = this.View.GetIndexFromLayoutLine(tp);

            FooMouseEventArgs newEventArgs = new FooMouseEventArgs(e.MouseDevice, e.Timestamp, e.StylusDevice, index);
            newEventArgs.RoutedEvent = e.RoutedEvent;
            base.OnMouseMove(newEventArgs);

            if (newEventArgs.Handled)
                return;

            if (p.X < this.image.ActualWidth && p.Y < this.image.ActualHeight)
            {

                if (this.Controller.IsMarker(tp, HilightType.Url))
                    this.Cursor = Cursors.Hand;
                else
                    this.Cursor = Cursors.IBeam;

                if (e.LeftButton == MouseButtonState.Pressed)
                {
                    this.Controller.MoveCaretAndSelect(tp);
                    this.Refresh();
                }
            }
            else
            {
                this.Cursor = Cursors.Arrow;
            }
        }

        /// <summary>
        /// ホイールを操作したときに呼ばれます
        /// </summary>
        /// <param name="e">イベントデータ</param>
        protected override void OnMouseWheel(MouseWheelEventArgs e)
        {
            int toRow;
            if(e.Delta > 0)
            {
                toRow = Math.Max(0, this.View.Src.Row - SystemParameters.WheelScrollLines);
                this.Controller.ScrollWithCaretMove(this.View.Src.X, toRow,false);
            }else{
                toRow = Math.Min(this.View.Src.Row + SystemParameters.WheelScrollLines, this.View.LayoutLines.Count - 1);
                this.Controller.ScrollWithCaretMove(this.View.Src.X, toRow,false);
            }
            this.Refresh();
            base.OnMouseWheel(e);
        }

        void SystemEvents_UserPreferenceChanged(object sender, UserPreferenceChangedEventArgs e)
        {
            if (e.Category == UserPreferenceCategory.Keyboard)
            {
                int blinkTime = (int)NativeMethods.GetCaretBlinkTime();
                this.View.CaretBlink = blinkTime >= 0;
                this.View.CaretBlinkTime = blinkTime * 2;
            }
            if (e.Category == UserPreferenceCategory.General)
            {
                this.View.CaretWidthOnInsertMode = SystemParameters.CaretWidth;
            }
        }

        void Document_Update(object sender, DocumentUpdateEventArgs e)
        {
            if (this.ComstionNow || e.type == UpdateType.Clear)
                return;
           this.textStore.NotifyTextChanged(e.startIndex,
                e.startIndex + e.removeLength,
                e.startIndex + e.insertLength);
        }

        void timer_Tick(object sender, EventArgs e)
        {
            if (this.image.ActualWidth == 0 || this.image.ActualHeight == 0)
                return;
            if (this.image.ActualWidth != this.imageSource.Width || this.image.ActualHeight != this.imageSource.Height)
            {
                this.Resize(this.image.ActualWidth, this.image.ActualHeight);
                this.Refresh();
                return;
            } 
            
            string str = this.View.LayoutLines[this.View.CaretPostion.row];
            double width = this.Render.GetWidthFromIndex(str, this.View.CaretPostion.col);
            if (width == 0.0)
                width = this.View.CaretWidthOnInsertMode;
            double height = this.Render.GetHeight(str);
            Rectangle updateRect = new Rectangle(
                this.View.CaretLocation.X,
                this.View.CaretLocation.Y,
                width,
                height);

            this.Refresh(updateRect);
        }

        void horizontalScrollBar_Scroll(object sender, ScrollEventArgs e)
        {
            if (this.horizontalScrollBar == null)
                return;
            this.View.TryScroll(this.horizontalScrollBar.Value, (int)this.View.Src.Row);
            this.Refresh();
        }

        void verticalScrollBar_Scroll(object sender, ScrollEventArgs e)
        {
            if (this.verticalScrollBar == null)
                return;
            int newRow = (int)this.verticalScrollBar.Value;
            if (newRow >= this.View.LayoutLines.Count)
                return;
            this.Controller.ScrollWithCaretMove(
                this.View.Src.X,
                newRow,
                false);
            this.Refresh();
        }

        void Controller_CaretMoved(object sender, EventArgs e)
        {
            if (this.horizontalScrollBar == null || this.verticalScrollBar == null)
                return;
            View view = this.View;
            if (view.Src.Row > this.verticalScrollBar.Maximum)
                this.verticalScrollBar.Maximum = view.Src.Row + view.LineCountOnScreen + 1;
            if (view.Src.X > this.horizontalScrollBar.Maximum)
                this.horizontalScrollBar.Maximum = view.Src.X + view.PageBound.Width + 1;
            this.verticalScrollBar.Value = view.Src.Row;
            this.horizontalScrollBar.Value = view.Src.X;
            this.SelStart = this.SelEnd = this.Controller.SelectionStart;   //同期をとらないと挿入位置がずれる
            this.View.CaretBlink = this.View.CaretBlink;
            SetValue(CaretPostionPropertyKey, this.View.CaretPostion);
            if(this.ComstionNow == false)
                this.textStore.NotifySelectionChanged();
        }

        void FooTextBox_Loaded(object sender, RoutedEventArgs e)
        {
            this.Resize(this.image.ActualWidth, this.image.ActualHeight);
            this.Focus();
            this.Refresh();
        }

        void Resize(double width, double height)
        {
            if (width == 0 || height == 0)
                throw new ArgumentOutOfRangeException();
            this.View.PageBound = new Rectangle(0, 0, width, height);

            if (this.WordRap)
                this.View.PerfomLayouts();

            if (this.CaretPostion.row > this.LayoutLineCollection.Count - 1)
                this.JumpCaret(this.LayoutLineCollection.Count - 1, 0);

            if (this.horizontalScrollBar != null)
            {
                this.horizontalScrollBar.LargeChange = this.View.PageBound.Width;
                this.horizontalScrollBar.Maximum = this.View.LongestWidth + this.horizontalScrollBar.LargeChange + 1;
            }
            if (this.verticalScrollBar != null)
            {
                this.verticalScrollBar.LargeChange = this.View.LineCountOnScreen;
                this.verticalScrollBar.Maximum = this.View.LayoutLines.Count + this.verticalScrollBar.LargeChange + 1;
            }

            this.Render.Resize(width, height);

            this.imageSource.Lock();
            this.Render.Resize(width, height);
            this.imageSource.SetBackBuffer(D3DResourceType.IDirect3DSurface9, this.Render.Surface.NativePointer);
            this.imageSource.Unlock();
        }

        void DestructResource()
        {
            this.timer.Stop();
            SystemEvents.UserPreferenceChanged -= new UserPreferenceChangedEventHandler(SystemEvents_UserPreferenceChanged);
            this.Render.Dispose();
            this.View.Dispose();
        }

        void host_SizeChanged(object sender, SizeChangedEventArgs e)
        {
            this.Resize(e.NewSize.Width, e.NewSize.Height);
        }

        /// <summary>
        /// プロパティーが変更されたときに呼ばれます
        /// </summary>
        /// <param name="e">イベントパラメーター</param>
        protected override void OnPropertyChanged(DependencyPropertyChangedEventArgs e)
        {
            switch (e.Property.Name)
            {
                case "WordRap":
                    this.View.isLineBreak = this.WordRap;
                    this.Controller.DeSelectAll();
                    this.View.AdjustCaret();
                    break;
                case "InsertMode":
                    this.View.InsertMode = this.InsertMode;
                    break;
                case "TabChars":
                    this.View.TabStops = this.TabChars;
                    break;
                case "RectSelectMode":
                    this.Controller.RectSelection = this.RectSelectMode;
                    break;
                case "DrawCaret":
                    this.View.HideCaret = !this.DrawCaret;
                    break;
                case "DrawCaretLine":
                    this.View.HideLineMarker = !this.DrawCaretLine;
                    break;
                case "DrawLineNumber":
                    this.View.DrawLineNumber = this.DrawLineNumber;
                    this.Controller.JumpCaret(this.View.CaretPostion.row, this.View.CaretPostion.col);
                    break;
                case "FontFamily":
                    this.Render.FontFamily = this.FontFamily;
                    break;
                case "FontSize":
                    this.Render.FontSize = this.FontSize;
                    break;
                case "Foreground":
                    this.Render.Foreground = this.Foreground;
                    break;
                case "Background":
                    this.Render.Background = this.Background;
                    break;
                case "ControlChar":
                    this.Render.ControlChar = this.ControlChar;
                    break;
                case "Hilight":
                    this.Render.Hilight = this.Hilight;
                    break;
                case "Keyword1":
                    this.Render.Keyword1 = this.Keyword1;
                    break;
                case "Keyword2":
                    this.Render.Keyword2 = this.Keyword2;
                    break;
                case "Comment":
                    this.Render.Comment = this.Comment;
                    break;
                case "Literal":
                    this.Render.Literal = this.Literal;
                    break;
                case "URL":
                    this.Render.Url = this.URL;
                    break;
                case "InsertCaret":
                    this.Render.InsertCaret = this.InsertCaret;
                    break;
                case "OverwriteCaret":
                    this.Render.OverwriteCaret = this.OverwriteCaret;
                    break;
                case "LineMarker":
                    this.Render.LineMarker = this.LineMarker;
                    break;
                case "MarkURL":
                    this.View.UrlMark = this.MarkURL;
                    break;
                case "ShowFullSpace":
                    this.Render.ShowFullSpace = this.ShowFullSpace;
                    break;
                case "ShowHalfSpace":
                    this.Render.ShowHalfSpace = this.ShowHalfSpace;
                    break;
                case "ShowTab":
                    this.Render.ShowTab = this.ShowTab;
                    break;
            }
            base.OnPropertyChanged(e);
        }
        #endregion
        #region property

        /// <summary>
        /// 文字列の描写に使用されるアンチエイリアシング モードを表します
        /// </summary>
        public TextAntialiasMode TextAntialiasMode
        {
            get
            {
                return this.Render.TextAntialiasMode;
            }
            set
            {
                this.Render.TextAntialiasMode = value;
            }
        }

        /// <summary>
        /// シンタックスハイライターを表す
        /// </summary>
        public IHilighter Hilighter
        {
            get
            {
                return this.View.Hilighter;
            }
            set
            {
                this.View.Hilighter = value;
                this.Render.ClearLayoutCache();
            }
        }

        /// <summary>
        /// ドキュメントを表す
        /// </summary>
        public Document Document
        {
            get;
            private set;
        }

        /// <summary>
        /// レイアウト行を表す
        /// </summary>
        public LineToIndexTable LayoutLineCollection
        {
            get { return this.View.LayoutLines; }
        }

        /// <summary>
        /// 選択中の文字列を表す
        /// </summary>
        public string SelectedText
        {
            get
            {
                return this.Controller.SelectedText;
            }
            set
            {
                int oldLength = this.Document.Length;
                this.Controller.SelectedText = value;
            }
        }

        /// <summary>
        /// 選択範囲の開始インデックス
        /// </summary>
        /// <remarks>SelectionLengthが0の場合はキャレット位置を表します</remarks>
        public int SelectionStart
        {
            get { return this.Controller.SelectionStart; }
        }

        /// <summary>
        /// 選択範囲の長さ
        /// </summary>
        /// <remarks>矩形選択モードの場合、選択範囲の文字数ではなく、開始位置から終了位置までの長さとなります</remarks>
        public int SelectionLength
        {
            get { return this.Controller.SelectionLength; }
        }

        /// <summary>
        /// キャレット位置を表す。これは依存プロパティです
        /// </summary>
        public TextPoint CaretPostion
        {
            get { return (TextPoint)GetValue(CaretPostionPropertyKey.DependencyProperty); }
        }

        static readonly DependencyPropertyKey CaretPostionPropertyKey =
            DependencyProperty.RegisterReadOnly("CaretPostion", typeof(TextPoint), typeof(FooTextBox), new FrameworkPropertyMetadata(new TextPoint()));
        
        /// <summary>
        /// デフォルトの文字色を表す。これは依存プロパティです
        /// </summary>
        public new Color Foreground
        {
            get { return (Color)GetValue(ForegroundProperty); }
            set { SetValue(ForegroundProperty, value); }
        }

        /// <summary>
        /// Foregroundの依存プロパティを表す
        /// </summary>
        public new static readonly DependencyProperty ForegroundProperty =
            DependencyProperty.Register("Foreground", typeof(Color), typeof(FooTextBox), new FrameworkPropertyMetadata(SystemColors.WindowTextColor));

        /// <summary>
        /// 背景色を表す。これは依存プロパティです
        /// </summary>
        public new Color Background
        {
            get { return (Color)GetValue(BackgroundProperty); }
            set { SetValue(BackgroundProperty, value); }
        }

        /// <summary>
        /// Backgroundの依存プロパティを表す
        /// </summary>
        public new static readonly DependencyProperty BackgroundProperty =
            DependencyProperty.Register("Background", typeof(Color), typeof(FooTextBox), new FrameworkPropertyMetadata(SystemColors.WindowColor));
        
        /// <summary>
        /// コントロールコードの文字色を表す。これは依存プロパティです
        /// </summary>
        public Color ControlChar
        {
            get { return (Color)GetValue(ControlCharProperty); }
            set { SetValue(ControlCharProperty, value); }
        }

        /// <summary>
        /// ControlCharの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty ControlCharProperty =
            DependencyProperty.Register("ControlChar", typeof(Color), typeof(FooTextBox),  new FrameworkPropertyMetadata(Colors.Gray));
        
        /// <summary>
        /// 選択時の背景色を表す。これは依存プロパティです
        /// </summary>
        public Color Hilight
        {
            get { return (Color)GetValue(HilightProperty); }
            set { SetValue(HilightProperty, value); }
        }

        /// <summary>
        /// Hilightの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty HilightProperty =
            DependencyProperty.Register("Hilight", typeof(Color), typeof(FooTextBox), new FrameworkPropertyMetadata(SystemColors.HighlightColor));
        
        /// <summary>
        /// キーワード１の文字色を表す。これは依存プロパティです
        /// </summary>
        public Color Keyword1
        {
            get { return (Color)GetValue(Keyword1Property); }
            set { SetValue(Keyword1Property, value); }
        }

        /// <summary>
        /// Keyword1の依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty Keyword1Property =
            DependencyProperty.Register("Keyword1", typeof(Color), typeof(FooTextBox), new FrameworkPropertyMetadata(Colors.Blue));

        /// <summary>
        /// キーワード2の文字色を表す。これは依存プロパティです
        /// </summary>
        public Color Keyword2
        {
            get { return (Color)GetValue(Keyword2Property); }
            set { SetValue(Keyword2Property, value); }
        }

        /// <summary>
        /// Keyword2の依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty Keyword2Property =
            DependencyProperty.Register("Keyword2", typeof(Color), typeof(FooTextBox), new FrameworkPropertyMetadata(Colors.DarkCyan));

        /// <summary>
        /// コメントの文字色を表す。これは依存プロパティです
        /// </summary>
        public Color Comment
        {
            get { return (Color)GetValue(CommentProperty); }
            set { SetValue(CommentProperty, value); }
        }

        /// <summary>
        /// Commentの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty CommentProperty =
            DependencyProperty.Register("Comment", typeof(Color), typeof(FooTextBox), new FrameworkPropertyMetadata(Colors.Green));

        /// <summary>
        /// 文字リテラルの文字色を表す。これは依存プロパティです
        /// </summary>
        public Color Literal
        {
            get { return (Color)GetValue(LiteralProperty); }
            set { SetValue(LiteralProperty, value); }
        }

        /// <summary>
        /// Literalの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty LiteralProperty =
            DependencyProperty.Register("Literal", typeof(Color), typeof(FooTextBox), new FrameworkPropertyMetadata(Colors.Brown));

        /// <summary>
        /// URLの文字色を表す。これは依存プロパティです
        /// </summary>
        public Color URL
        {
            get { return (Color)GetValue(URLProperty); }
            set { SetValue(URLProperty, value); }
        }

        /// <summary>
        /// URLの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty URLProperty =
            DependencyProperty.Register("URL", typeof(Color), typeof(FooTextBox), new FrameworkPropertyMetadata(Colors.Blue));


        /// <summary>
        /// ラインマーカーの色を表す
        /// </summary>
        public Color LineMarker
        {
            get { return (Color)GetValue(LineMarkerProperty); }
            set { SetValue(LineMarkerProperty, value); }
        }

        /// <summary>
        /// LineMarkerの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty LineMarkerProperty =
            DependencyProperty.Register("LineMarker", typeof(Color), typeof(FooTextBox), new FrameworkPropertyMetadata(SystemColors.WindowColor));

        /// <summary>
        /// 挿入モード時のキャレットの色を表す
        /// </summary>
        public Color InsertCaret
        {
            get { return (Color)GetValue(InsertCaretProperty); }
            set { SetValue(InsertCaretProperty, value); }
        }

        /// <summary>
        /// InsertCaretの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty InsertCaretProperty =
            DependencyProperty.Register("InsertCaret", typeof(Color), typeof(FooTextBox), new FrameworkPropertyMetadata(SystemColors.WindowTextColor));

        /// <summary>
        /// 上書きモード時のキャレット職を表す
        /// </summary>
        public Color OverwriteCaret
        {
            get { return (Color)GetValue(OverwriteCaretProperty); }
            set { SetValue(OverwriteCaretProperty, value); }
        }

        /// <summary>
        /// OverwriteCaretの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty OverwriteCaretProperty =
            DependencyProperty.Register("OverwriteCaret", typeof(Color), typeof(FooTextBox), new FrameworkPropertyMetadata(SystemColors.WindowTextColor));
        
        /// <summary>
        /// 挿入モードなら真を返し、そうでないなら、偽を返す。これは依存プロパティです
        /// </summary>
        public bool InsertMode
        {
            get { return (bool)GetValue(InsertModeProperty); }
            set { SetValue(InsertModeProperty, value); }
        }

        /// <summary>
        /// InsertModeの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty InsertModeProperty =
            DependencyProperty.Register("InsertMode",
            typeof(bool),
            typeof(FooTextBox),
            new FrameworkPropertyMetadata(true));

        /// <summary>
        /// タブの文字数を表す。これは依存プロパティです
        /// </summary>
        public int TabChars
        {
            get { return (int)GetValue(TabCharsProperty); }
            set { SetValue(TabCharsProperty, value); }
        }

        /// <summary>
        /// TabCharsの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty TabCharsProperty =
            DependencyProperty.Register("TabChars",
            typeof(int),
            typeof(FooTextBox),
            new FrameworkPropertyMetadata(4));

        /// <summary>
        /// 矩形選択モードなら真を返し、そうでないなら偽を返す。これは依存プロパティです
        /// </summary>
        public bool RectSelectMode
        {
            get { return (bool)GetValue(RectSelectModeProperty); }
            set { SetValue(RectSelectModeProperty, value); }
        }

        /// <summary>
        /// RectSelectModeの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty RectSelectModeProperty =
            DependencyProperty.Register("RectSelectMode", typeof(bool), typeof(FooTextBox), new FrameworkPropertyMetadata(false));

        /// <summary>
        /// ワードラップを行うなら真を返し、そうでないなら偽を返す。これは依存プロパティです
        /// </summary>
        public bool WordRap
        {
            get { return (bool)GetValue(WordRapProperty); }
            set { SetValue(WordRapProperty, value); }
        }

        /// <summary>
        /// WordRapの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty WordRapProperty =
            DependencyProperty.Register("WordRap", typeof(bool), typeof(FooTextBox), new FrameworkPropertyMetadata(false));

        /// <summary>
        /// キャレットを描くなら真。そうでないなら偽を返す。これは依存プロパティです
        /// </summary>
        public bool DrawCaret
        {
            get { return (bool)GetValue(DrawCaretProperty); }
            set { SetValue(DrawCaretProperty, value); }
        }

        /// <summary>
        /// DrawCaretの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty DrawCaretProperty =
            DependencyProperty.Register("DrawCaret", typeof(bool), typeof(FooTextBox), new FrameworkPropertyMetadata(true));

        
        /// <summary>
        /// キャレットラインを描くなら真。そうでないなら偽を返す。これは依存プロパティです
        /// </summary>
        public bool DrawCaretLine
        {
            get { return (bool)GetValue(DrawCaretLineProperty); }
            set { SetValue(DrawCaretLineProperty, value); }
        }

        /// <summary>
        /// DrawCaretLineの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty DrawCaretLineProperty =
            DependencyProperty.Register("DrawCaretLine", typeof(bool), typeof(FooTextBox), new FrameworkPropertyMetadata(true));

        /// <summary>
        /// 行番号を描くなら真。そうでなければ偽。これは依存プロパティです
        /// </summary>
        public bool DrawLineNumber
        {
            get { return (bool)GetValue(DrawLineNumberProperty); }
            set { SetValue(DrawLineNumberProperty, value); }
        }

        /// <summary>
        /// DrawLineNumberの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty DrawLineNumberProperty =
            DependencyProperty.Register("DrawLineNumber", typeof(bool), typeof(FooTextBox), new FrameworkPropertyMetadata(false));

        /// <summary>
        /// URLに下線を引くなら真。そうでないなら偽を表す。これは依存プロパティです
        /// </summary>
        public bool MarkURL
        {
            get { return (bool)GetValue(MarkURLProperty); }
            set { SetValue(MarkURLProperty, value); }
        }

        /// <summary>
        /// MarkURLの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty MarkURLProperty =
            DependencyProperty.Register("MarkURL", typeof(bool), typeof(FooTextBox), new FrameworkPropertyMetadata(true));

        /// <summary>
        /// 全角スペースを表示するなら真。そうでないなら偽
        /// </summary>
        public bool ShowFullSpace
        {
            get { return (bool)GetValue(ShowFullSpaceProperty); }
            set { SetValue(ShowFullSpaceProperty, value); }
        }

        /// <summary>
        /// ShowFullSpaceの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty ShowFullSpaceProperty =
            DependencyProperty.Register("ShowFullSpace", typeof(bool), typeof(FooTextBox), new UIPropertyMetadata(false));

        /// <summary>
        /// 半角スペースを表示するなら真。そうでないなら偽
        /// </summary>
        public bool ShowHalfSpace
        {
            get { return (bool)GetValue(ShowHalfSpaceProperty); }
            set { SetValue(ShowHalfSpaceProperty, value); }
        }

        /// <summary>
        /// ShowHalfSpaceの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty ShowHalfSpaceProperty =
            DependencyProperty.Register("ShowHalfSpace", typeof(bool), typeof(FooTextBox), new UIPropertyMetadata(false));

        /// <summary>
        /// タブを表示するなら真。そうでないなら偽
        /// </summary>
        public bool ShowTab
        {
            get { return (bool)GetValue(ShowTabProperty); }
            set { SetValue(ShowTabProperty, value); }
        }

        /// <summary>
        /// ShowTabの依存プロパティを表す
        /// </summary>
        public static readonly DependencyProperty ShowTabProperty =
            DependencyProperty.Register("ShowTab", typeof(bool), typeof(FooTextBox), new UIPropertyMetadata(false));
        
        #endregion
    }
    /// <summary>
    /// マウスボタン関連のイベントクラス
    /// </summary>
    public class FooMouseButtonEventArgs : MouseButtonEventArgs
    {
        /// <summary>
        /// イベントが発生したドキュメントのインデックス
        /// </summary>
        public int Index
        {
            get;
            private set;
        }

        /// <summary>
        /// コンストラクター
        /// </summary>
        /// <param name="mouse">マウスデバイス</param>
        /// <param name="timestamp">タイムスタンプ</param>
        /// <param name="button">ボタン</param>
        /// <param name="stylusDevice">スタイラスデバイス</param>
        /// <param name="index">インデックス</param>
        public FooMouseButtonEventArgs(MouseDevice mouse, int timestamp, MouseButton button, StylusDevice stylusDevice, int index)
            : base(mouse, timestamp, button, stylusDevice)
        {
            this.Index = index;
        }
    }
    /// <summary>
    /// マウス関連のイベントクラス
    /// </summary>
    public class FooMouseEventArgs : MouseEventArgs
    {
        /// <summary>
        /// イベントが発生したドキュメントのインデックス
        /// </summary>
        public int Index
        {
            get;
            private set;
        }

        /// <summary>
        /// コンストラクター
        /// </summary>
        /// <param name="mouse">マウスデバイス</param>
        /// <param name="timestamp">タイムスタンプ</param>
        /// <param name="stylusDevice">スタイラスデバイス</param>
        /// <param name="index">インデックス</param>
        public FooMouseEventArgs(MouseDevice mouse,
            int timestamp,
            StylusDevice stylusDevice,
            int index)
            : base(mouse, timestamp, stylusDevice)
        {
            this.Index = index;
        }
    }
}
