﻿using System;
using System.Text.RegularExpressions;
using System.Threading;
using System.Linq;
using System.Collections.Generic;
using System.Diagnostics;

namespace FooEditEngine
{
    internal class SpilitStringEventArgs : EventArgs
    {
        public Document buffer;
        public int index;
        public int length;
        public int row;
        public SpilitStringEventArgs(Document buf, int index, int length,int row)
        {
            this.buffer = buf;
            this.index = index;
            this.length = length;
            this.row = row;
        }
    }

    internal struct SyntaxInfo
    {
        public TokenType type;
        public int index;
        public int length;
        public SyntaxInfo(int index, int length, TokenType type)
        {
            this.type = type;
            this.index = index;
            this.length = length;
        }
    }

    internal enum EncloserType
    {
        None,
        Begin,
        Now,
        End,
    }

    internal class LineToIndexTableData
    {
        public int Index;
        public int Length;
        public bool LineEnd;
        public SyntaxInfo[] Syntax;
        public EncloserType EncloserType;
        public LineToIndexTableData()
        {
        }
        public LineToIndexTableData(int index, int length,bool lineend,SyntaxInfo[] syntax)
        {
            this.Index = index;
            this.Length = length;
            this.LineEnd = lineend;
            this.Syntax = syntax;
            this.EncloserType = EncloserType.None;
        }
    }

    internal delegate IList<LineToIndexTableData> SpilitStringEventHandler(object sender, SpilitStringEventArgs e);

    /// <summary>
    /// 行番号とインデックスを相互変換するためのクラス
    /// </summary>
    public class LineToIndexTable : IEnumerable<string>
    {
        List<LineToIndexTableData> Lines = new List<LineToIndexTableData>();
        Document Document;
        bool _UrlMarker;
        Regex urlPattern;

        internal LineToIndexTable(Document buf)
        {
            this.Document = buf;
        }

        internal SpilitStringEventHandler SpilitString;

        /// <summary>
        /// 行数を返す
        /// </summary>
        public int Count
        {
            get { return this.Lines.Count; }
        }

        /// <summary>
        /// シンタックスハイライター
        /// </summary>
        internal IHilighter Hilighter { get; set; }

        /// <summary>
        /// Urlに下線を引くなら真
        /// </summary>
        /// <remarks>変更を反映させるためには再描写する必要があります</remarks>
        internal bool UrlMark
        {
            get
            {
                return this._UrlMarker;
            }
            set
            {
                this._UrlMarker = value;
                this.Document.RemoveMarker(HilightType.Url);
                this.urlPattern = new Regex("(http|https|ftp)(:\\/\\/[-_.!~*\\'()a-zA-Z0-9;\\/?:\\@&=+\\$,%#]+)");
                this.SetUrlMarker(0, this.Count);
            }
        }

        /// <summary>
        /// 行番号に対応する文字列を返します
        /// </summary>
        /// <param name="n"></param>
        /// <returns></returns>
        public string this[int n]
        {
            get
            {
                LineToIndexTableData data = this.Lines[n];
                string str = this.Document.ToString(data.Index, data.Length);

                return str;
            }
        }

        internal void UpdateAsReplace(int index, int removedLength, int insertedLength)
        {
            //削除すべき行の開始位置と終了位置を求める
            int startRow = this.GetLineNumberFromIndex(index);
            if (startRow > 0 && this.Lines[startRow - 1].LineEnd == false)
                startRow--;

            int endRow = this.GetLineNumberFromIndex(index + removedLength);
            while (endRow < this.Lines.Count && this.Lines[endRow].LineEnd == false)
                endRow++;
            if (endRow >= this.Lines.Count)
                endRow = this.Lines.Count - 1;

            //SpilitStringの対象となる範囲を求める
            int HeadIndex = this.GetIndexFromLineNumber(startRow);

            int LastIndex = this.GetIndexFromLineNumber(endRow) + this.GetLengthFromLineNumber(endRow) - 1;

            int fisrtPartLength = index - HeadIndex;

            int secondPartLength = LastIndex - (index + removedLength - 1);

            int analyzeLength = fisrtPartLength + secondPartLength + insertedLength;

            System.Diagnostics.Debug.Assert(analyzeLength <= this.Document.Length - 1 - HeadIndex + 1);

            //行を削除する
            this.Lines.RemoveRange(startRow, endRow - startRow + 1);

            //行を分割し、削除した位置に挿入する
            SpilitStringEventArgs e = new SpilitStringEventArgs(this.Document, HeadIndex, analyzeLength, startRow);
            IList<LineToIndexTableData> newLines = SpilitString(this, e);

            this.Lines.InsertRange(startRow, newLines);

            //挿入された行以降を更新する
            int deltaLength = insertedLength - removedLength;

            for (int i = startRow + newLines.Count; i < this.Lines.Count; i++)
            {
                this.Lines[i].Index += deltaLength;
            }

            //最終行が削除された場合は追加する
            if (this.Lines.Count == 0)
            {
                this.Lines.Add(new LineToIndexTableData());
            }
            else
            {
                LineToIndexTableData lastLine = this.Lines.Last();
                if (lastLine.Length != 0 && this.Document[lastLine.Index + lastLine.Length - 1] == Document.NewLine)
                    this.Lines.Add(new LineToIndexTableData(lastLine.Index + lastLine.Length, 0, true, null));
            }

            this.Hilight(startRow, newLines.Count);
            SetUrlMarker(startRow, newLines.Count);

        }

        /// <summary>
        /// 行番号をインデックスに変換します
        /// </summary>
        /// <param name="row">行番号</param>
        /// <returns>0から始まるインデックスを返す</returns>
        public int GetIndexFromLineNumber(int row)
        {
            if (row < 0 || row > this.Lines.Count)
                throw new ArgumentOutOfRangeException();
            return this.Lines[row].Index;
        }

        /// <summary>
        /// 行の長さを得ます
        /// </summary>
        /// <param name="row">行番号</param>
        /// <returns>行の文字長を返します</returns>
        public int GetLengthFromLineNumber(int row)
        {
            if (row < 0 || row > this.Lines.Count)
                throw new ArgumentOutOfRangeException();
            return this.Lines[row].Length;
        }

        internal LineToIndexTableData GetData(int row)
        {
            if (row < 0 || row > this.Lines.Count)
                throw new ArgumentOutOfRangeException();
            return this.Lines[row];
        }

        int lastLineNumber;
        /// <summary>
        /// インデックスを行番号に変換します
        /// </summary>
        /// <param name="index">インデックス</param>
        /// <returns>行番号を返します</returns>
        public int GetLineNumberFromIndex(int index)
        {
            if (index < 0)
                throw new ArgumentOutOfRangeException("indexに負の値を設定することはできません");
            if (index == 0 && this.Lines.Count > 0)
                return 0;
            if (lastLineNumber > this.Lines.Count - 1 || index < this.Lines[lastLineNumber].Index)
                lastLineNumber = 0;
            int i;
            for (i = lastLineNumber; i < this.Lines.Count; i++)
            {
                if (index < this.Lines[i].Index + this.Lines[i].Length)
                {
                    lastLineNumber = i;
                    return i;
                }
            }
            if (index <= this.Lines[i - 1].Index + this.Lines[i - 1].Length)
            {
                lastLineNumber = i - 1;
                return i - 1;
            }
            throw new ArgumentOutOfRangeException("indexに対応する行が見つかりませんでした");
        }

        /// <summary>
        /// インデックスからテキストポイントに変換します
        /// </summary>
        /// <param name="index">インデックス</param>
        /// <returns>TextPoint構造体を返します</returns>
        public TextPoint GetTextPointFromIndex(int index)
        {
            TextPoint tp = new TextPoint();
            tp.row = GetLineNumberFromIndex(index);
            tp.col = index - this.Lines[tp.row].Index;
            Debug.Assert(tp.row < this.Lines.Count && tp.col <= this.Lines[tp.row].Length);
            return tp;
        }

        /// <summary>
        /// テキストポイントからインデックスに変換します
        /// </summary>
        /// <param name="tp">TextPoint構造体</param>
        /// <returns>インデックスを返します</returns>
        public int GetIndexFromTextPoint(TextPoint tp)
        {
            if(tp.row < 0 || tp.row > this.Lines.Count)
                throw new ArgumentOutOfRangeException("tp.rowが設定できる範囲を超えています");
            if (tp.col < 0 || tp.col > this.Lines[tp.row].Length)
                throw new ArgumentOutOfRangeException("tp.colが設定できる範囲を超えています");
            return this.Lines[tp.row].Index + tp.col;
        }

        /// <summary>
        /// すべての行に対しシンタックスハイライトを行います
        /// </summary>
        public void HilightAll()
        {
            this.Hilight(0, this.Lines.Count);
            SetUrlMarker(0, this.Lines.Count);
        }

        /// <summary>
        /// ハイライト関連の情報をすべて削除します
        /// </summary>
        public void ClearHilight()
        {
            foreach (LineToIndexTableData line in this.Lines)
                line.Syntax = null;
        }

        /// <summary>
        /// すべて削除します
        /// </summary>
        internal void Clear()
        {
            this.Lines.Clear();
            this.Lines.Add(new LineToIndexTableData());
        }

        void Hilight(int row, int rowCount)
        {
            if (this.Hilighter == null || rowCount == 0)
                return;

            //シンタックスハイライトの開始行を求める
            int startRow = row;
            EncloserType type = this.Lines[startRow].EncloserType;
            EncloserType prevLineType = startRow > 0 ? this.Lines[startRow - 1].EncloserType : EncloserType.None;
            if (type == EncloserType.Now || type == EncloserType.End ||
                prevLineType == EncloserType.Now || prevLineType == EncloserType.End)
                startRow = SearchStartRow(startRow);
            else if (prevLineType == EncloserType.Begin)
                startRow = startRow - 1;

            //シンタックスハイライトの終了行を求める
            int endRow = row + rowCount - 1;
            type = this.Lines[endRow].EncloserType;
            prevLineType = endRow > 0 ? this.Lines[endRow - 1].EncloserType : EncloserType.None;
            if (type == EncloserType.Begin || type == EncloserType.Now ||
                prevLineType == EncloserType.Begin || prevLineType == EncloserType.Now)
                endRow = SearchEndRow(endRow);
            else if (endRow + 1 <= this.Lines.Count - 1 && this.Lines[endRow + 1].EncloserType == EncloserType.Now)
                endRow = SearchEndRow(endRow + 1);

            //シンタックスハイライトを行う
            bool hasBeginEncloser = false;
            int i;
            for (i = startRow; i <= endRow; i++)
            {
                this.HilightLine(i, ref hasBeginEncloser);
            }

            if (hasBeginEncloser)   //終了エンクロージャーが見つかったかどうか
            {
                for (; i < this.Lines.Count; i++)
                {
                    if (this.HilightLine(i, ref hasBeginEncloser) < 0)
                        break;
                }
            }

            this.Hilighter.Reset();
        }

        private int HilightLine(int row, ref bool hasBeginEncloser)
        {
            //シンタックスハイライトを行う
            List<SyntaxInfo> syntax = new List<SyntaxInfo>();
            string str = this[row];
            int level = this.Hilighter.DoHilight(str, str.Length, (s) =>
            {
                if (s.type == TokenType.None || s.type == TokenType.Control)
                    return;
                if (str[s.index + s.length - 1] == Document.NewLine)
                    s.length--;
                syntax.Add(new SyntaxInfo(s.index, s.length, s.type));
            });

            LineToIndexTableData lineData = this.GetData(row);
            lineData.Syntax = syntax.ToArray();

            if (level > 0 && hasBeginEncloser == false)  //開始エンクロージャー
            {
                lineData.EncloserType = EncloserType.Begin;
                hasBeginEncloser = true;
            }
            else if (level < 0) //終了エンクロージャー
            {
                lineData.EncloserType = EncloserType.End;
                hasBeginEncloser = false;
            }
            else if (hasBeginEncloser) //エンクロージャーの範囲内
                lineData.EncloserType = EncloserType.Now;
            else
                lineData.EncloserType = EncloserType.None;

            return level;
        }

        private int SearchStartRow(int startRow)
        {
            for (startRow--; startRow >= 0; startRow--)
            {
                EncloserType type = this.Lines[startRow].EncloserType;
                if (type == EncloserType.Begin || type == EncloserType.None)
                    return startRow;
            }
            return 0;
        }

        private int SearchEndRow(int startRow)
        {
            for (startRow++ ; startRow < this.Lines.Count; startRow++)
            {
                EncloserType type = this.Lines[startRow].EncloserType;
                if (type == EncloserType.End)
                    return startRow;
            }
            return this.Lines.Count - 1;
        }

        void SetUrlMarker(int row, int count)
        {
            if (this.UrlMark == false)
                return;

            int startRow = row;
            int endRow = row + count - 1;

            for (int i = startRow; i <= endRow; i++)
            {
                Match m = this.urlPattern.Match(this[i]);

                if (m.Success)
                {
                    int lineHeadIndex = this.GetIndexFromLineNumber(i);
                    int start = lineHeadIndex + m.Index;
                    this.Document.RemoveMarker(start, 1);
                    this.Document.Markers.Add(Marker.Create(start, m.Length, HilightType.Url));
                }
            }
        }

        #region IEnumerable<string> メンバー

        /// <summary>
        /// コレクションを反復処理するためのIEnumeratorを返す
        /// </summary>
        /// <returns>IEnumeratorオブジェクト</returns>
        public IEnumerator<string> GetEnumerator()
        {
            for (int i = 0; i < this.Lines.Count; i++)
                yield return this[i];
        }

        #endregion

        #region IEnumerable メンバー

        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            for (int i = 0; i < this.Lines.Count; i++)
                yield return this[i];
        }

        #endregion
    }

}
