﻿using System;
using System.Collections.Generic;

namespace FooEditEngine
{
    struct Point
    {
        public double X;
        public double Y;
        public Point(double x, double y)
        {
            this.X = x;
            this.Y = y;
        }
#if WINFORM
        public static implicit operator Point(System.Drawing.Point p)
        {
            return new Point(p.X, p.Y);
        }
        public static implicit operator System.Drawing.Point(Point p)
        {
            return new System.Drawing.Point((int)p.X, (int)p.Y);
        }
        public static implicit operator DWriteWarpper.Point2F(Point p)
        {
            return new DWriteWarpper.Point2F((float)p.X, (float)p.Y);
        }
#endif
#if WPF
        public static implicit operator Point(System.Windows.Point p)
        {
            return new Point(p.X, p.Y);
        }
        public static implicit operator System.Windows.Point(Point p)
        {
            return new System.Windows.Point(p.X, p.Y);
        }
        public static implicit operator DWriteWarpper.Point2F(Point p)
        {
            return new DWriteWarpper.Point2F((float)p.X, (float)p.Y);
        }
#endif
    }
    struct Size
    {
        public double Width;
        public double Height;
        public Size(double width, double height)
        {
            this.Width = width;
            this.Height = height;
        }
#if WINFORM
        public static implicit operator Size(System.Drawing.Size p)
        {
            return new Size(p.Width, p.Height);
        }
        public static implicit operator System.Drawing.Size(Size p)
        {
            return new System.Drawing.Size((int)p.Width, (int)p.Height);
        }
#endif
#if WPF
        public static implicit operator Size(System.Windows.Size p)
        {
            return new Size(p.Width, p.Height);
        }
        public static implicit operator System.Windows.Size(Size p)
        {
            return new System.Windows.Size(p.Width, p.Height);
        }
#endif
    }
    struct Rectangle
    {
        public Point Location;
        public Size Size;
        public Point TopLeft
        {
            get { return this.Location; }
        }
        public Point TopRight
        {
            get { return new Point(this.Right, this.Location.Y); }
        }
        public Point BottomLeft
        {
            get { return new Point(this.Location.X, this.Bottom); }
        }
        public Point BottomRight
        {
            get { return new Point(this.Right, this.Bottom); }
        }
        public double Right
        {
            get { return this.X + this.Width; }
        }
        public double Bottom
        {
            get { return this.Y + this.Height; }
        }
        public double Height
        {
            get { return this.Size.Height; }
            set { this.Size.Height = value; }
        }
        public double Width
        {
            get { return this.Size.Width; }
            set { this.Size.Width = value; }
        }
        public double X
        {
            get { return this.Location.X; }
        }
        public double Y
        {
            get { return this.Location.Y; }
        }
        public Rectangle(double x, double y, double width, double height)
        {
            this.Location = new Point(x, y);
            this.Size = new Size(width, height);
        }
#if WINFORM
        public static implicit operator Rectangle(System.Drawing.Rectangle p)
        {
            return new Rectangle(p.X,p.Y,p.Width,p.Height);
        }
        public static implicit operator System.Drawing.Rectangle(Rectangle p)
        {
            return new System.Drawing.Rectangle((int)p.X, (int)p.Y, (int)p.Width, (int)p.Height);
        }
        public static implicit operator DWriteWarpper.RectF(Rectangle p)
        {
            return new DWriteWarpper.RectF((float)p.X, (float)p.Y, (float)p.Right, (float)p.Bottom);
        }
#endif
#if WPF
        public static implicit operator Rectangle(System.Windows.Rect p)
        {
            return new Rectangle(p.X,p.Y,p.Width,p.Height);
        }
        public static implicit operator System.Windows.Rect(Rectangle p)
        {
            return new System.Windows.Rect(p.X, p.Y, p.Width, p.Height);
        }
        public static implicit operator DWriteWarpper.RectF(Rectangle p)
        {
            return new DWriteWarpper.RectF((float)p.X, (float)p.Y, (float)p.Right, (float)p.Bottom);
        }
#endif
    }
    enum AlignDirection
    {
        Forward,
        Back,
    }
    delegate void ChangedRenderResourceEventHandler(object sender,EventArgs e);
    interface ITextRender
    {
        /// <summary>
        /// ドキュメントを表示する領域
        /// </summary>
        Rectangle ClipRect { get; set; }
        
        /// <summary>
        /// 行番号の幅
        /// </summary>
        double LineNemberWidth { get; }
        
        /// <summary>
        /// タブの文字数
        /// </summary>
        int TabWidthChar { get; set; }

        /// <summary>
        /// 保持しているリソースに変化があったことを通知する
        /// </summary>
        event ChangedRenderResourceEventHandler ChangedRenderResource;

        /// <summary>
        /// キャッシュされたビットマップを描写する
        /// </summary>
        /// <param name="rect">描く領域</param>
        void DrawCachedBitmap(Rectangle rect);

        /// <summary>
        /// 描写したものをキャッシュする
        /// </summary>
        void CacheContent();

        /// <summary>
        /// キャッシュが存在するなら真を返し、そうでないなら偽を返す
        /// </summary>
        bool IsVaildCache();

        /// <summary>
        /// 行番号を表示する
        /// </summary>
        /// <param name="lineNumber">表示すべき番号</param>
        /// <param name="x">行番号の左上を示すX座標</param>
        /// <param name="y">行番号の左上を示すY座標</param>
        void DrawLineNumber(int lineNumber, double x, double y);

        /// <summary>
        /// ラインマーカーを描く
        /// </summary>
        /// <param name="rect">文字列と外接する四角形</param>
        void DrawLineMarker(Rectangle rect);

        /// <summary>
        /// キャレットを描く
        /// </summary>
        /// <param name="rect"></param>
        /// <param name="transparent"></param>
        void DrawCaret(Rectangle rect,bool transparent);

        /// <summary>
        /// 背景を塗りつぶす
        /// </summary>
        /// <param name="rect">塗りつぶすべき領域</param>
        void FillBackground(Rectangle rect);

        /// <summary>
        /// 行を表示する
        /// </summary>
        /// <param name="lti">LineToIndexオブジェクト</param>
        /// <param name="row">行</param>
        /// <param name="x">行の左上を表すX座標</param>
        /// <param name="y">行の左上を表すY座標</param>
        /// <param name="SelectRanges">選択領域を保持しているコレクション。選択領域の開始位置は行の先頭を０とする相対位置としてください（位置が-1の場合表示されません）</param>
        /// <param name="MarkerRanges">マーカーを保持しているコレクション。マーカーの開始位置は行の先頭を０とする相対位置としてください（位置が-1の場合表示されません）</param>
        void DrawOneLine(LineToIndexTable lti, int row, double x, double y, IEnumerable<Selection> SelectRanges, IEnumerable<Marker> MarkerRanges);

        /// <summary>
        /// X座標に対応するインデックスを得る
        /// </summary>
        /// <param name="str">対象となる文字列</param>
        /// <param name="x">X座標</param>
        /// <returns>インデックス</returns>
        /// <remarks>行番号の幅は考慮されてないのでView以外のクラスは呼び出さないでください</remarks>
        int GetIndexFromX(string str, double x);

        /// <summary>
        /// インデックスに対応する文字の幅を得る
        /// </summary>
        /// <param name="str">対象となる文字列</param>
        /// <param name="index">インデックス</param>
        /// <returns>文字の幅</returns>
        double GetWidthFromIndex(string str, int index);

        /// <summary>
        /// 文字列の幅を得る
        /// </summary>
        /// <param name="str">文字列</param>
        /// <returns>文字列の幅</returns>
        double GetWidth(string str);

        /// <summary>
        /// インデックスに対応するX座標を得る
        /// </summary>
        /// <param name="str">対象となる文字列</param>
        /// <param name="index">インデックス</param>
        /// <returns>X座標</returns>
        /// <remarks>行頭にEOFが含まれている場合、0が返ります</remarks>
        double GetXFromIndex(string str, int index);

        /// <summary>
        /// 文字列の高さを返す
        /// </summary>
        /// <param name="str">対象となる文字列</param>
        /// <returns>文字列の高さ</returns>
        double GetHeight(string str);

        /// <summary>
        /// 適切な位置にインデックスを調整する
        /// </summary>
        /// <param name="str">対象となる文字列</param>
        /// <param name="index">インデックス</param>
        /// <param name="flow">真の場合は隣接するクラスターを指すように調整し、
        /// そうでない場合は対応するクラスターの先頭を指すように調整します</param>
        /// <returns>調整後のインデックス</returns>
        int AlignIndexToNearestCluster(string str, int index, AlignDirection flow);

        /// <summary>
        /// 行を分割する
        /// </summary>
        /// <param name="doc">ドキュメント</param>
        /// <param name="startIndex">行の開始位置</param>
        /// <param name="endIndex">行の終了位置</param>
        /// <param name="wrapwidth">幅</param>
        /// <returns>折り返し後の行全体を表すコレクション</returns>
        List<LineToIndexTableData> BreakLine(Document doc, int startIndex, int endIndex, double wrapwidth);
    }

    enum StringAlignment
    {
        Left,
        Center,
        Right,
    }
    interface IPrintableTextRender : ITextRender
    {
        /// <summary>
        /// 文字列を表示する
        /// </summary>
        /// <param name="str">文字列</param>
        /// <param name="x">x座標</param>
        /// <param name="y">y座標</param>
        /// <param name="align">書式方向</param>
        void DrawString(string str, double x, double y,StringAlignment align);

        /// <summary>
        /// ヘッダーの高さ
        /// </summary>
        float HeaderHeight { get; }

        /// <summary>
        /// フッターの高さ
        /// </summary>
        float FooterHeight { get; }
    }
}
